/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"
#define BUFFER_SIZE  14
/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK),
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_spi_inittypedef SPI1_InitStruct =    /**< SPI init structure */
{
    MD_SPI_MODE_MASTER,
    MD_SPI_PHASE_1EDGE,
    MD_SPI_POLARITY_LOW,
    MD_SPI_BAUDRATEPRESCALER_DIV256,
    MD_SPI_MSB_FIRST,
    MD_SPI_FULL_DUPLEX,
    MD_SPI_FRAME_FORMAT_8BIT,
    MD_SPI_NSS_HARD,
};


/* Private variables ----------------------------------------------------------*/
/**
  * @brief SPI state.
  */
typedef enum
{
    TX,
    RX,
    IDEL
} txrx_status_enum;

/* Tx data buffer */
uint8_t SPI_TxBuffer[BUFFER_SIZE] = "eastsoft demo";
/* Tx buffer index */
volatile uint8_t TxBufferIndex = 0;
/* Rx data buffer */
uint8_t SPI_RxBuffer[BUFFER_SIZE] = {0};
/* Rx buffer index */
volatile uint8_t RxBufferIndex = 0;

/* Status flag */
txrx_status_enum TxRxStatus = IDEL;

/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void spi_tx_handle(void);
void spi_rx_handle(void);
/* Public functions -----------------------------------------------------------*/

void iomux()
{
    /* UART1 io mux */
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_output_type(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_OUTPUT_PUSHPULL);
    md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_3,  MD_GPIO_PULL_UP);

    /* SPI1 io mux */
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_2, MD_GPIO_AF1);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_3, MD_GPIO_AF1);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_4, MD_GPIO_AF1);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_5, MD_GPIO_AF1);

    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_2, MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_3, MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_4, MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_5, MD_GPIO_MODE_FUNCTION);
}

void per_enable()
{

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_spi1(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiod(RCU);

}


/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    int i;

    __disable_irq();
    md_rcu_init_set(RCU, &rcu_initStruct);
    per_enable();
    __enable_irq();

    iomux();

    md_uart_init(UART1, &uart_initStruct);

    spi_txit_callback = spi_tx_handle;
    spi_rxit_callback = spi_rx_handle;
    md_spi_init(SPI1, &SPI1_InitStruct);



    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);

    NVIC->ICER[0] = 0xFFFFFFFFUL;
    NVIC_EnableIRQ((IRQn_Type) SPI1_IRQn);

    TxRxStatus = TX;
    md_spi_enable_it_txe(SPI1);

    while (TxRxStatus == TX);

    for (i = 0; i < 1000; i++);

//  printf("IVS=%x\r\n",SPI1->IVS);
//  while (md_spi_get_rxfifo_value(SPI1) > 0);

//  printf("rx_LV:%d\r\n",md_spi_get_rxfifo_value(SPI1));
//  md_spi_get_data_reg(SPI1);
    md_spi_clear_it_rxth(SPI1);

    md_spi_enable_it_rxth(SPI1);

//  TxRxStatus = RX;
    while (TxRxStatus != RX);

    //RxBufferIndex--;
    printf("IVS=%x\r\n", SPI1->IVS);

    for (i = 0; i < BUFFER_SIZE; i++)
    {
        while (md_spi_is_active_flag_txf(SPI1));

        md_spi_set_data_reg(SPI1, i);
//    printf("%d\r\n",md_spi_get_rxfifo_value(SPI1));
    }

    while (TxRxStatus == RX);

    for (i = 0; i < BUFFER_SIZE; i++)
        printf("SPI_RxBuffer[%d]=%x\r\n", i, SPI_RxBuffer[i]);

//  printf("RXTH IVS:%d\r\n",md_spi_is_enabled_it_rxth(SPI1));

    while (1);



}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (UART1->STAT & (UART_STAT_TFFULL_MSK));  /* Tx FIFO full */

    UART1->TXBUF = ch;            /* Sent byte */
    return (ch);
}


/**
  * @brief  spi_tx_handle.
  * @param  None
  * @retval None
  */
void spi_tx_handle(void)
{
    volatile uint8_t temp;

    if (TxRxStatus == TX)
    {
        /* Send data */
        md_spi_set_data_reg(SPI1, SPI_TxBuffer[TxBufferIndex++]);

        if (TxBufferIndex == BUFFER_SIZE)
        {
            TxRxStatus = RX;
            TxBufferIndex = 0;
        }
    }
    else
    {
        while (md_spi_is_active_flag_busy(SPI1) == 1);

        //printf("%d\r\n",md_spi_get_rxfifo_value(SPI1));
        while (md_spi_get_rxfifo_value(SPI1) > 0)
        {
            temp = md_spi_get_data_reg(SPI1);
//      printf("%c\r\n",(char)temp);
        }

//    printf("%d\r\n",md_spi_get_rxfifo_value(SPI1));
        md_spi_disable_it_txe(SPI1);
    }
}

/**
  * @brief  spi_rx_handle.
  * @param  None
  * @retval None
  */
void spi_rx_handle(void)
{
//  printf("rx_handle %d\r\n",RxBufferIndex);
    uint8_t data_temp;
    /* Read data and clear flag */
    data_temp = md_spi_get_data_reg(SPI1);

//  printf("%d\r\n",data_temp);
    if (TxRxStatus == RX)
    {
        SPI_RxBuffer[RxBufferIndex] = data_temp;
        RxBufferIndex++;

        if (RxBufferIndex == 14)
        {
            /* Stop Rx */
            RxBufferIndex = 0;
            TxRxStatus = IDEL;
            md_spi_disable_it_rxth(SPI1);
        }
    }

//  else
//  {
//    md_spi_disable_it_rxth(SPI1);
//  }
}


/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


