/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK),
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_spi_inittypedef SPI1_InitStruct =    /**< SPI init structure */
{
    MD_SPI_MODE_MASTER,
    MD_SPI_PHASE_1EDGE,
    MD_SPI_POLARITY_LOW,
    MD_SPI_BAUDRATEPRESCALER_DIV8,
    MD_SPI_MSB_FIRST,
    MD_SPI_FULL_DUPLEX,
    MD_SPI_FRAME_FORMAT_8BIT,
    MD_SPI_NSS_HARD,
};


/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
/* Public functions -----------------------------------------------------------*/

void iomux()
{
    /* UART1 io mux */
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_output_type(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_OUTPUT_PUSHPULL);
    md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_3,  MD_GPIO_PULL_UP);

    /* SPI1 io mux */
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_2, MD_GPIO_AF1);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_3, MD_GPIO_AF1);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_4, MD_GPIO_AF1);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_5, MD_GPIO_AF1);

    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_2, MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_3, MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_4, MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_5, MD_GPIO_MODE_FUNCTION);
}


/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t data[32];
    int i, j = 0;

    __disable_irq();
    md_rcu_init_set(RCU, &rcu_initStruct);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_spi1(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiod(RCU);
    __enable_irq();



    iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_spi_init(SPI1, &SPI1_InitStruct);

    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);
    printf("\r\n========================\r\n");
    printf(" SPI Master Tx example\r\n");
    printf(" Mode: Master\r\n");
    printf(" CPHA: First edge \r\n");
    printf(" CPOL: Clk to 0 when idle \r\n");
    printf(" BaudRate: divide 8 \r\n");
    printf(" Communication mode: Full-duplex \r\n");
    printf(" Data size: 8-bit \r\n");
    printf(" NSS: hardware control \r\n");
    printf("========================\r\n");

    while (1)
    {
        printf("Press to start master Tx\r\n");

        while (md_uart_is_active_flag_rfempty(UART1));

        *data = md_uart_get_recv_data8(UART1);


        for (i = 0; i < 5; i++)
        {
            while (md_spi_is_active_flag_txf(SPI1));

            md_spi_set_data_reg(SPI1, 0x00 + j);
            j++;

            while (md_spi_is_active_flag_rxe(SPI1));

            data[i] = md_spi_get_data_reg(SPI1);
        }

        for (i = 0; i < 5; i++)
            printf("data[%d] = %x\r\n", i, data[i]);
    }
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (UART1->STAT & (UART_STAT_TFFULL_MSK));  /* Tx FIFO full */

    UART1->TXBUF = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


