/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes--------------------------------------------------------*/
#include "main.h"
#include "md_mmc.h"
#include <math.h>
/** @addtogroup Projects_Examples_MD
  * @{
  */


/* Public Functions -----------------------------------------------------------*/
/* Public Init Structure ------------------------------------------------------*/
/* Public Macros --------------------------------------------------------------*/
/* Public Constants -----------------------------------------------------------*/
/* Private Variables ----------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK),
};
/* Private Macros -------------------------------------------------------------*/
#define GREF_VOLT       1.769
#define VREF_VOLT       2.962
#define TWO_POWER_23    8388608

/* Private Constants-----------------------------------------------------------*/

/* Private Functions ----------------------------------------------------------*/

void SysInit(void);
void IrqInit(void);
void iomux(void);
void UARTInit(void);
void MMCInit(void);
void ACMInit(uint32_t);
double md_adc_current_tempture(double ADC_Vref_P_voltage, double ADC_Vref_N_voltage);
/**
  * @brief main.
  * @param none
  * @retval none
  */
int main(void)
{
    md_rcu_init_set(RCU, &rcu_initStruct);
    SysInit();
    IrqInit();
    UARTInit();
    iomux();
    ACMInit(MMC_ADC_CTRL0_SEL_VCM_VDDA_DIV2);    
    MMCInit();;
    bsp_key_init();
    bsp_led_init();
    printf("MMC ADC Tempture demo\r\n");


    printf("ADIP=TS\r\n");
    printf("ADIN=VSS\r\n");
    printf("ADVRP=VDDA=3.3V\r\n");
    printf("ADVRN=VSS=0V\r\n");
    printf("input voltage=ADIP-ADIN=AIN1-AIN0\r\n\r\n");
    printf("VDDA needs an external 3.3V\r\n");
    printf("Press any key continue\r\n");

    while (md_uart_is_active_flag_rfempty(UART1));


    while (1)
    {
        printf("cuurent tempture:%.15lf\r\n", md_adc_current_tempture(3.3, 0));
    }
}

double md_adc_current_tempture(double ADC_Vref_P_voltage, double ADC_Vref_N_voltage)
{
    uint32_t adc_val = 0;
    uint32_t cnt = 0;
    double Vts = 0;
    double Vt30 = 0;
    const double tempture_slope = 0.003595f;

    for (cnt = 0; cnt < 8; cnt++)
    {
        while (md_mmc_is_active_it_adc_done(MMC) == 0);

        md_mmc_clear_it_icr_adc_done(MMC);

        if (cnt > 3)
        {
            // Abandon 4 data
            adc_val = adc_val + (md_mmc_get_adc_val(MMC) >> 8);
        }
    }

    adc_val /= 4;

    Vts = (double)(((double)adc_val / (double)(1U << 23)) * (double)(ADC_Vref_P_voltage - ADC_Vref_N_voltage));
    Vt30 = md_fc_read_adc_temperature_voltage();
    printf("Vts:%.15lf\t", Vts);

    return ((Vts - Vt30) / tempture_slope) + 30;
}

void SysInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_disable_hdiven(RCU);

    md_rcu_disable_suart1en(RCU);
    md_rcu_disable_iwdten(RCU);
    md_rcu_disable_wwdt(RCU);
    md_rcu_disable_bs16t1(RCU);
    md_rcu_disable_gp32c4t1(RCU);

    md_rcu_disable_gp16c2t2(RCU);
    md_rcu_disable_gp16c2t1(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_disable_spi1(RCU);
    md_rcu_enable_mmcen(RCU);
    md_rcu_disable_lcden(RCU);

    md_rcu_disable_ipcfg_mmcen(RCU);
    md_rcu_set_ipcfg_mmcdiv(RCU, 1);
    md_rcu_set_ipcfg_mmcsw_hrc(RCU);


    md_rcu_set_ipcfg_vpumpsw_hrc(RCU);
    md_rcu_disable_ipcfg_vpumpen(RCU);
    md_rcu_set_ipcfg_vpumpdiv(RCU, 4);

    md_rcu_enable_ipcfg_mmcen(RCU);
    md_rcu_enable_ipcfg_vpumpen(RCU);
}

void IrqInit(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
}

void iomux(void)
{
    md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_AF0);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_MODE_FUNCTION);
}

void UARTInit(void)
{
    md_uart_disable_tx(UART1);
    md_uart_disable_rx(UART1);
    md_uart_set_parity(UART1, MD_UART_LCON_PS_EVEN);
    md_uart_set_stop(UART1, MD_UART_LCON_STOP_1);
    md_uart_set_datawidth(UART1, MD_UART_LCON_DLS_8);
    md_uart_enable_tfrst(UART1);
    md_uart_enable_rfrst(UART1);
    md_uart_set_baudrate_calculate(UART1, 115200);
    md_uart_enable_tx(UART1);
    md_uart_enable_rx(UART1);
}

/**
  * @brief  ACM Initial function
  * @param  select: Analog-to-Digital Converter Common-Mode Voltage Selection
            @arg MMC_ADC_CTRL0_SEL_VCM_VDDA_DIV2
            @arg MMC_ADC_CTRL0_SEL_VCM_1_2V
  * @retval none
  */
void ACMInit(uint32_t select)
{
#define CHIP_ID_ADDR 0x0898
#define QFN32_CHIP_ID 0x099102C0
    
    uint32_t chip_id;
    
    md_fc_read_info(CHIP_ID_ADDR, &chip_id);
    printf("chipid: 0x%x\r\n", chip_id);
    if(chip_id == QFN32_CHIP_ID)
        SYSCFG->CFG |= 1<<8;  
    
    md_mmc_set_sel_vcm(MMC, select);
    md_mmc_enable_ad_acmbuf(MMC);
}

void MMCInit(void)
{
    //MMC Power Setting
    md_mmc_disable_en_func_op(MMC);
    md_mmc_enable_en_vdda_float(MMC);
    md_mmc_enable_en_vdda(MMC);
    md_mmc_enable_en_mmc(MMC);

    md_mmc_set_sel_frq(MMC, MD_MMC_ADC_SEL_FRQ_FREF_DIV_10);

    // Enable temperature sensor
    md_mmc_enable_en_ts(MMC);

    //ADC sample point
    md_mmc_set_ad_nfilt(MMC, MD_MMC_ADC_CTRL0_AD_NFILT_VSS);
    md_mmc_set_ad_pfilt(MMC, MD_MMC_ADC_CTRL0_AD_PFILT_TS); //

    //enable ADC sample point buffer
    md_mmc_enable_adi_pbuf(MMC);
    md_mmc_enable_adi_nbuf(MMC);

    //setting pfilt channel
    md_mmc_set_ad_pfilt_ch(MMC, MD_MMC_ADC_CTRL0_PFILT_CH_NOFILTER);

    //ADC reference point
    md_mmc_set_sel_nadr(MMC, MD_MMC_ADC_SEL_NADR_VSS);    //ADVRN=VSS=0V
    md_mmc_set_sel_padr(MMC, MD_MMC_ADC_SEL_PADR_VDDA);   //ADVRP=VDDA=3.2V

    //enable ADC reference point buffer
    md_mmc_disable_adr_pbuf(MMC);
    md_mmc_disable_adr_nbuf(MMC);

    //ADC sample point
    md_mmc_set_sel_adi(MMC, MD_MMC_ADC_SEL_ADI_ADSFP_AND_ADSFN);

    //ADC OSR
    md_mmc_set_ad_osr(MMC, MD_MMC_ADC_AD_OSR_20000);

    //reset adc&comb
    md_mmc_set_rst_ad(MMC, MD_MMC_ADC_RESET);
    md_tick_wait10us(10, 2);
    md_mmc_set_rst_ad(MMC, MD_MMC_ADC_RUN);
    md_mmc_is_clear_flag_rst_comb(MMC);

    //enable MMC ADC
    md_mmc_enable_en_ad(MMC);
    md_mmc_enable_en_comb(MMC);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t sendchar(uint8_t ch)
{
    while (UART1->STAT & (UART_STAT_TFFULL_MSK));  // Tx FIFO full

    UART1->TXBUF = ch;            // Sent byte
    return (ch);
}


/**
  * @}Projects_Examples_MD
  */
/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
