/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK),
};

md_rcu_lcd_init_typedef lcd_init =
{
    MD_RCU_IPCFG_VLCDCPSW_HRC,
    4,
    RCU_IPCFG_VLCDEN_MSK,
    MD_RCU_IPCFG_LCDSW_PCLK,
    0xFF
};

md_lcd_mode_typedef lcd_mode =
{
    LCD_MODE_CPEN_MSK,
    LCD_MODE_BUFEN_MSK,
    MD_LCD_MODE_VLS_VLCD_2V55,
    MD_LCD_MODE_RSEL_225K,
    MD_LCD_MODE_TYPE_A,
    MD_LCD_MODE_BIAS_ONETHIRD,
    MD_LCD_MODE_DUTY_ONEQUARTER
};

md_lcd_ctrl_typedef lcd_ctrl =
{
    8,
    MD_LCD_CTRL_UPDCTRL_EVERYFRAME,
    MD_LCD_CTRL_WSBLINK_DISABLE,
    1,
    0,
    0,
    1
};
md_lcd_comx_segx_typedef lcd_comx_segx =
{
    MD_LCD_COMEN_COM_EN_COM0 | MD_LCD_COMEN_COM_EN_COM1 | MD_LCD_COMEN_COM_EN_COM2 | MD_LCD_COMEN_COM_EN_COM3,
    MD_LCD_SEGEN_SEG_EN_SEG0 | MD_LCD_SEGEN_SEG_EN_SEG1 | MD_LCD_SEGEN_SEG_EN_SEG2 | MD_LCD_SEGEN_SEG_EN_SEG3 | MD_LCD_SEGEN_SEG_EN_SEG4 | MD_LCD_SEGEN_SEG_EN_SEG5 | MD_LCD_SEGEN_SEG_EN_SEG6 | MD_LCD_SEGEN_SEG_EN_SEG7 | MD_LCD_SEGEN_SEG_EN_SEG8 | MD_LCD_SEGEN_SEG_EN_SEG9 | MD_LCD_SEGEN_SEG_EN_SEG10 | MD_LCD_SEGEN_SEG_EN_SEG11 | MD_LCD_SEGEN_SEG_EN_SEG12 | MD_LCD_SEGEN_SEG_EN_SEG13 | MD_LCD_SEGEN_SEG_EN_SEG14 | MD_LCD_SEGEN_SEG_EN_SEG15
};
/* Private types --------------------------------------------------------------*/
void TimerInit(void)
{
    GP32C4T1->AR    = TIMER_PERIOD_1KHZ - 1; /* Set Auto-reload value (1ms)*/
    GP32C4T1->PRES  = 0;                     /* Set prescaler value (0+1)*/
    GP32C4T1->ICR   = (1 << 0);               /* Update interrupt is cleared(TIMx_RIF). clear interrupt*/
    GP32C4T1->IER   = (1 << 0);               /* Update interrupt is enabled.enable interrupt*/
    GP32C4T1->CON1  = (1 << 7) | (1 << 4) | (1 << 0); /* Auto-reload preload enable;Counter used as downcounter;Counter enabled*/
}

/**
  * @brief  IrqInit
  * @param  None
  * @retval None
  */
void IrqInit(void)
{
//  NVIC->ICER[0] = 0xFFFFFFFFUL;

    NVIC_EnableIRQ((IRQn_Type) GP32C4T1_IRQn);  // GP32C4T1_IRQn interrupt
}

/**
  * @brief  set irq priority
  * @param  None
  * @retval None
  */
void IrqPriority(void)
{
    NVIC_SetPriority((IRQn_Type) GP32C4T1_IRQn,  0);  //priority 0
}

void Iomux(void)
{
    md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_AF0);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_MODE_FUNCTION);

    md_gpio_set_pin_pull(GPIOD, MD_GPIO_PIN_2,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_pull(GPIOD, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_2,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_2,   MD_GPIO_MODE_OUTPUT);
    md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_3,   MD_GPIO_MODE_OUTPUT);
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_disable_hdiven(RCU);

    md_rcu_disable_suart1en(RCU);
    md_rcu_disable_iwdten(RCU);
    md_rcu_disable_wwdt(RCU);
    md_rcu_disable_bs16t1(RCU);
    md_rcu_enable_gp32c4t1(RCU);

    md_rcu_disable_gp16c2t2(RCU);
    md_rcu_disable_gp16c2t1(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_disable_spi1(RCU);
    md_rcu_disable_mmcen(RCU);
    md_rcu_enable_lcden(RCU);

}

/**
  * @brief  uart1 init
  * @param  None
  * @retval None
  */
void UartInit(void)
{
    md_uart_set_baudrate_calculate(UART1, MD_UART_BAUDRATE_115200);
    md_uart_disable_tx(UART1);
    md_uart_disable_rx(UART1);
    md_uart_set_datawidth(UART1, MD_UART_LCON_DLS_8);
    md_uart_set_stop(UART1, MD_UART_LCON_STOP_1);
    md_uart_set_parity(UART1, MD_UART_LCON_PS_EVEN);
    md_uart_enable_tfrst(UART1);
    md_uart_enable_rfrst(UART1);
    md_uart_enable_tx(UART1);
    md_uart_enable_rx(UART1);
}


/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t u32LedTime, u32Led5Timer, u32GpioC_Sts;

    __disable_irq();
    md_rcu_init_set(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    UartInit();
    TimerInit();
    IrqPriority();
    IrqInit();
    md_rcu_lcd_init(RCU, &lcd_init);
    md_lcd_mode(LCD, &lcd_mode);
    md_lcd_comx_segx_enable(LCD, &lcd_comx_segx);
    md_lcd_ctrl(LCD, &lcd_ctrl);
    __enable_irq();

    bsp_led_init();
    bsp_key_init();
    md_lcd_display_all();

    md_lcd_clear_all();

    printf("On %s at %s", __DATE__, __TIME__);
    u32LedTime  = 500;  /* 500*1ms */
    u32GpioC_Sts = 0x00000001;
    UpdateTime(u32Led5Timer);

    while (1)
    {


        if (CheckTime(u32Led5Timer) >= u32LedTime)
        {
            UpdateTime(u32Led5Timer);
            u32GpioC_Sts ^= 0x00000001;

            if (u32GpioC_Sts)
            {
//                md_gpio_set_pin_high(GPIOD, MD_GPIO_PIN_2);
//                md_gpio_set_pin_low(GPIOD, MD_GPIO_PIN_3);
                bsp_led3_on();
                bsp_led4_off();
                md_lcd_clear_all();
                md_lcd_display_float_number(9876);

            }
            else
            {
//                md_gpio_set_pin_high(GPIOD, MD_GPIO_PIN_3);
//                md_gpio_set_pin_low(GPIOD, MD_GPIO_PIN_2);
                bsp_led3_off();
                bsp_led4_on();
                md_lcd_clear_all();
                md_lcd_display_float_number(54.32);
            }
        }
    }
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (UART1->STAT & (UART_STAT_TFFULL_MSK));    /* Tx FIFO full */

    UART1->TXBUF = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


