/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    25 Oct 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          25 Oct 2023     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

uint8_t intrEXTI  = 0;

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK),
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};
/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
/* Public functions -----------------------------------------------------------*/

void uart1_iomux()
{
    md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_AF0);
    md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_AF0);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_MODE_FUNCTION);
}

void EXTI_setting()
{
    md_syscfg_enable_lvd(SYSCFG);
    md_syscfg_set_pwcon_pvdls(SYSCFG, MD_SYSCFG_PVDLS_R2V9_F3V);
    
    md_exti_enable_it_pvd(EXTI);
    md_exti_enable_rising_edge_trigger(EXTI, MD_EXTI_EVENT_PVD);
    md_exti_enable_falling_edge_trigger(EXTI, MD_EXTI_EVENT_PVD);
}

void irq_init()
{
    NVIC_EnableIRQ((IRQn_Type) PVD_IRQn);          /* PVD interrupt */    
}

void per_enable()
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_disable_hdiven(RCU);

    md_rcu_disable_suart1en(RCU);
    md_rcu_disable_iwdten(RCU);
    md_rcu_disable_wwdt(RCU);
    md_rcu_disable_bs16t1(RCU);
    md_rcu_disable_gp32c4t1(RCU);

    md_rcu_disable_gp16c2t2(RCU);
    md_rcu_disable_gp16c2t1(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_disable_spi1(RCU);
    md_rcu_disable_mmcen(RCU);
    md_rcu_disable_lcden(RCU);
}

uint8_t UARTByteRd(UART_TypeDef *UARTx)
{
    while (md_uart_is_active_flag_rfempty(UARTx));           // Rx FIFO empty

    return (md_uart_get_recv_data8(UARTx));
}
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t flag = 1;     

    __disable_irq();
    md_rcu_init_set(RCU, &rcu_initStruct);
    per_enable();
    __enable_irq();

    irq_init();
    uart1_iomux();
    md_uart_init(UART1, &uart_initStruct);
    EXTI_setting();
    bsp_led_init();
    bsp_key_init();

    printf("EXTI LVD example\r\n");
    
    bsp_led1_off();
    bsp_led2_off();
    bsp_led3_off();
    bsp_led4_off();   

    intrEXTI = 0;   

    while (1)
    {
        while(intrEXTI == 0);
        
        intrEXTI = 0;
        printf("Detect LVD Flag Interrupt!! \r\n");
        flag ^=1;
        if(flag)
        {
            bsp_led1_off();
            bsp_led3_on();
        }
        else
        {
            bsp_led1_on();
            bsp_led3_off();
        }
    }
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (UART1->STAT & (UART_STAT_TFFULL_MSK));  /* Tx FIFO full */

    UART1->TXBUF = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


