/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    10 Mar 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          10 Mar 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"


md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HOSCON_MSK | RCU_CON_HRCON_MSK),
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Public Variables ---------------------------------------------------------- */
uint32_t  len_hex;

/* Public Function ---------------------------------------------------------- */


/* Private Macros ------------------------------------------------------------ */


/* Private Variables--------------------------------------------------------- */

//static md_uart_init_t s_init;

/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Private Function ---------------------------------------------------------- */
/**
  * @brief  init pins used fot uart communication.
  * @param  None
  * @retval None
  */
void gpio_init(void)
{
    md_gpio_inittypedef x;

    /* LED1 */
    x.Pin   = MD_GPIO_PIN_5;
    x.Mode  = MD_GPIO_MODE_OUTPUT;
    x.OutputType  = MD_GPIO_OUTPUT_PUSHPULL;
    x.Pull  = MD_GPIO_PULL_UP;
    x.OutDrive  = MD_GPIO_DS_8mA;
    x.Function  = MD_GPIO_AF0;
    md_gpio_init(GPIOC, &x);

    /* LED2 */
    x.Pin   = MD_GPIO_PIN_0;
    x.Mode  = MD_GPIO_MODE_OUTPUT;
    x.OutputType  = MD_GPIO_OUTPUT_PUSHPULL;
    x.Pull  = MD_GPIO_PULL_UP;
    x.OutDrive  = MD_GPIO_DS_8mA;
    x.Function  = MD_GPIO_AF0;
    md_gpio_init(GPIOD, &x);
    
    /* LED3 */
    x.Pin   = MD_GPIO_PIN_6;
    x.Mode  = MD_GPIO_MODE_OUTPUT;
    x.OutputType  = MD_GPIO_OUTPUT_PUSHPULL;
    x.Pull  = MD_GPIO_PULL_UP;
    x.OutDrive  = MD_GPIO_DS_8mA;
    x.Function  = MD_GPIO_AF0;
    md_gpio_init(GPIOD, &x);

    /* LED4 */
    x.Pin   = MD_GPIO_PIN_2;
    x.Mode  = MD_GPIO_MODE_OUTPUT;
    x.OutputType  = MD_GPIO_OUTPUT_PUSHPULL;
    x.Pull  = MD_GPIO_PULL_UP;
    x.OutDrive  = MD_GPIO_DS_8mA;
    x.Function  = MD_GPIO_AF0;
    md_gpio_init(GPIOC, &x);
    
    md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_5);
    md_gpio_set_pin_low(GPIOD, MD_GPIO_PIN_0);
    md_gpio_set_pin_low(GPIOD, MD_GPIO_PIN_6);
    md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_2);  

    return;
}

/**
  * @brief  init pins used fot uart communication.
  * @param  None
  * @retval None
  */
void uart_pins_init(void)
{
    md_gpio_inittypedef x;

    /* Uart1 Initialize tx pin */
    x.Pin   = MD_GPIO_PIN_4;
    x.Mode  = MD_GPIO_MODE_FUNCTION;
    x.OutputType  = MD_GPIO_OUTPUT_PUSHPULL;
    x.Pull  = MD_GPIO_PULL_UP;
    x.OutDrive  = 0;
    x.Function  = MD_GPIO_AF0;
    md_gpio_init(GPIOC, &x);

    /* Uart1 Initialize rx pin */
    x.Pin   = MD_GPIO_PIN_3;
    x.Mode  = MD_GPIO_MODE_FUNCTION;
    x.OutputType  = MD_GPIO_OUTPUT_PUSHPULL;
    x.Pull  = MD_GPIO_PULL_UP;
    x.OutDrive  = 0;
    x.Function  = MD_GPIO_AF0;
    md_gpio_init(GPIOC, &x);

    /* Uart1 Initialize tx pin */
    x.Pin   = MD_GPIO_PIN_7;
    x.Mode  = MD_GPIO_MODE_FUNCTION;
    x.OutputType  = MD_GPIO_OUTPUT_PUSHPULL;
    x.Pull  = MD_GPIO_PULL_UP;
    x.OutDrive  = 0;
    x.Function  = MD_GPIO_AF0;
    md_gpio_init(GPIOD, &x);

    /* Uart1 Initialize rx pin */
    x.Pin   = MD_GPIO_PIN_8;
    x.Mode  = MD_GPIO_MODE_FUNCTION;
    x.OutputType  = MD_GPIO_OUTPUT_PUSHPULL;
    x.Pull  = MD_GPIO_PULL_UP;
    x.OutDrive  = 0;
    x.Function  = MD_GPIO_AF0;
    md_gpio_init(GPIOD, &x);

    return;
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_suart1en(RCU);
}

/**
  * @brief  Tick Init
  * @note   Set Tick Interrupt 1ms
  * @param  None
  * @retval None
  */
void  tick_init(void)
{
    md_tick_set_rvr_reload(TICK, 16000 - 1);                  /*  Maximun ReLoad  */
    md_tick_set_cvr_current(TICK, 0);                         /*  Reset CVR  */
    md_tick_set_csr_clksrc(TICK, MD_SYSTICK_CLKSRC_MCU);      /*  MCU Clock  */
    md_tick_enable_csr_tickie(TICK);                          /*  Tick Interrupt */
    md_tick_enable_csr_enable(TICK);                          /*  Counter Enable  */
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_rcu_init_set(RCU, &rcu_initStruct);

    /* Initialize SysTick Interrupt */
    tick_init();
    /* Enable ALL peripheral */
    SysPeriInit();

    /* Config uart1 interrupt */
    NVIC_SetPriority(UART1_IRQn, 0x01);
    NVIC_EnableIRQ(UART1_IRQn);
    
    gpio_init();    

    /* Init communication parameters */
    uart_pins_init();

    md_uart_init(UART1, &uart_initStruct);
    md_uart_init(SUART1, &uart_initStruct);
    md_uart_set_txth(UART1, MD_UART_FCON_TXTH_0);
    md_uart_set_rxth(UART1, MD_UART_FCON_RXTH_1);
    md_uart_enable_it_rfth(UART1);

    printf("SystemFrequency_SysClk:%d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk:%d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk:%d\r\n", SystemFrequency_APBClk);

    /* Init state machine */
    fsm_comm_func_init();
    fsm_exec_func_init();
    fsm_para_init();

    while (1)
    {
        while (fsm_is_trigger())
        {
            proc_fsm();
        }

    }
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (SUART1->STAT & (1 << 17)); /* Tx FIFO full */

    SUART1->TXBUF = ch;            /* Sent byte */
    return (ch);
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
