/**********************************************************************************
 *
 * @file    exec_proc.c
 * @brief   define the exection functions for state machine
 *
 * @date    10 Mar 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          10 Mar 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "exec_proc.h"
#include "main.h"


/* Private Macros ------------------------------------------------------------ */
typedef  void (*FunVoidType)(void);

/* Public Variables ---------------------------------------------------------- */


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Public function prototypes ----------------------------------------------- */


/* Private Variables--------------------------------------------------------- */


/* Public Function ---------------------------------------------------------- */


/* Private Function ---------------------------------------------------------- */

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    md_rcu_enable_gpiod_reset(RCU);
    md_rcu_enable_gpioc_reset(RCU);
    md_rcu_enable_uart1_reset(RCU);
}

/**
  * @brief  define the page erase function.
  * @param  None
  * @retval None
  */
static fsm_iap_result_t fsm_page_erase(uint32_t PageCode)
{
    return IAP_FAIL;
}

/**
  * @brief  define the words program function.
  * @param  None
  * @retval None
  */
static fsm_iap_result_t fsm_words_program(uint32_t address, uint32_t data_address, uint32_t len)
{
    return IAP_FAIL;
}

/**
  * @brief  define Reverse8 function.
  * @param  None
  * @retval None
  */

uint8_t     Reverse8(uint8_t U8)
{
    U8 = (((U8 & 0xaa) >> 1) | ((U8 & 0x55) << 1));
    U8 = (((U8 & 0xcc) >> 2) | ((U8 & 0x33) << 2));
    U8 = (((U8 & 0xf0) >> 4) | ((U8 & 0x0f) << 4));
    return (U8);
}

/**
  * @brief  define Reverse32 function, used to crc check.
  * @param  None
  * @retval None
  */

uint32_t    Reverse32(uint32_t U32)
{
    U32 = (((U32 & 0xaaaaaaaa) >> 1) | ((U32 & 0x55555555) << 1));
    U32 = (((U32 & 0xcccccccc) >> 2) | ((U32 & 0x33333333) << 2));
    U32 = (((U32 & 0xf0f0f0f0) >> 4) | ((U32 & 0x0f0f0f0f) << 4));
    U32 = (((U32 & 0xff00ff00) >> 8) | ((U32 & 0x00ff00ff) << 8));
    U32 = (((U32 & 0xffff0000) >> 16) | ((U32 & 0x0000ffff) << 16));
    return (U32);
}

/**
  * @brief  define the checksum function, used to crc check.
  * @param  None
  * @retval None
  */
static uint32_t fsm_get_crc32(uint32_t *data_ptr, uint32_t len)
{
    uint32_t    crc32, poly, data32;
    uint8_t     data8;
    uint8_t     ii;
    uint32_t    *pU32Buf;

    pU32Buf = (uint32_t *)((uint32_t)data_ptr + APP_ADDR);

    crc32 = 0xffffffffUL;
    poly = 0x04c11db7;
    len >>= 2;

    while (len--)
    {
        data32 = *pU32Buf++;

        for (ii = 0 ; ii < 4 ; ii++)
        {
            data8 = data32 & 0xff;
            data8 = Reverse8(data8);
            data32 >>= 8;
            data8 ^= (crc32 >> 24);
            crc32 <<= 8;

            if (data8 & 0x80) crc32 ^= 0x690CE0EE; // (poly << 7) ^ (poly << 1)

            if (data8 & 0x40) crc32 ^= 0x34867077; // (poly << 6) ^ (poly << 0)

            if (data8 & 0x20) crc32 ^= 0x9823B6E0; // (poly << 5)

            if (data8 & 0x10) crc32 ^= 0x4C11DB70; // (poly << 4)

            if (data8 & 0x08) crc32 ^= 0x2608EDB8; // (poly << 3)

            if (data8 & 0x04) crc32 ^= 0x130476DC; // (poly << 2)

            if (data8 & 0x02) crc32 ^= 0x09823B6E; // (poly << 1)

            if (data8 & 0x01) crc32 ^= 0x04C11DB7; // (poly << 0)
        }
    }

    crc32 = ~Reverse32(crc32);

    return crc32;
}

/**
  * @brief  define the check empty function.
  * @param  None
  * @retval None
  */
static fsm_result_t fsm_check_empty(uint32_t *data_ptr, uint32_t len)
{
    return FAIL;
}

/**
  * @brief  define the memory read function.
  * @param  None
  * @retval None
  */
static void fsm_read_memo(uint8_t *dest, void *src, uint32_t len)
{
    return;
}

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    uint32_t addr;       
    md_fc_ControlTypeDef  FCControlPara;
    uint16_t  page, ii;    
//    printf("%s", __FUNCTION__);

    // Lock FC_CMD Register
    md_fc_lock();
    
    if(para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if(para == GO_BOOT) 
    {
        addr = BOOT_ADDR;
        printf("GO_BOOT\r\n");
        printf("\r\n"); 
        page = CRC_CAL_ADDR_PAGE;
        FCControlPara.u32SAddr = page * MD_FC_PC_EF_MPAGESZ;
        FCControlPara.u32SAddrC = ~(page * MD_FC_PC_EF_MPAGESZ);
        FCControlPara.u16BCnt = MD_FC_PC_EF_MPAGESZ;
        md_fc_unlock();
        md_fc_page_erase(&FCControlPara);
        md_fc_lock(); 
    }      

    // REMAP
    // 0x00: EFLASH         -> Remap EFLASH Any 4K
    // 0x01: Bootloader     -> Remap EFLASH 0x00000000
    // 0x02: SRAM           -> Remap SRAM

    // Remap EFLASH
    md_syscfg_set_memory_mapping(SYSCFG, MD_SYSCFG_MEMMOD_MAIN);
    md_syscfg_set_flash_remap_base(SYSCFG, addr / (4 * 2 * MD_FC_PC_EF_MPAGESZ));
    md_syscfg_enable_memory_remap(SYSCFG);
    // Reset MCU
    NVIC_SystemReset();

    while (1);

    return;
}

/**
  * @brief  assign function pointer to all the state machine subfunction.
  * @param  None
  * @retval None
  */
void fsm_exec_func_init(void)
{
    g_isp_data.p_page_erase   = &fsm_page_erase;
    g_isp_data.p_words_progrm = &fsm_words_program;
    g_isp_data.p_get_crc32    = &fsm_get_crc32;
    g_isp_data.p_check_empty  = &fsm_check_empty;
    g_isp_data.p_go           = &fsm_go;
    g_isp_data.p_read_memo    = &fsm_read_memo;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
