/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for demo
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples_ADC
  * @{
  */


/**
  * @brief  adc_handle_t structure.
  */
 
#define USR_ADC_CH_NR	4 
md_adc_init_t h_adc;

extern md_dma_descriptor_t md_dma0_ctrl_base[28];
md_dma_config_t dma_cfg = {0};
uint16_t adc_buf[USR_ADC_CH_NR];
volatile uint8_t cflag = 0;

/**
  * @brief  Initializate pin of adc module.
  * @retval None
  */
void adc_pin_init(void)
{
	md_gpio_init_t x;

	memset(&x, 0x0, sizeof(md_gpio_init_t));
	
	/* Initialize sck pin */
	x.mode = MD_GPIO_MODE_INPUT;
	x.pupd = MD_GPIO_PUSH_UP;
	x.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
	x.flt  = MD_GPIO_FILTER_DISABLE;
	x.type = MD_GPIO_TYPE_CMOS;
	x.func = MD_GPIO_FUNC_0;
	
	md_gpio_init(GPIO_CH14_PORT, ADC_CH14_PIN, &x);
	md_gpio_init(GPIO_CH0_PORT, ADC_CH0_PIN, &x);
	md_gpio_init(GPIO_CH2_PORT, ADC_CH2_PIN, &x);
	md_gpio_init(GPIO_CH3_PORT, ADC_CH3_PIN, &x);
}

void dma_prepare(void)
{
	md_dma_reset(DMA0);
	memset(md_dma0_ctrl_base, 0x0, sizeof(md_dma0_ctrl_base));
	/* DMA interrupt */
	NVIC_SetPriority(DMA_IRQn, 2);
	NVIC_EnableIRQ(DMA_IRQn);
	md_dma_enable_it_err();
	/* Channel0 descriptor in RAM */
	WRITE_REG(DMA0->CTRLBASE, (uint32_t)&md_dma0_ctrl_base[0]);
	/* Enable DMA controller */
	SET_BIT(DMA0->CFG, DMA_CFG_MASTER_ENABLE_MSK);
}

void dma_init(uint16_t *desbuf)
{
	dma_prepare();
	
	md_dma_config_struct(&dma_cfg);
	dma_cfg.src  = (void*)&ADC0->NCHDR;
	dma_cfg.dst  = (void*)desbuf;
	dma_cfg.size = USR_ADC_CH_NR;
	dma_cfg.data_width = MD_DMA_DATA_SIZE_HALFWORD;
	dma_cfg.src_inc = MD_DMA_DATA_INC_NONE;
	dma_cfg.dst_inc = MD_DMA_DATA_INC_HALFWORD;
	dma_cfg.msel    = MD_DMA_MSEL_ADC0;
	dma_cfg.msigsel = MD_DMA_MSIGSEL_ADC;
	dma_cfg.channel = 0;
	md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &dma_cfg);
	md_dma_enable_channel(MD_DMA_CH_0);	
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	md_adc_nch_conf_t conf;
	
	/* Initialize 1ms tick */
	md_init_1ms_tick();
	/* clear adc_handle_t structure */
	memset(&h_adc, 0x0, sizeof(h_adc));
	/* Initialize adc pin */
	adc_pin_init();

	/* Initialize ADC0 */
	md_adc_struct_init(&h_adc);
	
	h_adc.cont   = DISABLE;
	h_adc.scan   = ENABLE;
	h_adc.align  = MD_ADC_DATAALIGN_RIGHT;
	h_adc.nch_nr = MD_ADC_NCH_NR_4;
	h_adc.data_bit = MD_ADC_CONV_BIT_12;
	h_adc.div      = MD_ADC_CKDIV_4;
	md_adc_init(ADC0, &h_adc);
	
	/* Configure normal channel0 */
	conf.ch   = MD_ADC_CHANNEL_0;
	conf.idx  = MD_ADC_NCH_IDX_1;
	conf.samp = MD_ADC_SAMPLETIME_4;
	md_adc_normal_channel_config(ADC0, &conf);
	
	/* Configure normal channel2 */
	conf.ch   = MD_ADC_CHANNEL_2;
	conf.idx  = MD_ADC_NCH_IDX_2;
	conf.samp = MD_ADC_SAMPLETIME_4;
	md_adc_normal_channel_config(ADC0, &conf);
	
	/* Configure normal channel1 */
	conf.ch   = MD_ADC_CHANNEL_3;
	conf.idx  = MD_ADC_NCH_IDX_3;
	conf.samp = MD_ADC_SAMPLETIME_4;
	md_adc_normal_channel_config(ADC0, &conf);

	/* Configure normal channel14 */
	conf.ch   = MD_ADC_CHANNEL_14;
	conf.idx  = MD_ADC_NCH_IDX_4;
	conf.samp = MD_ADC_SAMPLETIME_4;
	md_adc_normal_channel_config(ADC0, &conf);
	
	dma_init(adc_buf);
	/* Enable ADC0 */
	MD_ADC_ENABLE(ADC0);
	/* Start normal convert */
	md_adc_set_normal_channel_conv_start_nchtrg(ADC0);

	while (1) {		
		if (cflag == 1) {
			cflag = 0;

			/* Here: handle adc_buf data */
			
			/* DMA descriptor cycle type */
			md_dma0_ctrl_base[0].ctrl.cycle_ctrl = MD_DMA_CYCLE_CTRL_BASIC;
			/* DMA carry times */
			md_dma0_ctrl_base[0].ctrl.n_minus_1 = USR_ADC_CH_NR - 1;
			/* Enable DMA channel */
			md_dma_enable_channel(MD_DMA_CH_0);
			/* Trigger ADC */
			MD_ADC_NH_TRIG_BY_SOFT(ADC0);	
		}
	}
}
/**
  * @}
  */
/**
  * @}
  */
