/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 


#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

usart_handle_t h_usart;
volatile uint8_t frame_flag = 0;
volatile uint8_t tx_flag = 1;
uint8_t frame_buf[BUFFER_SIZE];
uint8_t frame_len = 0;
adc_handle_t h_adc;
timer_handle_t h_tim;
pis_handle_t h_pis;
volatile uint8_t adcflag = 0;
uint16_t adc_buf[14];
uint8_t __recv_cnt = 0;

/**
  * @brief  Initializate pin of usart module.
  * @retval None
  */
void usart_pin_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(USART0_TX_PORT, USART0_TX_PIN, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(USART0_RX_PORT, USART0_RX_PIN, &x);

	return;
}

/**
  * @brief  dma init.
  * @param  hperh: USART instance pointer.
  * @retval None.
  */
void dma_init(usart_handle_t *hperh)
{
	hperh->hdmarx.perh = DMA0;

	/* Configure callback function */
	hperh->hdmarx.cplt_cbk = NULL;
	hperh->hdmarx.cplt_arg = NULL;
	hperh->hdmarx.err_cbk  = NULL;
	hperh->hdmarx.err_arg  = NULL;

	/* Configure DMA Receive */
	ald_dma_config_struct(&hperh->hdmarx.config);
	hperh->hdmarx.config.src       = (void *)&hperh->perh->DATA;
	hperh->hdmarx.config.dst       = (void *)&frame_buf[1];
	hperh->hdmarx.config.size      = BUFFER_SIZE - 1;
	hperh->hdmarx.config.src_inc   = DMA_DATA_INC_NONE;
	hperh->hdmarx.config.dst_inc   = DMA_DATA_INC_BYTE;
	hperh->hdmarx.config.msel      = hperh->perh == USART0 ? DMA_MSEL_USART0 : DMA_MSEL_USART1;
	hperh->hdmarx.config.msigsel   = DMA_MSIGSEL_USART_RNR;
	hperh->hdmarx.config.high_prio = ENABLE;
	hperh->hdmarx.config.channel   = 0;

	ald_dma_config_basic(&hperh->hdmarx);
}

/**
  * @brief  usart receive a frame by dma.
  * @retval None.
  */
void usart_rec_frame_by_dma(void) 
{
	ald_usart_interrupt_config(&h_usart, USART_IT_RXNE, ENABLE);	
}

/**
  * @brief  Send a frame by DMA.
  * @param  buf: Pointer to the buffer.
  * @param  num: Length of the buffer.
  * @retval None.
  */
void usart_send_frame_by_dma(uint8_t *buf, uint16_t len)
{
	h_usart.hdmatx.perh = DMA0;

	/* Configure callback function */
	h_usart.hdmatx.cplt_cbk = NULL;
	h_usart.hdmatx.cplt_arg = NULL;
	h_usart.hdmatx.err_cbk  = NULL;
	h_usart.hdmatx.err_arg  = NULL;

	/* Configure USART DMA transmit */
	ald_dma_config_struct(&h_usart.hdmatx.config);
	h_usart.hdmatx.config.src     = (void *)buf;
	h_usart.hdmatx.config.dst     = (void *)&h_usart.perh->DATA;
	h_usart.hdmatx.config.size    = len;
	h_usart.hdmatx.config.src_inc = DMA_DATA_INC_BYTE;
	h_usart.hdmatx.config.dst_inc = DMA_DATA_INC_NONE;
	h_usart.hdmatx.config.msel    = DMA_MSEL_USART0;
	h_usart.hdmatx.config.msigsel = DMA_MSIGSEL_USART_TXEMPTY;
	h_usart.hdmatx.config.channel = 1;
	ald_dma_config_basic(&h_usart.hdmatx);

	ald_usart_clear_flag_status(&h_usart, USART_FLAG_TC);
	SET_BIT(h_usart.perh->CON2, USART_CON2_TXDMAEN_MSK);
	tx_flag = 0;

	return;
}


/**
  * @brief  Initializate ADC pin.
  * @retval None.
  */
void adc_pin_init(void)
{
	gpio_init_t x;

	memset(&x, 0, sizeof(gpio_init_t));

	/* Initialize ADC pin */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;

	ald_gpio_init(GPIO_CH0_PORT, ADC_CH0_PIN, &x);
	ald_gpio_init(GPIO_CH1_PORT, ADC_CH1_PIN, &x);
	ald_gpio_init(GPIO_CH2_PORT, ADC_CH2_PIN, &x);
	ald_gpio_init(GPIO_CH3_PORT, ADC_CH3_PIN, &x);
	ald_gpio_init(GPIO_CH4_PORT, ADC_CH4_PIN, &x);
	ald_gpio_init(GPIO_CH5_PORT, ADC_CH5_PIN, &x);
	ald_gpio_init(GPIO_CH6_PORT, ADC_CH6_PIN, &x);
	ald_gpio_init(GPIO_CH7_PORT, ADC_CH7_PIN, &x);
	ald_gpio_init(GPIO_CH8_PORT, ADC_CH8_PIN, &x);
	ald_gpio_init(GPIO_CH9_PORT, ADC_CH9_PIN, &x);
	ald_gpio_init(GPIO_CH10_PORT, ADC_CH10_PIN, &x);
	ald_gpio_init(GPIO_CH11_PORT, ADC_CH11_PIN, &x);
	ald_gpio_init(GPIO_CH12_PORT, ADC_CH12_PIN, &x);
	ald_gpio_init(GPIO_CH13_PORT, ADC_CH13_PIN, &x);
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
	adc_nch_conf_t config;

	/* clear structures */
	memset(&h_adc, 0x0, sizeof(adc_handle_t));
	memset(&config, 0x0, sizeof(adc_nch_conf_t));

	/* Initialize adc */
	h_adc.perh            = ADC0;
	h_adc.init.align      = ADC_DATAALIGN_RIGHT;
	h_adc.init.scan       = ENABLE;
	h_adc.init.cont       = ENABLE;
	h_adc.init.nch_nr     = ADC_NCH_NR_14;
	h_adc.init.disc       = ADC_ALL_DISABLE;
	h_adc.init.disc_nr    = ADC_DISC_NR_1;
	h_adc.init.data_bit   = ADC_CONV_BIT_12;
	h_adc.init.div        = ADC_CKDIV_128;
	h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ONE;
	h_adc.init.n_ref      = ADC_NEG_REF_VSS;
	h_adc.init.p_ref      = ADC_POS_REF_VDD;
	h_adc.normal_cplt_cbk = NULL;
	h_adc.insert_cplt_cbk = NULL;
	h_adc.wdg_cbk         = NULL;
	h_adc.error_cbk       = NULL;
	h_adc.ovr_cbk         = NULL;
	ald_adc_init(&h_adc);
	ADC_SPEED_HIGH_ENABLE(&h_adc);

	/* Initialize normal convert channel */
	config.ch   = ADC_CHANNEL_0;
	config.idx  = ADC_NCH_IDX_1;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_1;
	config.idx  = ADC_NCH_IDX_2;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_2;
	config.idx  = ADC_NCH_IDX_3;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_3;
	config.idx  = ADC_NCH_IDX_4;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_4;
	config.idx  = ADC_NCH_IDX_5;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_5;
	config.idx  = ADC_NCH_IDX_6;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_6;
	config.idx  = ADC_NCH_IDX_7;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_8;
	config.idx  = ADC_NCH_IDX_9;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_9;
	config.idx  = ADC_NCH_IDX_10;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_10;
	config.idx  = ADC_NCH_IDX_11;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_11;
	config.idx  = ADC_NCH_IDX_12;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_12;
	config.idx  = ADC_NCH_IDX_13;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	config.ch   = ADC_CHANNEL_13;
	config.idx  = ADC_NCH_IDX_14;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	return;
}

/**
  * @brief  Trigger ADC
  * @param  dstbuf: Pointer to ADC result.
  * @param  num: Length of ADC result.
  * @retval None.
  */
void adc_dma_trigger(uint16_t* dstbuf, uint16_t num)
{
	h_adc.hdma.perh = DMA0;
	h_adc.hdma.cplt_cbk = NULL;
	h_adc.hdma.cplt_arg = NULL;
	h_adc.hdma.err_cbk  = NULL;
	h_adc.hdma.err_arg  = NULL;

	ald_dma_config_struct(&h_adc.hdma.config);
	h_adc.hdma.config.src        = (void *)&h_adc.perh->NCHDR;
	h_adc.hdma.config.dst        = (void *)dstbuf;
	h_adc.hdma.config.size       = num;
	h_adc.hdma.config.data_width = DMA_DATA_SIZE_HALFWORD;
	h_adc.hdma.config.src_inc    = DMA_DATA_INC_NONE;
	h_adc.hdma.config.dst_inc    = DMA_DATA_INC_HALFWORD;
	h_adc.hdma.config.msel       = DMA_MSEL_ADC0;
	h_adc.hdma.config.msigsel    = DMA_MSIGSEL_ADC;
	h_adc.hdma.config.channel    = 2;
	ald_dma_config_basic(&h_adc.hdma);

	 ald_adc_normal_start(&h_adc);/* Enable pis trigger */
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1,48000000);
	ald_mcu_irq_config(USART0_IRQn, 0, ENABLE);

	/* clear usart_handle_t structure */
	memset(&h_usart, 0x0, sizeof(h_usart));
	/* Initialize pin */
	usart_pin_init();

	adc_pin_init();
	/* Initialize ADC */
	adc_init();
	/* Initialize USART */
	h_usart.perh             = USART0;
	h_usart.init.baud        = 116200;
	h_usart.init.word_length = USART_WORD_LENGTH_8B;
	h_usart.init.stop_bits   = USART_STOP_BITS_1;
	h_usart.init.parity      = USART_PARITY_NONE;
	h_usart.init.mode        = USART_MODE_TX_RX;
	h_usart.init.fctl        = USART_HW_FLOW_CTL_NONE;
	h_usart.tx_cplt_cbk      = NULL;
	h_usart.rx_cplt_cbk      = NULL;
	h_usart.tx_rx_cplt_cbk   = NULL;
	h_usart.error_cbk        = NULL;
	ald_usart_init(&h_usart);
	dma_init(&h_usart);
	usart_rec_frame_by_dma();
	adc_dma_trigger(adc_buf, 14);

	while (1) {
		if (frame_flag == 1) {
			frame_flag = 0;
			__recv_cnt++;
			
			if (tx_flag) {
				usart_send_frame_by_dma(frame_buf, (frame_len << 1));
			}

			usart_rec_frame_by_dma();
		}
		
		if (adcflag) {
			adcflag = 0;

			/* Do something */

			adc_dma_trigger(adc_buf, 14);
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
