/**
  *********************************************************************************
  *
  * @file    irq.c
  * @brief   Interrupt handler
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include "main.h"
#include "utils.h"
#include "ald_cmu.h"
#ifdef ALD_DMA
#include "ald_dma.h"
#endif

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
  
extern dma_descriptor_t dma0_ctrl_base[28] __attribute__ ((aligned(512)));
extern uint8_t frame_buf[BUFFER_SIZE];
extern uint8_t frame_flag ;
extern uint8_t frame_len;
extern usart_handle_t h_usart;
extern uint16_t adcbuf;
extern uint8_t adcflag;
extern timer_handle_t h_tim;
extern uint8_t tx_flag;

/**
  * @brief  NMI IRQ handler
  * @retval None
  */
void NMI_Handler(void)
{
	/* Added Emergency operation */
	return;
}

/**
  * @brief  Hardfault IRQ handler
  * @retval None
  */
void HardFault_Handler(void)
{
	/* Added debug information */
	while (1)
		;
}

/**
  * @brief  Supervisor Call IRQ handler
  * @retval None
  */
void SVC_Handler(void)
{
	/* Added system callback */
	return;
}

/**
  * @brief  Debug Monitor IRQ handler
  * @retval None
  */
void DebugMon_Handler(void)
{
	/* Added debug operation */
	return;
}

/**
  * @brief  PendSV IRQ handler
  * @retval None
  */
void PendSV_Handler(void)
{
	/* Added thread switching operation */
	return;
}

/**
  * @brief  SysTick IRQ handler
  * @retval None
  */
void SysTick_Handler(void)
{
	ald_inc_tick();
	return;
}

#ifdef ALD_DMA
/**
  * @brief  DMA IRQ#9 handler
  * @retval None
  */
void DMA_Handler(void)
{
	/* Hanlde ADC-DMA interrupt */
	if (DMA0->IFLAG & (1 << 2)) {
		ADC0->CON1   &= ~0x01;		/* Stop ADC */
		DMA0->CHENCLR = (0x1 << 2);	/* Disable DMA channel 2*/
		DMA0->ICFR    = (0x1 << 2);	/* Clear DMA flag */
		adcflag = 1;			/* SET Flag */
	}

	/* Hanlde ADC-UART-TX interrupt */
	if (DMA0->IFLAG & (1 << 1)) {
		DMA0->CHENCLR = (0x1 << 1);	/* Disable DMA channel 1*/
		DMA0->ICFR    = (0x1 << 1);	/* Clear DMA flag */
		CLEAR_BIT(USART0->CON2, USART_CON2_TXDMAEN_MSK);	/* Disable DMA request */
		SET_BIT(USART0->CON0, (1U << 6));	/* Enable TC interrupt */
		tx_flag = 1;			/* SET Flag */
	}
	
	/* Handle USART0 DMA channel0 interrupt */
	if (DMA0->IFLAG & (1 << 0)) {
		CLEAR_BIT(USART0->CON0, (0x1 << 4));  
		/* Calculate frame data length */
		frame_len = BUFFER_SIZE -1 - dma0_ctrl_base[0].ctrl.n_minus_1;
		/* Receive frame flag, frame receive completely */
		frame_flag = 1;
		/* Disable DMA*/
		DMA0->CHENCLR = 1;	/* Disable DMA channel */
		DMA0->ICFR    = 1;	/* Clear DMA flag */
	}
}
#endif

/**
  * @brief  USART IRQ handler
  * @retval None
  */

void USART0_Handler(void)
{	
	/* RXNE interrupt */
	if (READ_BIT(USART0->STAT, USART_FLAG_RXNE)) { 
		/* Disable RXNE interrupt */
		CLEAR_BIT(USART0->CON0, (0x1 << 5));  	 
		/* Get first rec data */
		frame_buf[0] = USART0->DATA;
		/* Enable DMA */
		dma0_ctrl_base[0].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_BASIC;
		dma0_ctrl_base[0].ctrl.n_minus_1 = BUFFER_SIZE -2;	
		WRITE_REG(DMA0->CHENSET, 1);
		/* Enable USART DMA */
		SET_BIT(USART0->CON2, USART_CON2_RXDMAEN_MSK);
		/* Enable USART IDLE interrupt */
		SET_BIT(USART0->CON0, (0x1 << 4));
	}
	
	/* IDLE interrupt */
	if ((READ_BIT(USART0->STAT, USART_FLAG_IDLE)) && (READ_BIT(USART0->CON0, (0x1 << 4)))) {
		/* Disable IDLE interrupt */
		CLEAR_BIT(USART0->CON0, (0x1 << 4));
		CLEAR_BIT(USART0->CON2, USART_CON2_RXDMAEN_MSK);
		/* Calculate frame data length */
		frame_len = BUFFER_SIZE -1 - dma0_ctrl_base[0].ctrl.n_minus_1;
		/* Receive frame flag, frame receive completely */
		frame_flag = 1;
		/* Disable DMA*/
		dma0_ctrl_base[0].ctrl.n_minus_1 = 0;
		dma0_ctrl_base[0].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_NONE;
		DMA0->CHENCLR = 1;	/* Disable DMA channel */
		DMA0->ICFR    = 1;	/* Clear DMA flag */
	}

	/* USART0 transmit completely interrupt */
	if (READ_BIT(USART0->STAT, (1U << 6)) && READ_BIT(USART0->CON0, ((1U << 6)))) {
		CLEAR_BIT(USART0->CON0, (1U << 6));
		CLEAR_BIT(h_usart.state, USART_STATE_TX_MASK);
		if (h_usart.tx_cplt_cbk != NULL)
			h_usart.tx_cplt_cbk(&h_usart);
	}
	
	return;
}
/**
  * @}
  */

/**
  * @}
  */
