/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
  
timer_handle_t h_tim;
timer_clock_config_t tim_clock;
timer_ic_init_t tim_icinit;

/* Captured Values */
uint32_t uwIC2Value1   = 0;
uint32_t uwIC2Value2   = 0;
uint32_t uwDiffCapture = 0;

/* Capture index */
uint16_t uhCaptureIndex = 0;

/* Frequency Value */
uint32_t uwFrequency = 0;

/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void tim_pin_init(void)
{
	gpio_init_t x;
	/* Initialize tim0 ch1 pin */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_2;
	ald_gpio_init(TIMER_INPUTCAPTURE_PORT, TIMER_INPUTCAPTURE_PIN, &x);
}

void ald_timer_ic_capturecallback(struct timer_handle_s *htim);

/**
  * @brief  Main function
  * @retval Status
  */
int main()
{
	/* Initialize  */
	ald_cmu_init();
        ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1,48000000);
    
      /* Initialize pin and set the default state*/
	tim_pin_init();
	
	/* Initialize AD16C4T0 */
	h_tim.perh           = AD16C4T0;
	h_tim.init.prescaler = 0;
	h_tim.init.mode      = TIMER_CNT_MODE_UP;
	h_tim.init.period    = 0xFFFF;
	h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim.init.re_cnt    = 0;
        h_tim.capture_cbk    = ald_timer_ic_capturecallback;
	ald_timer_ic_init(&h_tim);

	/* Initialize clock source */
	tim_clock.source     = TIMER_SRC_INTER;
	ald_timer_config_clock_source(&h_tim, &tim_clock);

	/* Initialize input capture channel */
	tim_icinit.polarity = TIMER_IC_POLARITY_RISE;
	tim_icinit.sel      = TIMER_IC_SEL_DIRECT;
	tim_icinit.psc      = TIMER_IC_PSC_DIV1;
	tim_icinit.filter   = 0;
	ald_timer_ic_config_channel(&h_tim,&tim_icinit,TIMER_CHANNEL_1);

	ald_mcu_irq_config(AD16C4T0_CC_IRQn, 1, ENABLE);	
    
	/* Enable CC interrupt */
	ald_timer_interrupt_config(&h_tim, TIMER_IT_CC1, ENABLE);

        /* Start input capture */
	ald_timer_ic_start(&h_tim, TIMER_CHANNEL_1);

        ald_timer_ic_start_by_it(&h_tim, TIMER_CHANNEL_1);

	while(1) {}
}

/**
  * @brief  Input capture callback  function
  * @retval None
  */
void ald_timer_ic_capturecallback(struct timer_handle_s *htim)
{
	if (htim->ch == TIMER_ACTIVE_CHANNEL_1) {
		if(uhCaptureIndex == 0) {
			/* Get the 1st Input Capture value */
			uwIC2Value1    = ald_timer_read_capture_value(htim, TIMER_CHANNEL_1);
			uhCaptureIndex = 1;
		}
		else if(uhCaptureIndex == 1) {
			/* Get the 1st Input Capture value */
			uwIC2Value2 = ald_timer_read_capture_value(htim, TIMER_CHANNEL_1); 

			/* Capture computation */
			if (uwIC2Value2 > uwIC2Value1) {
				uwDiffCapture = (uwIC2Value2 - uwIC2Value1); 
			}
			else if (uwIC2Value2 < uwIC2Value1) {
				/* 0xFFFF is max TIMER0_CCRx value */
				uwDiffCapture = ((0xFFFF - uwIC2Value1) + uwIC2Value2) + 1;
			}
			else {
				/* If capture values are equal, we have reached the limit of frequency
				 * measures
				 */
				while(1);
			}
			/* Frequency computation: for this example TIMERx (TIM0) is clocked by
			 * SystemCoreClock
			 */      
			uwFrequency = ald_cmu_get_pclk1_clock() / uwDiffCapture;
			uhCaptureIndex = 0;
		}
	}
}


/**
  * @}
  */
/**
  * @}
  */
