/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 
#include "main.h"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

rtc_time_t time;
rtc_date_t date;
uart_handle_t h_uart;
char uart_buf[64];

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = UART_FUNC_TX;
	ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = UART_FUNC_RX;
	ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

	/* Initialize uart */
	h_uart.perh             = UART0;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	return;
}


/**
  * @brief  This is main function
  * @retval Status
  */
int main()
{
	rtc_init_t init;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	/* Set RTC clock */
	ald_rtc_source_select(RTC_SOURCE_LOSC);
	/* Initialize stdio */
	uart_stdio_init();

	/* Initialize RTC */
	init.asynch_pre_div  = 0;
	init.synch_pre_div   = 32767;
	init.hour_format     = RTC_HOUR_FORMAT_24;
	init.output          = RTC_OUTPUT_DISABLE;
	init.output_polarity = RTC_OUTPUT_POLARITY_HIGH;
	ald_rtc_init(&init);

	printf_e("System start...\r\n");

	/* Set current time and date */
	time.hour    = 23;
	time.minute  = 59;
	time.second  = 55;
	date.day     = 2;
	date.month   = 7;
	date.year    = 18;
	date.week    = 1;
	ald_rtc_set_time(&time, RTC_FORMAT_DEC);
	ald_rtc_set_date(&date, RTC_FORMAT_DEC);
	printf_e("year month day hour minute second\r\n");
	printf_e("----------------------------------------\r\n");
	printf_e("20%d-%d-%d %d:%d:%d\r\n", date.year, date.month,
	                          date.day, time.hour, time.minute, time.second);

	while (1) {
		ald_delay_ms(1000);
		ald_rtc_get_time(&time, RTC_FORMAT_DEC);
		ald_rtc_get_date(&date, RTC_FORMAT_DEC);
		printf_e("20%d-%d-%d %d:%d:%d\r\n", date.year, date.month,
	                          date.day, time.hour, time.minute, time.second);
	}
}

/**
  * @}
  */
/**
  * @}
  */
