/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include "main.h"
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

pis_handle_t h_pis;
adc_handle_t h_adc;
timer_handle_t h_tim;

timer_clock_config_t tim_clock;
timer_oc_init_t tim_ocinit;
timer_ic_init_t tim_icinit;
timer_master_config_t tim_master_config;
uint8_t buffer[32]   = {0};
uint8_t read_buf[32] = {0};

uint32_t nm_value = 0;
adc_nch_conf_t nm_config;

/**
  * @brief  Initializate pin of pis module.
  * @retval None
  */
void pis_pin_init(void)
{
	gpio_init_t x;
	
	/* Initialize timer0 ch1 pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_2;
	ald_gpio_init(GPIO_TIMER0_CH_PORT, TIMER0_CH1_PIN, &x);
	
	/* Initialize pis ch0 pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_6;
	ald_gpio_init(GPIO_PIS_CH_PORT, PIS_CH0_PIN, &x);
	
	/* Initialize adc ch15 pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIO_ADC_CH_PORT, ADC_CH15_PIN, &x);
	
	/* Initialize OUTPUT TEST pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(GPIO_OUTTEST_PORT, GPIO_OUTTEST_PIN, &x);
	ald_gpio_write_pin(GPIO_OUTTEST_PORT, GPIO_OUTTEST_PIN, 0);
}

/**
  * @brief  Initializate timer0 module.
  * @retval None
  */
void pis_tim_init(void)
{
	/* clear timer_handle_t structure */
	memset(&h_tim,0,sizeof(timer_handle_t));
	/* clear timer_clock_config_t structure */
	memset(&tim_clock,0,sizeof(timer_clock_config_t));
	/* clear timer_oc_init_t structure */
	memset(&tim_ocinit,0,sizeof(timer_oc_init_t));
	
	/* Initialize TIM0 */
	h_tim.perh           = AD16C4T0;
	h_tim.init.prescaler = 0xfe;
	h_tim.init.mode      = TIMER_CNT_MODE_UP;
	h_tim.init.period    = 0x300;
	h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim.init.re_cnt    = 0;
	ald_timer_base_init(&h_tim);
	
	tim_icinit.filter   = 3;
	tim_icinit.polarity = TIMER_IC_POLARITY_RISE;
	tim_icinit.psc      = TIMER_IC_PSC_DIV1;
	tim_icinit.sel      = TIMER_IC_SEL_DIRECT;
	ald_timer_ic_config_channel(&h_tim, &tim_icinit, TIMER_CHANNEL_1);	
}

/**
  * @brief  Start tim0 module.
  * @retval None
  */
void pis_tim_start(void)
{
	ald_timer_ic_start(&h_tim, TIMER_CHANNEL_1);
}

/**
  * @brief  Stop tim0 module.
  * @retval None
  */
void pis_tim_stop(void)
{
	ald_timer_ic_stop(&h_tim, TIMER_CHANNEL_1);
}

/* adc */
/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void normal_convert_complete(adc_handle_t *arg)
{
	nm_value = ald_adc_normal_get_value(&h_adc);
	ald_adc_normal_stop_by_it(&h_adc);
	return;
}

/**
  * @brief  Adc insert convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void insert_convert_complete(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc level out of window complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void out_window_complete(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc error complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void error_complete(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc over complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void ovr_complete(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Initializate adc module.
  * @retval None
  */
void pis_adc_init(void)	
{
	/* clear adc_handle_t structure */
	memset(&h_adc, 0x0, sizeof(adc_handle_t));
	/* clear adc_nch_conf_t structure */
	memset(&nm_config, 0x0, sizeof(adc_nch_conf_t));
	/* Enable adc interrupt */
	ald_mcu_irq_config(ADC_ACMP_IRQn, 1, ENABLE);

	/* Initialize adc */
	h_adc.perh            = ADC0;
	h_adc.init.align      = ADC_DATAALIGN_RIGHT;
	h_adc.init.scan       = DISABLE;
	h_adc.init.cont       = DISABLE;
	h_adc.init.nch_nr     = ADC_NCH_NR_1;
	h_adc.init.disc       = ADC_ALL_DISABLE;
	h_adc.init.disc_nr    = ADC_DISC_NR_1;
	h_adc.init.data_bit   = ADC_CONV_BIT_12;
	h_adc.init.div        = ADC_CKDIV_128;
	h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ALL;
	h_adc.init.n_ref      = ADC_NEG_REF_VSS;
	h_adc.init.p_ref      = ADC_POS_REF_VDD;
	h_adc.normal_cplt_cbk = normal_convert_complete;
	h_adc.insert_cplt_cbk = insert_convert_complete;
	h_adc.wdg_cbk         = out_window_complete;
	h_adc.error_cbk       = error_complete;
	h_adc.ovr_cbk         = ovr_complete;
	ald_adc_init(&h_adc); 	
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	volatile uint32_t cmu = 0;
	/* Initialize ALD */
	ald_cmu_init();
	
	/* Clear pis_handle_t struct */
	memset(&h_pis,0,sizeof(pis_handle_t));
	
	/* Initialize pin */
	pis_pin_init();
	/* Initialize timer0 */
	pis_tim_init();
	
	pis_adc_init();
	
	/* Initialize normal convert channel */
	nm_config.ch   = ADC_CHANNEL_15;
	nm_config.idx  = ADC_NCH_IDX_1;
	nm_config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &nm_config);
	
	ald_adc_clear_flag_status(&h_adc, ADC_FLAG_NCH);
	ald_adc_interrupt_config(&h_adc, ADC_IT_NCH, ENABLE);
	
	ADC_ENABLE(&h_adc);
	
	/* Initialize PIS */
	h_pis.perh               = PIS;
	h_pis.init.producer_src  = PIS_TIMER0_INPUT;
	h_pis.init.producer_clk  = PIS_CLK_PCLK1;
	h_pis.init.producer_edge = PIS_EDGE_UP;
	h_pis.init.consumer_trig = PIS_CH6_ADC0_NORMAL; 
	h_pis.init.consumer_clk  = PIS_CLK_PCLK1;
	ald_pis_create(&h_pis);
	
	pis_tim_start();
	
	ald_pis_output_start(&h_pis, PIS_OUT_CH_0);

	ald_delay_ms(1);
	ald_gpio_write_pin(GPIO_OUTTEST_PORT, GPIO_OUTTEST_PIN, 1);
	while (1) {
		ald_delay_ms(1000);
	}
}

/**
  * @}
  */

/**
  * @}
  */
