/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint8_t src0_buf[16];
uint8_t dst0_buf[16];
uint8_t src1_buf[32];
uint8_t dst1_buf[32];
uint8_t src2_buf[64];
uint8_t dst2_buf[64];
uint8_t src3_buf[10];
uint8_t dst3_buf[10];

dma_descriptor_t __alt_desc[4];
uart_handle_t h_uart;

/**
  * @brief  DMA send complete.
  * @param  arg: Pointer.
  * @retval None.
  */
void dma_complete(void *arg)
{
	ald_uart_dma_req_config(&h_uart, DISABLE);
	return;
}

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void uart_pin_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(UART1_TX_PORT, UART1_TX_PIN, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(UART1_RX_PORT, UART1_RX_PIN, &x);

	return;
}

/**
  * @brief  Config alternate descriptor for memory scatter-gather.
  * @retval None.
  */
void dma_config_alt_desc()
{
	dma_config_t config;

	/* Configure the first descriptor */
	memset(&config, 0x0, sizeof(dma_config_t));
	config.src        = src0_buf;
	config.dst        = (void *)&UART1->TBR;
	config.size       = 16;
	config.R_power    = DMA_R_POWER_1;
	config.data_width = DMA_DATA_SIZE_BYTE;
	config.src_inc    = DMA_DATA_INC_BYTE;;
	config.dst_inc    = DMA_DATA_INC_NONE;
	ald_dma_config_sg_alt_desc(&__alt_desc[0], &config, 0);

	/* Configure the second descriptor */
	config.src        = src1_buf;
	config.dst        = (void *)&UART1->TBR;
	config.size       = 32;
	config.R_power    = DMA_R_POWER_1;
	config.data_width = DMA_DATA_SIZE_BYTE;
	config.src_inc    = DMA_DATA_INC_BYTE;;
	config.dst_inc    = DMA_DATA_INC_NONE;
	ald_dma_config_sg_alt_desc(&__alt_desc[1], &config, 0);

	/* Configure the third descriptor */
	config.src        = src2_buf;
	config.dst        = (void *)&UART1->TBR;
	config.size       = 64;
	config.R_power    = DMA_R_POWER_1;
	config.data_width = DMA_DATA_SIZE_BYTE;
	config.src_inc    = DMA_DATA_INC_BYTE;;
	config.dst_inc    = DMA_DATA_INC_NONE;
	ald_dma_config_sg_alt_desc(&__alt_desc[2], &config, 0);

	/* Configure the fourth descriptor */
	config.src        = src3_buf;
	config.dst        = (void *)&UART1->TBR;
	config.size       = 10;
	config.R_power    = DMA_R_POWER_1;
	config.data_width = DMA_DATA_SIZE_BYTE;
	config.src_inc    = DMA_DATA_INC_BYTE;;
	config.dst_inc    = DMA_DATA_INC_NONE;
	ald_dma_config_sg_alt_desc(&__alt_desc[3], &config, 0);

	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();

	/* Initialize data */
	src0_buf[0] = 0x0;
	src0_buf[1] = 0x1;
	src0_buf[2] = 0x2;
	src0_buf[3] = 0x3;
	src0_buf[4] = 0x4;
	src0_buf[5] = 0x5;
	memset(&src0_buf[6], 0x88, 10);
	memset(src1_buf, 0x33, 32);
	memset(src2_buf, 0x44, 64);
	memset(src3_buf, 0x55, 10);

	/* Clear uart_handle_t structure */
	memset(&h_uart, 0x0, sizeof(h_uart));
	/* Initialize UART pin */
	uart_pin_init();

	/* Initialize uart */
	h_uart.perh             = UART1;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	/* Configure four alternate descriptors */
	dma_config_alt_desc();
	/* Start DMA in peripheral scatter-gather mode */
	ald_dma_config_sg_per(DMA0, __alt_desc, 4, 1, DMA_MSEL_UART1, DMA_MSIGSEL_UART_TXEMPTY, 0, dma_complete);
	/* Enable UART module DMA request */
	ald_uart_dma_req_config(&h_uart, ENABLE);

	while (1) {
		ald_delay_ms(1000);
	}
}

/**
  * @}
  */
/**
  * @}
  */
