/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include "main.h"
#include <string.h>

  
/** @addtogroup Projects_Examples_ALD
  * @{
  */


/** @addtogroup Public_Function
  * @{
  */

crc_handle_t h_crc;
uint32_t result = 0;

/**
  * @brief  calculate completed callback.
  * @param  arg: Pointer to crc_handle_t structure.
  * @retval None.
  */
void calculate_complete(crc_handle_t *arg)
{
	return;
}

/**
  * @brief  calculate error callback.
  * @param  arg: Pointer to crc_handle_t structure.
  * @retval None.
  */
void error_complete(crc_handle_t *arg)
{
	return;
}

/**
  * @brief  Test case 0:
  * CRC-8       refin false
  *		refout false
  *		init 0x00
  * input: 0x01 0x02
  * expect output: 0x1b
  * @retval None.
  */
void test_crc8_1()
{
	uint8_t crc_buf[2] = {1,2};
	/* Clear crc_handle_t structure */
	memset(&h_crc, 0x0, sizeof(crc_handle_t));
	
	/* Initialize CRC */
	h_crc.perh = CRC;
	h_crc.init.mode = CRC_MODE_8;
	h_crc.init.seed = 0x0;
	h_crc.init.data_rev = DISABLE;
	h_crc.init.data_inv = DISABLE;
	h_crc.init.chs_rev = DISABLE;
	h_crc.init.chs_inv = DISABLE;
	
	h_crc.cal_cplt_cbk = calculate_complete;
	h_crc.err_cplt_cbk = error_complete;
	ald_crc_init(&h_crc);
	
	/* start calculate and result is 0x1b*/
	result = ald_crc_calculate(&h_crc, crc_buf, sizeof(crc_buf));
}

/**
  * @brief  Test case 1:
  * CRC-16/MODBUS refin  true
  *		  refout true
  *		  init   0x0000
  * input: 0x0001 0x0002
  * expect output: 0x25d0
  * @retval None.
  */
void test_crc16_1()
{
	uint8_t crc_buf[4] = {0,1,0,2};
	/* Clear crc_handle_t structure */
	memset(&h_crc, 0x0, sizeof(crc_handle_t));

	/* Initialize CRC */
	h_crc.perh = CRC;
	h_crc.init.mode = CRC_MODE_16;
	h_crc.init.seed = 0xffff;
	h_crc.init.data_rev = ENABLE;
	h_crc.init.data_inv = DISABLE;
	h_crc.init.chs_rev = ENABLE;
	h_crc.init.chs_inv = DISABLE;
	
	h_crc.cal_cplt_cbk = calculate_complete;
	h_crc.err_cplt_cbk = error_complete;
	ald_crc_init(&h_crc);
	
	/* start calculate and result is 0x25d0*/
	result = ald_crc_calculate(&h_crc, crc_buf, sizeof(crc_buf));
}

/**
  * @brief  Test CRC-16/CCITT-FALSE:
  * CRC-16/CCITT-FALSE refin  false
  *		      refout false
  *		      init   0x0000
  * input: 0x0001 0x0002
  * output: 0x93b2
  * @retval None.
  */
void test_crc_ccitt()
{
	uint8_t crc_buf[4] = {0,1,0,2};
	/* Clear crc_handle_t structure */
	memset(&h_crc, 0x0, sizeof(crc_handle_t));
	
	/* Initialize CRC */
	h_crc.perh = CRC;
	h_crc.init.mode = CRC_MODE_CCITT;
	h_crc.init.seed = 0xffff;
	h_crc.init.data_rev = DISABLE;
	h_crc.init.data_inv = DISABLE;
	h_crc.init.chs_rev = DISABLE;
	h_crc.init.chs_inv = DISABLE;
	
	h_crc.cal_cplt_cbk = calculate_complete;
	h_crc.err_cplt_cbk = error_complete;
	ald_crc_init(&h_crc);
	
	/* start calculate and result is 0x93b2*/
	result = ald_crc_calculate(&h_crc, crc_buf, sizeof(crc_buf));
}

/**
  * @brief  Test CRC-32:
  * crc32	refin false
  *		refout false 
  *		init 0xffffffff
  * input: 0x01020304 0x05060708
  * expect output: 0x140B8DD8
  * @retval None.
  */
void test_crc32_1()
{
	uint8_t crc_buf[8] = {1,2,3,4,5,6,7,8};
	/* Clear crc_handle_t structure */
	memset(&h_crc, 0x0, sizeof(crc_handle_t));	
	
	/* Initialize CRC */
	h_crc.perh = CRC;
	h_crc.init.mode = CRC_MODE_32;
	h_crc.init.seed = 0xffffffff;
	h_crc.init.data_rev = DISABLE;
	h_crc.init.data_inv = DISABLE;
	h_crc.init.chs_rev = DISABLE;
	h_crc.init.chs_inv = DISABLE;
	
	h_crc.cal_cplt_cbk = calculate_complete;
	h_crc.err_cplt_cbk = error_complete;
	ald_crc_init(&h_crc);
	
	/* start calculate and result is 0x140B8DD8*/
	result = ald_crc_calculate(&h_crc, crc_buf, sizeof(crc_buf));
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	
	/* Initialize ALD */
	ald_cmu_init();
	test_crc8_1();
	test_crc16_1();
	test_crc_ccitt();
	test_crc32_1();

	while (1) {
		ald_delay_ms(1000);
	}
}


/**
  * @} Public_Function_CRC
  */

/**
  * @} Projects_Examples_ALD
  */
