/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

adc_handle_t h_adc;
uart_handle_t h_uart;
char uart_buf[64];
extern dma_descriptor_t dma0_ctrl_base[28];

timer_handle_t h_tim;
pis_handle_t h_pis;
uint16_t adcbuf[BUFFER_SIZE];
volatile uint8_t adcflag = 0;
uint16_t buffer[4][BUFFER_SIZE/4] = {0};

/**
  * @brief  Initializate pin of adc module.
  * @retval None.
  */
void adc_pin_init(void)
{
	gpio_init_t x;

	memset(&x, 0, sizeof(gpio_init_t));

	/* Initialize ADC pin */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;

	ald_gpio_init(GPIO_CH0_PORT, ADC_CH0_PIN, &x);
	ald_gpio_init(GPIO_CH2_PORT, ADC_CH2_PIN, &x);
	ald_gpio_init(GPIO_CH3_PORT, ADC_CH3_PIN, &x);
	ald_gpio_init(GPIO_CH9_PORT, ADC_CH9_PIN, &x);

	return;
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
	adc_nch_conf_t config;

	/* clear structures */
	memset(&h_adc, 0x0, sizeof(adc_handle_t));
	memset(&config, 0x0, sizeof(adc_nch_conf_t));

	/* Initialize adc */
	h_adc.perh            = ADC0;
	h_adc.init.align      = ADC_DATAALIGN_RIGHT;
	h_adc.init.scan       = ENABLE;
	h_adc.init.cont       = ENABLE;
	h_adc.init.nch_nr     = ADC_NCH_NR_4;
	h_adc.init.disc       = ADC_ALL_DISABLE;
	h_adc.init.disc_nr    = ADC_DISC_NR_1;
	h_adc.init.data_bit   = ADC_CONV_BIT_12;
	h_adc.init.div        = ADC_CKDIV_2;
	h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ONE;
	h_adc.init.n_ref      = ADC_NEG_REF_VSS;
	h_adc.init.p_ref      = ADC_POS_REF_VDD;
	h_adc.normal_cplt_cbk = NULL;
	h_adc.insert_cplt_cbk = NULL;
	h_adc.wdg_cbk         = NULL;
	h_adc.error_cbk       = NULL;
	h_adc.ovr_cbk         = NULL;
	ald_adc_init(&h_adc);
	ADC_SPEED_HIGH_ENABLE(&h_adc);

	/* Initialize normal convert channel */
	config.ch   = ADC_CHANNEL_0;
	config.idx  = ADC_NCH_IDX_1;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_2;
	config.idx  = ADC_NCH_IDX_2;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_3;
	config.idx  = ADC_NCH_IDX_3;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_9;
	config.idx  = ADC_NCH_IDX_4;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);
	
	return;
}

/**
  * @brief  Test io Initialization.
  * @retval None
  */
void toggle_io_init(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(GPIOC, GPIO_PIN_11, &x);	/* DMA finished interrupt toggle pin */	
	ald_gpio_init(GPIOA, GPIO_PIN_8, &x);   /* Timer 10us interrupt toggle pin */
}

/**
  * @brief  Pis module start.
  * @retval None
  */
void pis_tim_start(void)
{
	ald_timer_base_start(&h_tim);
}

/**
  * @brief  Pis module stop.
  * @retval None
  */
void pis_tim_stop(void)
{
	ald_timer_base_stop(&h_tim);
}

/**
  * @brief  Initializate pis module.
  * @retval None
  */
void pis_tim_init(void)
{
	/* clear timer_handle_t structure */
	memset(&h_tim,0,sizeof(timer_handle_t));

	/* Initialize BS16T0 */
	h_tim.perh           = BS16T0;
	h_tim.init.prescaler = 47;
	h_tim.init.mode      = TIMER_CNT_MODE_UP;
	h_tim.init.period    = 9;
	h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim.init.re_cnt    = 0;
	h_tim.period_elapse_cbk = NULL;
	ald_timer_base_init(&h_tim);

	/* Initialize PIS */
	h_pis.perh               = PIS;
	h_pis.init.producer_src  = PIS_TIMER1_UPDATA;
	h_pis.init.producer_clk  = PIS_CLK_PCLK1;
	h_pis.init.producer_edge = PIS_EDGE_UP;
	h_pis.init.consumer_trig = PIS_CH6_ADC0_NORMAL; 
	h_pis.init.consumer_clk  = PIS_CLK_PCLK2;
	ald_pis_create(&h_pis);		
}

void adc_dma_trig_by_pis(uint16_t* dstbuf, uint16_t num)
{
	h_adc.hdma.perh = DMA0;
	h_adc.hdma.cplt_cbk = NULL;
	h_adc.hdma.cplt_arg = NULL;
	h_adc.hdma.err_cbk  = NULL;
	h_adc.hdma.err_arg  = NULL;

	ald_dma_config_struct(&h_adc.hdma.config);
	h_adc.hdma.config.src        = (void *)&h_adc.perh->NCHDR;
	h_adc.hdma.config.dst        = (void *)dstbuf;
	h_adc.hdma.config.size       = num;
	h_adc.hdma.config.data_width = DMA_DATA_SIZE_HALFWORD;
	h_adc.hdma.config.src_inc    = DMA_DATA_INC_NONE;
	h_adc.hdma.config.dst_inc    = DMA_DATA_INC_HALFWORD;
	h_adc.hdma.config.msel       = DMA_MSEL_ADC0;
	h_adc.hdma.config.msigsel    = DMA_MSIGSEL_ADC;
	h_adc.hdma.config.channel    = 0;
	ald_dma_config_basic(&h_adc.hdma);
	
	ADC_ENABLE(&h_adc); /* Enable ADC */	
	TIMER_ENABLE(&h_tim); /* Enable pis trigger */
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint16_t i, j;
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	/* PCLK2 bus divide by 2 */ 
	ald_cmu_div_config(CMU_PCLK_2, CMU_DIV_2);
	/* Initialize IO pin */
	toggle_io_init();
	/* Initialize ADC pin */
	adc_pin_init();
	/* Initialize ADC */
	adc_init();
	/* Initialize PIS TIMER */
	pis_tim_init();

	/* Start data data collection */
	adc_dma_trig_by_pis(adcbuf, 1024);

	while (1) {
		if (adcflag) {
			adcflag = 0;

			/* Handle data format */
			for (i = 0, j = 0; i < BUFFER_SIZE; i += 4, j++) {
				buffer[0][j] = adcbuf[i];
				buffer[1][j] = adcbuf[i + 1];
				buffer[2][j] = adcbuf[i + 2];
				buffer[3][j] = adcbuf[i + 3];
			}

			/* Do something */

			adc_dma_trig_by_pis(adcbuf, 1024);
		}
	}
}

/**
  * @}
  */

/**
  * @}
  */
