/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Example DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 

#include <string.h>
#include "main.h"
#include "home_page.h"

/**
  * @defgroup APP 
  * @{
  */

/**
  * @defgroup Compositive Compositive Display Example
  * @{
  */
  
/**
  * @defgroup Compositive_Public_Params Gui private param
  * @{
  */

app_env_t app_env;

/**
  * @}
  */

/**
  * @defgroup DISPLAY Screen display module
  * @{
  */

/**
  * @defgroup DISPLAY_Public_Functions Screen display module
  * @{
  */

bsp_display_env_t disp_env;

/**
  * @}
  */
  
/**
  * @}
  */

/**
  * @defgroup Power_Up_Check printf chip pwoer up check info
  * @{
  */

/**
  * @defgroup Power_Up_Check_Private_Functions  pwoer up check private functions
  * @{
  */
/**
  * @brief  power on ship check
  * @retval None.
  */  
static void power_up_check(void)
{
	uint16_t tmp_color;
	app_env.busy_flag = 1;
	disp_env.point_color = COLOR_TRAN(GRAY);

	bsp_display_image(&disp_env, (uint8_t *)image_logo, 35520, 1);

	bsp_display_by_row(&disp_env, "Eastsoft ES32F0 DevBoard V1.1", 6);
	
	if (app_env.chip_type == CHIP1)
		bsp_display_by_row(&disp_env, "MCU:   ES32F0654LX ", 7);
	
	if (app_env.chip_type == CHIP2)
		bsp_display_by_row(&disp_env, "MCU:   ES32F0334LX ", 7);
	
	bsp_display_by_row(&disp_env, "Core:  ARM Cortex-M0", 8);
	bsp_display_by_row(&disp_env, "Freq:  48MHZ", 9);
	bsp_display_by_row(&disp_env, "Flash: 256KBytes", 10);
	bsp_display_by_row(&disp_env, "Ram:   32KBytes", 11);

	if (bsp_sd_init() != 0) {
 		tmp_color = disp_env.point_color;
		disp_env.point_color = COLOR_TRAN(RED);
		bsp_printf_d(&disp_env, 0, 16 * 12, "SD Card: Fail         [ERROR]");
		disp_env.point_color = tmp_color;
	}
	else {
		bsp_printf_d(&disp_env, 0, 16 * 12, "SD Card: %d KB      [OK]", bsp_sd_get_size() / 2);
	}

	bsp_display_by_row(&disp_env, "EEPROM: 16 KBytes        [OK]", 14);

	if (app_env.app_eth.con_err == 1) {
		tmp_color = disp_env.point_color;
		disp_env.point_color = COLOR_TRAN(RED);
		bsp_display_by_row(&disp_env, "ETH Connect: Fail     [ERROR]", 15);
		disp_env.point_color = tmp_color;
	}
	else {
		bsp_display_by_row(&disp_env, "ETH Connect: Sucess      [OK]", 15);
	}

	bsp_display_by_row(&disp_env, "SPIFLASH: 4MBytes        [OK]", 16);
	disp_env.point_color = COLOR_TRAN(WHITE);
	
	ald_mcu_irq_config(RTC_TSENSE_IRQn, 3, ENABLE);
	
	app_env.busy_flag = 0;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup Menu Display Menu Opration
  * @{
  */

/**
  * @defgroup Menu_Private_Function Menu Opration private Functions
  * @{
  */
/**
  * @brief  show menu information
  * @param  arg: Pointer to parameter.
  * @retval None.
  */
static void menu_display_fun(void *arg)
{
	if (sem_pend(app_env.app_menu.sem_menu))
		return;

	app_env.busy_flag = 1;
	bsp_display_image_zip(&disp_env, get_home_page_frame);
	app_env.busy_flag = 0;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup Press Screen Press Detect Module
  * @{
  */

/**
  * @defgroup Press_Private_Function Screen Press Private functions
  * @{
  */

/**
  * @brief  screen switch opeart.
  * @param  arg: Pointer to parameter.
  * @retval None.
  */
static void screen_switch(void *arg)
{
	uint16_t tmp_color = 0;

	if (sem_pend(app_env.app_pre.sem_press))
		return;

	app_env.busy_flag = 1;

	bsp_display_clear(&disp_env, &disp_env.back_color);	
	
	disp_env.point_color = COLOR_TRAN(WHITE);

	if ((app_env.screen_index != 0) && (app_env.screen_index != 1)) {
		bsp_display_by_row(&disp_env, "Eastsoft", 1);
		bsp_display_by_row(&disp_env, "-----------------------------", 2);
		bsp_display_by_row(&disp_env, "Return home:  >>>>>>>>>>>>>>>>", 20);
	}
	
	switch (app_env.screen_index) {
		case 5:
			bsp_display_by_row(&disp_env, " +-------------------------+", 3);
			bsp_display_string(&disp_env, 0, 16 * 3, " |         < LED >         |", 0);
			bsp_display_string(&disp_env, 0, 16 * 4, " |     LED Light Flashe    |", 0);
			bsp_display_by_row(&disp_env, " +-------------------------+", 6);
			break;
		case 6:
			bsp_display_by_row(&disp_env, " +-------------------------+", 3);
			bsp_display_string(&disp_env, 0, 16 * 3, " |       < RGB Card >      |", 0);
			bsp_display_string(&disp_env, 0, 16 * 4, " |  Pess K3: Add direct    |", 0);
			bsp_display_string(&disp_env, 0, 16 * 5, " |  Pess K4: RED Weight    |", 0);
			bsp_display_string(&disp_env, 0, 16 * 6, " |  Pess K5: Green Weight  |", 0);
			bsp_display_string(&disp_env, 0, 16 * 7, " |  Pess K6: Bule Weight   |", 0);
			bsp_display_by_row(&disp_env, " +-------------------------+", 9);

			app_env.app_rgb.rgb_duty.red_duty   = 1;
			app_env.app_rgb.rgb_duty.green_duty = 1;
			app_env.app_rgb.rgb_duty.blue_duty  = 1;

			lptim_pwm_set_duty(&app_env.app_rgb.lptim, app_env.app_rgb.rgb_duty.blue_duty);
			ald_timer_pwm_set_duty(&app_env.app_rgb.tim.tim3_h, TIMER_CHANNEL_1, 100 - app_env.app_rgb.rgb_duty.red_duty * 6);
			ald_timer_pwm_set_duty(&app_env.app_rgb.tim.tim3_h, TIMER_CHANNEL_2, 100 - app_env.app_rgb.rgb_duty.green_duty * 6);

			if (app_env.app_rgb.rgb_duty.count_tran == 0) {
				bsp_display_by_row(&disp_env, "    Mode: Increase Weight", 10);
			}
			else {
				bsp_display_by_row(&disp_env, "      Mode: Reduce Weight ", 10);
			}

			bsp_printf_d(&disp_env, 0, 16 * 12, "       Red Weight  : %d ", app_env.app_rgb.rgb_duty.red_duty * 17);
			bsp_printf_d(&disp_env, 0, 16 * 13, "       Green Weight: %d ", app_env.app_rgb.rgb_duty.green_duty * 17);
			bsp_printf_d(&disp_env, 0, 16 * 14, "       Blue Weight : %d ", app_env.app_rgb.rgb_duty.blue_duty * 17);
			break;
		case 7:
			bsp_display_by_row(&disp_env, " +-------------------------+", 3);
			bsp_display_string(&disp_env, 0, 16 * 3, " |         < KEY >         |", 0);
			bsp_display_string(&disp_env, 0, 16 * 4, " |  Please Press K3 To K6  |", 0);
			bsp_display_by_row(&disp_env, " +-------------------------+", 6);

			bsp_display_by_row(&disp_env, "          K3: Release", 9);
			bsp_display_by_row(&disp_env, "          K4: Release", 11);
			bsp_display_by_row(&disp_env, "          K5: Release", 13);
			bsp_display_by_row(&disp_env, "          K6: Release", 15);
			break;
		case 8:
			bsp_display_by_row(&disp_env, " +-------------------------+", 3);
			bsp_display_string(&disp_env, 0, 16 * 3, " |         < ADC >         |", 0);
			bsp_display_string(&disp_env, 0, 16 * 4, " |    Display ADC Value    |", 0);
			bsp_display_by_row(&disp_env, " +-------------------------+", 6);
		
			bsp_printf_d(&disp_env, 0, 16*8, "    Sample Value: %.4f V", 0);
			break;
		case 9:
			bsp_display_by_row(&disp_env, " +-------------------------+", 3);
			bsp_display_string(&disp_env, 0, 16 * 3, " |         < TEMP >        |", 0);
			bsp_display_string(&disp_env, 0, 16 * 4, " |    Display Temperature  |", 0);
			bsp_display_string(&disp_env, 0, 16 * 5, " |    Display Humidity     |", 0);
			bsp_display_by_row(&disp_env, " +-------------------------+", 7);
			break;
		case 10:
			bsp_display_by_row(&disp_env, " +-------------------------+", 3);
			bsp_display_string(&disp_env, 0, 16 * 3, " |        < TF Card >      |", 0);
			bsp_display_string(&disp_env, 0, 16 * 4, " |   Pess K3: Write Data   |", 0);
			bsp_display_string(&disp_env, 0, 16 * 5, " |   Pess K4: Erase Data   |", 0);
			bsp_display_string(&disp_env, 0, 16 * 6, " |   Pess K5: Read Data    |", 0);
			bsp_display_by_row(&disp_env, " +-------------------------+", 8);
			break;
		case 11:
			bsp_display_by_row(&disp_env, " +-------------------------+", 3);
			bsp_display_string(&disp_env, 0, 16 * 3, " |         < GYRO >        |", 0);
			bsp_display_string(&disp_env, 0, 16 * 4, " |   Display Acceleration  |", 0);
			bsp_display_by_row(&disp_env, " +-------------------------+", 6);
			break;
		case 12:
			bsp_display_by_row(&disp_env, " +-------------------------+", 3);
			bsp_display_string(&disp_env, 0, 16 * 3, " |         < ETH >         |", 0);
			bsp_display_string(&disp_env, 0, 16 * 4, " |  Recv/Send Data By UDP  |", 0);
			bsp_display_by_row(&disp_env, " +-------------------------+", 6);
		
			if (app_env.app_eth.con_err == 1) {
				tmp_color = disp_env.point_color;
				disp_env.point_color = COLOR_TRAN(RED);
				bsp_display_by_row(&disp_env, "      ETH Connect: Fail", 8);
				disp_env.point_color = tmp_color;
			}
			else {
				bsp_display_by_row(&disp_env, "      ETH Connect: Sucess", 8);
			}
			break;
		default:
			break;
	}
	app_env.busy_flag = 0;
}
/**
  * @brief  show chip information
  * @param  x: x axes Point.
  * @param  y: y axes Point.
  * @param  flag: slide flag.
  * @retval None.
  */
static void screen_press_detect(uint16_t x, uint16_t y, uint8_t flag)
{
	if (flag == 1) { 	//right
		if ((app_env.screen_index != 1) && (app_env.screen_index != 0)) {
			app_env.screen_index = 1;
			sem_post(app_env.app_menu.sem_menu);
		}

		return;
	}
	
	if (flag == 2) { 	//left
		switch (app_env.screen_index){
		case 1:	
		case 2:
		case 3:
		case 4:
			if (app_env.screen_index == 1)
				app_env.screen_index = 2;
			
			app_env.screen_index++;
		
			if (app_env.screen_index > 4)
				app_env.screen_index = 3;

			sem_post(app_env.app_chip.sem_chip);
			break;
		default:
			break;
		}
		
		return;
	}

	if (app_env.screen_index == 0) {
		app_env.screen_index = 1;
		sem_post(app_env.app_menu.sem_menu);
		return;
	}
	
	if (app_env.screen_index == 1) {
		if ((x >= 13) && (x <= 114)) {		//LED
			if ((y >= 12) && (y <= 57)) {
				app_env.screen_index = 5;
				sem_post(app_env.app_pre.sem_press);
			}
		}
		
		if ((x >= 128) && (x <= 228)) {		//RGB
			if ((y >= 12) && (y <= 57)) {
				app_env.screen_index = 6;
				sem_post(app_env.app_pre.sem_press);
			}
		}

		if ((x >= 13) && (x <= 115)) {		//KEY
			if ((y >= 96) && (y <= 141)) {
				app_env.screen_index = 7;
				sem_post(app_env.app_pre.sem_press);
			}
		}

		if ((x >= 128) && (x <= 240)) {		//ADC
			if ((y >= 96) && (y <= 141)) {
				app_env.screen_index = 8;
				sem_post(app_env.app_pre.sem_press);
			}
		}

		if ((x >= 12) && (x <= 114)) {		//TEMP
			if ((y >= 179) && (y <= 224)) {
				app_env.screen_index = 9;
				sem_post(app_env.app_pre.sem_press);
			}
		}

		if ((x >= 128) && (x <= 228)) {		//TF
			if ((y >= 180) && (y <= 225)) {
				app_env.screen_index = 10;
				sem_post(app_env.app_pre.sem_press);
			}
		}

		if ((x >= 12) && (x <= 114)) {		//GYRO
			if ((y >= 257) && (y <= 300)) {
				app_env.screen_index = 11;
				sem_post(app_env.app_pre.sem_press);
			}
		}

		if ((x >= 128) && (x <= 228)) {		//ETH
			if ((y >= 256) && (y <= 302)) {
				app_env.screen_index = 12;
				sem_post(app_env.app_pre.sem_press);
			}
		}		
	}
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup INFO chip info display
  * @{
  */

/**
  * @defgroup INFO_Private_Functions chip private functions
  * @{
  */

/**
  * @brief  show chip information
  * @param  arg: Pointer to parameter.
  * @retval None.
  */

static void chip_info_display(void* arg)
{
	uint16_t tmp_color;
	uint16_t tmp_back = 0x8430;
	
	if (sem_pend(app_env.app_chip.sem_chip))
		return;

	app_env.busy_flag = 1;	
	tmp_color = disp_env.back_color;
	disp_env.back_color = tmp_back;
	bsp_display_clear(&disp_env, &tmp_back);

	if (app_env.screen_index == 2) {
		
	}
	else if (app_env.screen_index == 3) {
		disp_env.point_color = COLOR_TRAN(WHITE);
		bsp_display_by_row(&disp_env, "Eastsoft", 1);
		bsp_display_by_row(&disp_env, "-----------------------------", 2);
		bsp_display_by_row(&disp_env, "MCU:   ES32F0654LX", 3);
		bsp_display_by_row(&disp_env, "Core:  ARM Cortex-M0", 4);
		bsp_display_by_row(&disp_env, "Freq:  48MHZ", 5);
		bsp_display_by_row(&disp_env, "Flash: 256KBytes", 6);
		bsp_display_by_row(&disp_env, "Ram:   32KBytes", 7);
		bsp_display_by_row(&disp_env, "Pack:  LQFP100", 8);
		bsp_display_by_row(&disp_env, "IO:    88", 9);
		bsp_display_by_row(&disp_env, "ADTIM: 1", 10);
		bsp_display_by_row(&disp_env, "GPTIM: 3", 11);
		bsp_display_by_row(&disp_env, "BSTIM: 4", 12);
		bsp_display_by_row(&disp_env, "DMA-C: 6", 13);
		bsp_display_by_row(&disp_env, "SPI:   2", 14);
		bsp_display_by_row(&disp_env, "I2C:   2", 15);
		bsp_display_by_row(&disp_env, "USART: 2", 16);
		bsp_display_by_row(&disp_env, "UART:  4", 17);
		bsp_display_by_row(&disp_env, "CAN:   1", 18);
		bsp_display_by_row(&disp_env, "Next page:    <<<<<<<<<<<<<<<<", 19);
		bsp_display_by_row(&disp_env, "Return home:  >>>>>>>>>>>>>>>>", 20);
	}
	else if (app_env.screen_index == 4) {
		disp_env.point_color = COLOR_TRAN(WHITE);
		bsp_display_by_row(&disp_env, "Eastsoft", 1);
		bsp_display_by_row(&disp_env, "-----------------------------", 2);
		bsp_display_by_row(&disp_env, "MCU:    ES32F0334LX", 3);
		bsp_display_by_row(&disp_env, "Core:   ARM Cortex-M0", 4);
		bsp_display_by_row(&disp_env, "Freq:   48MHZ", 5);
		bsp_display_by_row(&disp_env, "Flash:  256KBytes", 6);
		bsp_display_by_row(&disp_env, "Ram:    32KBytes", 7);
		bsp_display_by_row(&disp_env, "Pack:   LQFP100", 8);
		bsp_display_by_row(&disp_env, "IO:     88", 9);
		bsp_display_by_row(&disp_env, "ADTIM:  1", 10);
		bsp_display_by_row(&disp_env, "GPTIM:  3", 10);
		bsp_display_by_row(&disp_env, "BSTIM:  4", 11);
		bsp_display_by_row(&disp_env, "LPTIM:  1", 11);
		bsp_display_by_row(&disp_env, "DMA-C:  6", 12);
		bsp_display_by_row(&disp_env, "SPI:    2", 13);
		bsp_display_by_row(&disp_env, "I2C:    2", 14);
		bsp_display_by_row(&disp_env, "USART:  2", 15);
		bsp_display_by_row(&disp_env, "UART:   2", 16);
		bsp_display_by_row(&disp_env, "LPUART: 1", 17);
		bsp_display_by_row(&disp_env, "LCD:    8 * 36", 18);
		bsp_display_by_row(&disp_env, "Next page:    <<<<<<<<<<<<<<<<", 19);
		bsp_display_by_row(&disp_env, "Return home:  >>>>>>>>>>>>>>>>", 20);
	}
	else {}
	
	disp_env.back_color = tmp_color;
	app_env.busy_flag = 0;
	return;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup GYRO GYRO moudule
  *@{
  */

/**
  * @defgroup GYRO_Public_Params GYRO module public param
  * @{
  */

/**
  * @brief GYRO moudle initialization and read struct
  */

bsp_mpu9250_env_t h_mpu;
bsp_mpu9250_data_t h_data;

/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup RGB rgb module
  * @{
  */
 
/**
  * @defgroup RGB_Private_Functions rgb mdoule private functions
  * @{
  */

/**
  * @brief  rgb task function
  * @param  arg: Pointer to parameter.
  * @retval None
  */
static void rgb_task_func(void * arg)
{
	if (sem_pend(app_env.app_rgb.sem_rgb))
		return;
		
	if (app_env.app_rgb.rgb_duty.red_flag > 1) {
		app_env.app_rgb.rgb_duty.red_flag--;

		if (app_env.app_rgb.rgb_duty.count_tran == 0) {
			app_env.app_rgb.rgb_duty.red_duty  += 1;

			if (app_env.app_rgb.rgb_duty.red_duty >= 15) 
				app_env.app_rgb.rgb_duty.red_duty = 15;
		}
		if (app_env.app_rgb.rgb_duty.count_tran == 1) {
			app_env.app_rgb.rgb_duty.red_duty  -= 1;

			if (app_env.app_rgb.rgb_duty.red_duty <= 1) 
				app_env.app_rgb.rgb_duty.red_duty = 1;
		}

		ald_timer_pwm_set_duty(&app_env.app_rgb.tim.tim3_h, TIMER_CHANNEL_2, 100 - app_env.app_rgb.rgb_duty.red_duty * 6);
		bsp_printf_d(&disp_env, 0, 16 * 12, "       Red Weight  : %d ", app_env.app_rgb.rgb_duty.red_duty * 17);
	}

	if (app_env.app_rgb.rgb_duty.green_flag > 1) {
		app_env.app_rgb.rgb_duty.green_flag--;

		if (app_env.app_rgb.rgb_duty.count_tran == 0) {
			app_env.app_rgb.rgb_duty.green_duty  += 1;

			if (app_env.app_rgb.rgb_duty.green_duty >= 15) 
				app_env.app_rgb.rgb_duty.green_duty = 15;
		}
		if (app_env.app_rgb.rgb_duty.count_tran == 1) {
			app_env.app_rgb.rgb_duty.green_duty  -= 1;

			if (app_env.app_rgb.rgb_duty.green_duty <= 1) 
				app_env.app_rgb.rgb_duty.green_duty = 1;
		}

		ald_timer_pwm_set_duty(&app_env.app_rgb.tim.tim3_h, TIMER_CHANNEL_1, 100 - app_env.app_rgb.rgb_duty.green_duty * 6);
		bsp_printf_d(&disp_env, 0, 16 * 13, "       Green Weight: %d ", app_env.app_rgb.rgb_duty.green_duty * 17);
	}

	if (app_env.app_rgb.rgb_duty.blue_flag > 1) {
		app_env.app_rgb.rgb_duty.blue_flag--;
			
		if (app_env.app_rgb.rgb_duty.count_tran == 0) {
			app_env.app_rgb.rgb_duty.blue_duty  += 1;

			if (app_env.app_rgb.rgb_duty.blue_duty >= 15) 
				app_env.app_rgb.rgb_duty.blue_duty = 15;
		}
		if (app_env.app_rgb.rgb_duty.count_tran == 1) {
			app_env.app_rgb.rgb_duty.blue_duty  -= 1;

			if (app_env.app_rgb.rgb_duty.blue_duty <= 1) 
				app_env.app_rgb.rgb_duty.blue_duty = 1;
		}

		lptim_pwm_set_duty(&app_env.app_rgb.lptim, app_env.app_rgb.rgb_duty.blue_duty);
		bsp_printf_d(&disp_env, 0, 16 * 14, "       Blue Weight : %d ", app_env.app_rgb.rgb_duty.blue_duty * 17);
	}
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup ADC adc module
  * @{
  */
 
/**
  * @defgroup ADC_Private_Functions adc mdoule private functions
  * @{
  */

/**
  * @brief  adc task function
  * @param  arg: Pointer to parameter.
  * @retval None
  */
static void adc_task_func(void * arg)
{
	if (sem_pend(app_env.app_adc.sem_adc))
		return;

	if (app_env.screen_index == 8) {
		uint32_t adc_value = 0;

		/* Get the voltage value */
		adc_value = adc_average();
		app_env.busy_flag = 1;
		bsp_printf_d(&disp_env, 0, 16 * 8, "    Sample Value: %.4f V", adc_value / (1000.0));
		app_env.busy_flag = 0;
	}	
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup GYRO adc module
  * @{
  */
 
/**
  * @defgroup GYRO_Private_Functions gyro mdoule private functions
  * @{
  */

/**
  * @brief  gyro task function
  * @param  arg: Pointer to parameter.
  * @retval None
  */
static void gyro_task_func(void * arg)
{
	static int16_t x_pos, y_pos, tmp;
	static int16_t curx_pos = 120, cury_pos = 200;

	if (sem_pend(app_env.app_gyro.sem_gyro))
		return;

	if (app_env.screen_index == 11) {
		bsp_draw_circle(&disp_env, 120, 200, 60, WHITE);
		bsp_mpu_data_measure(&h_mpu, &h_data);
		x_pos = ((int16_t)h_data.acce[0] >> 8);
		y_pos = ((int16_t)h_data.acce[1] >> 7) / 2;

		if (x_pos > 60)
			x_pos = 60;
		if (x_pos < -60)
			x_pos = -60;
		if (y_pos > 60)
			y_pos = 60;
		if (y_pos < -60)
			y_pos = -60;

		tmp = x_pos;
		x_pos = 120 + y_pos;
		y_pos = 200 + tmp;

		if ((abs(x_pos - curx_pos) > 1) || (abs(y_pos - cury_pos) > 1)) {
			app_env.busy_flag = 1;
			bsp_draw_clear_big_point(&disp_env, x_pos, y_pos, WHITE, curx_pos, cury_pos);
			
			curx_pos = x_pos;
			cury_pos = y_pos;
			app_env.busy_flag = 0;
		}
	}
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup RTC rtc module
  * @{
  */
 
/**
  * @defgroup RTC_Private_Functions rtc mdoule private functions
  * @{
  */

/**
  * @brief  rtc second irq show function
  * @param  arg: Pointer to parameter.
  * @retval None
  */
static void rtc_second_show(void * arg)
{
	uint8_t temp, humi;
	uint16_t point_color;
	uint16_t color;

	if (sem_pend(app_env.app_rtc.sem_rtc))
		return;

	if (app_env.screen_index == 0) {
		app_env.app_menu.wait++;
		if (app_env.app_menu.wait == 6) {
			app_env.screen_index = 1;
			sem_post(app_env.app_menu.sem_menu);
		}
	}
	
	if (app_env.screen_index == 2) {
		color = disp_env.back_color;
		disp_env.back_color = 0x8430;		
	}
	else if (app_env.screen_index == 3) {
		color = disp_env.back_color;
		disp_env.back_color = 0x8430;
	}
	else if (app_env.screen_index == 4) {
		color = disp_env.back_color;
		disp_env.back_color = 0x8430;
	}
	else {}

	if ((app_env.screen_index != 0) && (app_env.screen_index != 1)) {
		ald_rtc_get_time(&app_env.app_rtc.time, RTC_FORMAT_DEC);
		app_env.busy_flag = 1;
		bsp_printf_d(&disp_env, 168, 0, "%.2d:%.2d:%.2d", app_env.app_rtc.time.hour, app_env.app_rtc.time.minute, app_env.app_rtc.time.second);
		app_env.busy_flag = 0;
	}

	if ((app_env.screen_index == 2) || (app_env.screen_index == 3) || (app_env.screen_index == 4))
		disp_env.back_color = color;

	if (app_env.screen_index == 9) {
		app_env.app_rtc.sec += 1;
		if (app_env.app_rtc.sec % 2) {
			app_env.busy_flag = 1;
			bsp_dht11_read_data(&temp, &humi);
			bsp_printf_d(&disp_env, 0, 16 * 8, "       Current Temp: %.2d ", temp);
			bsp_printf_d(&disp_env, 0, 16 * 9, "       Current Humi: %.2d ", humi);
			app_env.busy_flag = 0;
		}
	}
	if (app_env.screen_index == 12) {
		app_env.busy_flag = 1;

		if (test_connect() == 0) {			
			if (app_env.app_eth.con_err == 1)
				bsp_display_by_row(&disp_env, "      ETH Connect: Sucess", 8);
			
			app_env.app_eth.con_err = 0;
		}
		else {
			if (app_env.app_eth.con_err == 0) {
				point_color = disp_env.point_color;
				disp_env.point_color = COLOR_TRAN(RED);
				bsp_display_by_row(&disp_env, "      ETH Connect: Fail  ", 8);	
				disp_env.point_color = point_color;

				bsp_display_string(&disp_env, 0, 16 * 9, "                              ", 0);
				bsp_display_string(&disp_env, 0, 16 * 10, "                              ", 0);
			}

			app_env.app_eth.con_err = 1;
		}

		app_env.busy_flag = 0;
	}
}
/**
  * @defgroup RTC_Public_Functions rtc mdoule public functions
  * @{
  */

 /**
  * @brief  RTC Second IRQ handler
  * @retval None
  */
__isr__ void rtc_irq_handler(void)
{
	sem_post(app_env.app_rtc.sem_rtc);
	return;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */

/**
  * @defgroup KEY key moudle
  * @{
  */

/**
  * @defgruop KEY_Private_Functions key module private functions
  * @{
  */
/**
  * @brief  Callback function of key's event.
  * @param  event: Type of key's event.
  * @retval None
  */
static void key_proc(bsp_key_event_t event)
{
	ke_msg_t *msg;

	if ((msg = (ke_msg_t *)ke_malloc(sizeof(ke_msg_t))) == NULL)
		KE_PANIC();

	msg->rsv = (uint32_t)event;
	que_enq(app_env.app_key.que_key, msg);

	return;
}

/**
  * @brief  Callback function of key's task.
  * @param  arg: Pointer to parameter.
  * @retval None
  */
static void task_key_func(void *arg)
{
	uint8_t tf_data[512] = {0};
	uint8_t tmp = 0;
	uint8_t data_falg = 0;

	ke_msg_t *msg;

	if ((msg = que_deq(app_env.app_key.que_key)) == NULL)
		return;

	app_env.app_key.tmp = (bsp_key_event_t)msg->rsv;
	UNUSED(app_env.app_key.tmp);
	free(msg);

	app_env.busy_flag = 1;
	if (app_env.screen_index == 7) {	//KEY
		if (app_env.app_key.tmp == BSP_KEY1_PRESS)
			bsp_display_by_row(&disp_env, "          K3: Press  ", 9);
		if (app_env.app_key.tmp == BSP_KEY1_RELEASE)
			bsp_display_by_row(&disp_env, "          K3: Release", 9);
		if (app_env.app_key.tmp == BSP_KEY2_PRESS)
			bsp_display_by_row(&disp_env, "          K4: Press  ", 11);
		if (app_env.app_key.tmp == BSP_KEY2_RELEASE)
			bsp_display_by_row(&disp_env, "          K4: Release", 11);
		if (app_env.app_key.tmp == BSP_KEY3_PRESS)
			bsp_display_by_row(&disp_env, "          K5: Press  ", 13);
		if (app_env.app_key.tmp == BSP_KEY3_RELEASE)
			bsp_display_by_row(&disp_env, "          K5: Release", 13);
		if (app_env.app_key.tmp == BSP_KEY4_PRESS)
			bsp_display_by_row(&disp_env, "          K6: Press  ", 15);
		if (app_env.app_key.tmp == BSP_KEY4_RELEASE)
			bsp_display_by_row(&disp_env, "          K6: Release", 15);
	}
	
	if (app_env.screen_index == 10) {	//TF
		bsp_sd_init();
		if (app_env.app_key.tmp == BSP_KEY1_PRESS) {
			sprintf((char *)tf_data, "RTC Time: %.2d:%.2d:%.2d", app_env.app_rtc.time.hour, app_env.app_rtc.time.minute, app_env.app_rtc.time.second);
			
			if (bsp_sd_write_sector((uint8_t *)tf_data, 0) == 0) {
				bsp_display_by_row(&disp_env, " Write Data To Sector 0:    ", 11);				
				bsp_printf_d(&disp_env, 0, 16 * 11, " %s", (char *)tf_data);
			}
			else {
				disp_env.point_color = COLOR_TRAN(RED);
				bsp_display_by_row(&disp_env, " Write Data To Sector 0: Err", 11);
				bsp_printf_d(&disp_env, 0, 16 * 11, "                   ");
				disp_env.point_color = COLOR_TRAN(WHITE);
			}
		}

		if (app_env.app_key.tmp == BSP_KEY2_PRESS) {
			if (bsp_sd_erase(0, 1) == 0) {
				bsp_display_by_row(&disp_env, " Delete Data On Sector 0: OK ", 14);
			}
			else {
				disp_env.point_color = COLOR_TRAN(RED);
				bsp_display_by_row(&disp_env, " Delete Data On Sector 0: Err", 14);
				bsp_printf_d(&disp_env, 0, 16 * 14, "                   ");
				disp_env.point_color = COLOR_TRAN(WHITE);
			}			
		}

		if (app_env.app_key.tmp == BSP_KEY3_PRESS) {			
			if (bsp_sd_read_sector((uint8_t *)tf_data, 0) == 0) {
				bsp_display_by_row(&disp_env, " Read Data From Sector 0:    ", 16);
				bsp_printf_d(&disp_env, 0, 16 * 16, "                   ");
				
				for (tmp = 0; tmp < 18; tmp++) {
					if (tf_data[tmp] != 0xff) {
						data_falg = 1;
						break;
					}
				}
				
				if (data_falg == 1)
					bsp_printf_d(&disp_env, 0, 16*16, " %s", (char *)tf_data);
			}
			else {
				disp_env.point_color = COLOR_TRAN(RED);
				bsp_display_by_row(&disp_env, " Read Data From Sector 0: Err", 16);
				bsp_printf_d(&disp_env, 0, 16 * 16, "                   ");
				disp_env.point_color = COLOR_TRAN(WHITE);
			}
		}
	}
	
	if (app_env.screen_index == 6) {	//RGB		
		if (app_env.app_key.tmp == BSP_KEY1_PRESS) {
			if (app_env.app_rgb.rgb_duty.count_tran == 1) {
				app_env.app_rgb.rgb_duty.count_tran = 0;
				bsp_display_by_row(&disp_env, "     Mode: Increase Weight", 10);
			}
			else {
				app_env.app_rgb.rgb_duty.count_tran = 1;
				bsp_display_by_row(&disp_env, "      Mode: Reduce Weight ", 10);
			}
		}
		
		if (app_env.app_key.tmp == BSP_KEY2_PRESS) {
			if (app_env.app_rgb.rgb_duty.count_tran == 0) {
				app_env.app_rgb.rgb_duty.red_duty  += 1;

				if (app_env.app_rgb.rgb_duty.red_duty >= 15) 
					app_env.app_rgb.rgb_duty.red_duty = 15;
			}
			if (app_env.app_rgb.rgb_duty.count_tran == 1) {
				app_env.app_rgb.rgb_duty.red_duty  -= 1;

				if (app_env.app_rgb.rgb_duty.red_duty <= 1) 
					app_env.app_rgb.rgb_duty.red_duty = 1;
			}

			ald_timer_pwm_set_duty(&app_env.app_rgb.tim.tim3_h, TIMER_CHANNEL_2, 100 - app_env.app_rgb.rgb_duty.red_duty * 6);
			bsp_printf_d(&disp_env, 0, 16 * 12, "       Red Weight  : %d ", app_env.app_rgb.rgb_duty.red_duty * 17);
		}
		if (app_env.app_key.tmp == BSP_KEY3_PRESS) {
			if (app_env.app_rgb.rgb_duty.count_tran == 0) {
				app_env.app_rgb.rgb_duty.green_duty  += 1;

				if (app_env.app_rgb.rgb_duty.green_duty >= 15) 
					app_env.app_rgb.rgb_duty.green_duty = 15;
			}
			if (app_env.app_rgb.rgb_duty.count_tran == 1) {
				app_env.app_rgb.rgb_duty.green_duty  -= 1;

				if (app_env.app_rgb.rgb_duty.green_duty <= 1) 
					app_env.app_rgb.rgb_duty.green_duty = 1;
			}

			ald_timer_pwm_set_duty(&app_env.app_rgb.tim.tim3_h, TIMER_CHANNEL_1, 100 - app_env.app_rgb.rgb_duty.green_duty * 6);
			bsp_printf_d(&disp_env, 0, 16 * 13, "       Green Weight: %d ", app_env.app_rgb.rgb_duty.green_duty * 17);
		}
		if (app_env.app_key.tmp == BSP_KEY4_PRESS) {			
			if (app_env.app_rgb.rgb_duty.count_tran == 0) {
				app_env.app_rgb.rgb_duty.blue_duty  += 1;

				if (app_env.app_rgb.rgb_duty.blue_duty >= 15) 
					app_env.app_rgb.rgb_duty.blue_duty = 15;
			}
			if (app_env.app_rgb.rgb_duty.count_tran == 1) {
				app_env.app_rgb.rgb_duty.blue_duty  -= 1;

				if (app_env.app_rgb.rgb_duty.blue_duty <= 1) 
					app_env.app_rgb.rgb_duty.blue_duty = 1;
			}

			lptim_pwm_set_duty(&app_env.app_rgb.lptim, app_env.app_rgb.rgb_duty.blue_duty);
			bsp_printf_d(&disp_env, 0, 16 * 14, "       Blue Weight : %d ", app_env.app_rgb.rgb_duty.blue_duty * 17);
		}
	}
	
	if (app_env.screen_index == 12) {	//ETH
		
	}

	app_env.busy_flag = 0;

	return;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup ETH net work module
  * @{
  */

/**
  * @defgroup ETH_Private_Functions ETH Private Functions
  * @{
  */

/**
  * @brief  set udp socket param.
  * @retval None.
  */
static void set_local_net_param(void)
{
	memset(&bsp_w5500_env, 0x00, sizeof(bsp_w5500_env_t));

	bsp_w5500_env.geteway[0] = 192;
	bsp_w5500_env.geteway[1] = 168;
	bsp_w5500_env.geteway[2] = 1;
	bsp_w5500_env.geteway[3] = 1;

	bsp_w5500_env.sub_mask[0] = 255;
	bsp_w5500_env.sub_mask[1] = 255;
	bsp_w5500_env.sub_mask[2] = 255;
	bsp_w5500_env.sub_mask[3] = 0;

	bsp_w5500_env.phy_addr[0] = 0x0c;
	bsp_w5500_env.phy_addr[1] = 0x29;
	bsp_w5500_env.phy_addr[2] = 0xab;
	bsp_w5500_env.phy_addr[3] = 0x7c;
	bsp_w5500_env.phy_addr[4] = 0x00;
	bsp_w5500_env.phy_addr[5] = 0x01;

	bsp_w5500_env.local_ip[0] = 192;
	bsp_w5500_env.local_ip[1] = 168;
	bsp_w5500_env.local_ip[2] = 1;
	bsp_w5500_env.local_ip[3] = 199;

	bsp_w5500_env.local_port[0] = 0x13;
	bsp_w5500_env.local_port[1] = 0x88;

	bsp_w5500_env.mode = UDP_MODE;
	return;
}

/**
  * @brief  configuration udp socket param.
  * @retval None.
  */
static void set_socket(void)
{
	if (bsp_w5500_env.state != 0)
		return;

	if (bsp_w5500_env.mode == TCP_SERVER) {
		if (bsp_w5500_socket_listen(0) == TRUE)
			bsp_w5500_env.state = S_INIT;
		else
			bsp_w5500_env.state = 0;
	}
	else if (bsp_w5500_env.mode == TCP_CLIENT) {
		if (bsp_w5500_socket_connect(0) == TRUE)
			bsp_w5500_env.state = S_INIT;
		else
			bsp_w5500_env.state = 0;
	}
	else {
		if (bsp_w5500_socket_udp(0) == TRUE)
			bsp_w5500_env.state = S_INIT | S_CONN;
		else
			bsp_w5500_env.state = 0;
	}

	return;
}

/**
  * @brief  handler for socket data.
  * @param  s_idx: socket number.
  * @retval None.
  */
static void socket_data_handle(uint8_t s_idx)
{
	uint16_t size;
	
	size = bsp_w5500_read_socket_data(s_idx, bsp_w5500_env.rx_buffer);
	bsp_w5500_env.udp_dip[0] = bsp_w5500_env.rx_buffer[0];
	bsp_w5500_env.udp_dip[1] = bsp_w5500_env.rx_buffer[1];
	bsp_w5500_env.udp_dip[2] = bsp_w5500_env.rx_buffer[2];
	bsp_w5500_env.udp_dip[3] = bsp_w5500_env.rx_buffer[3];

	bsp_w5500_env.udp_dport[0] = bsp_w5500_env.rx_buffer[4];
	bsp_w5500_env.udp_dport[1] = bsp_w5500_env.rx_buffer[5];

	memcpy(bsp_w5500_env.tx_buffer, bsp_w5500_env.rx_buffer + 8, size - 8);
	memcpy(bsp_w5500_env.tx_buffer + size - 8, "\r\n", 2);
	bsp_w5500_write_socket_data(s_idx, bsp_w5500_env.tx_buffer, size - 6);
}

/**
  * @brief  Callback function of Eth's task.
  * @param  arg: Pointer to parameter.
  * @retval None
  */
static void eth_task(void *arg)
{
	if (sem_pend(app_env.app_eth.sem_eth))
		return;

	memset(bsp_w5500_env.rx_buffer, 0, sizeof(bsp_w5500_env.rx_buffer));

	set_socket();
	bsp_w5500_interrupt_process();

	if ((bsp_w5500_env.flag & S_RECEIVE) == S_RECEIVE) {
		bsp_w5500_env.flag &= ~S_RECEIVE;
		socket_data_handle(0);
		app_env.busy_flag = 1;

		bsp_display_string(&disp_env, 0, 16 * 10, "                              ", 0);

		bsp_display_string(&disp_env, 0, 16 * 9, "         Recv Data :", 0);
		bsp_printf_d(&disp_env, 0, 16 * 10, "     %s ", bsp_w5500_env.rx_buffer + 8);

		app_env.busy_flag = 0;
		if (bsp_w5500_env.state == (S_INIT | S_CONN)) {
			bsp_w5500_env.flag &= ~S_TRANSMITOK;
			bsp_w5500_write_socket_data(0, bsp_w5500_env.tx_buffer, strlen((const char *)bsp_w5500_env.tx_buffer));
		}
	}
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @defgroup KERNEL kernel module
  * @{
  */

/**
  * @defgroup KERNEL_Public_Functions key module public functions
  * @{
  */

/**
  * @brief  Callback function of 10ms interrupt.
  * @retval None
  */
void ke_10ms_irq_cbk()
{
	static uint16_t red_go= 0;
	static uint16_t green_go = 0;
	static uint16_t blue_go = 0;
	static uint16_t red = 0;
	static uint16_t green = 0;
	static uint16_t blue = 0;
	static uint8_t  rgb_count = 0;
	
	bsp_key_scan();
	
	if (app_env.screen_index == 12) {
		if (app_env.app_eth.con_err == 1)
			return;

		app_env.app_eth.ms_10 += 1;

		if ((app_env.app_eth.ms_10 % 10) == 0) {
			app_env.app_eth.ms_10 = 0;
			sem_post(app_env.app_eth.sem_eth);
		}
	}

	if (app_env.screen_index == 5) {
		app_env.app_led.led_10ms += 1;
		
		if ((app_env.app_led.led_10ms % 50) == 0) {
			led_operate(0, LED_OFF);
			led_operate(1, LED_OFF);
			led_operate(2, LED_OFF);
			led_operate(3, LED_OFF);
			led_operate(4, LED_OFF);
			led_operate(5, LED_OFF);

			if (app_env.app_led.led_10ms == 50)
				led_operate(0, LED_ON);
			
			if (app_env.app_led.led_10ms == 100)
				led_operate(1, LED_ON);
			
			if (app_env.app_led.led_10ms == 150)
				led_operate(2, LED_ON);

			if (app_env.app_led.led_10ms == 200)
				led_operate(3, LED_ON);

			if (app_env.app_led.led_10ms == 250)
				led_operate(4, LED_ON);

			if (app_env.app_led.led_10ms == 300) {
				led_operate(5, LED_ON);
				app_env.app_led.led_10ms = 0;
			}
		}
	}
	
	if (app_env.screen_index == 6) {
		if (rgb_count % 2) {
			if (app_env.app_key.tmp == BSP_KEY2_PRESS)
				red_go = 1;
			if (app_env.app_key.tmp == BSP_KEY2_RELEASE)
				red_go = 0;

			if (app_env.app_key.tmp == BSP_KEY3_PRESS)
				green_go = 1;
			if (app_env.app_key.tmp == BSP_KEY3_RELEASE)
				green_go = 0;

			if (app_env.app_key.tmp == BSP_KEY4_PRESS)
				blue_go = 1;
			if (app_env.app_key.tmp == BSP_KEY4_RELEASE)
				blue_go = 0;

			if (red_go) {
				red++;

				if (red >= 25) {
					red = 0;
					app_env.app_rgb.rgb_duty.red_flag++;
					sem_post(app_env.app_rgb.sem_rgb);
				}
			}
			else {
				red -= (red % 25);
			}

			if (green_go) {
				green++;

				if (green >= 25) {
					green = 0;
					app_env.app_rgb.rgb_duty.green_flag++;
					sem_post(app_env.app_rgb.sem_rgb);
				}
			}
			else {
				green -= (green % 25);
			}

			if (blue_go) {
				blue++;

				if (blue >= 25) {
					blue = 0;
					app_env.app_rgb.rgb_duty.blue_flag++;
					sem_post(app_env.app_rgb.sem_rgb);
				}
			}
			else {
				blue -= (blue % 25);
			}
		}
		rgb_count++;
	}

	if (app_env.screen_index == 8) {
		app_env.app_adc.ms_10++;

		if (app_env.app_adc.ms_10 >= 30) {
			app_env.app_adc.ms_10 = 0;
			sem_post(app_env.app_adc.sem_adc);
		}
	}

	if (app_env.screen_index == 11) {
		app_env.app_gyro.ms_10++;

		if (app_env.app_gyro.ms_10 >= 1) {
			app_env.app_gyro.ms_10 = 0;
			sem_post(app_env.app_gyro.sem_gyro);
		}
	}

	return;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @brief  This is main function.
  * @retval Status
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();

	/* Set system clock to 48MHz */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);

        #if defined (__ICCARM__)
	ke_init(0x8000, NULL, 0x1000);
        #else
	ke_init(0x8000, NULL);
        #endif

	/* Initialize display */
	memset(&disp_env, 0, sizeof(bsp_display_env_t));
	disp_env.back_color  = COLOR_TRAN(BLACK);
	disp_env.point_color = COLOR_TRAN(WHITE);
	bsp_display_init(&disp_env, screen_press_detect);
	bsp_display_clear(&disp_env, &disp_env.back_color);

	/* Initialize display info */
	app_env.screen_index = 0;

#ifdef ES32F0654
	app_env.chip_type = CHIP1;
#endif

#ifdef ES32F0334
	app_env.chip_type = CHIP2;
#endif
	
	/* Initialize module */
	bsp_dht11_init();
	led_module_init();
	rtc_module_init(&app_env.app_rtc);
	rgb_module_init(&app_env.app_rgb.tim, &app_env.app_rgb.lptim);
	adc_module_init();
	bsp_mpu_init(&h_mpu);

	set_local_net_param();
	bsp_w5500_init(&app_env.app_eth.con_err);

	bsp_key_init();
	bsp_key_bond_func(BSP_KEY_ALL, key_proc);

	/* Initialize module task */
	task_create(screen_switch, NULL, TASK_PRESS_PRIO);
	if ((app_env.app_pre.sem_press = sem_create(TASK_PRESS_PRIO)) == NULL)
		KE_PANIC();

	task_create(menu_display_fun, NULL, TASK_MENU_PRIO);
	if ((app_env.app_menu.sem_menu = sem_create(TASK_MENU_PRIO)) == NULL)
		KE_PANIC();

	task_create(chip_info_display, NULL, TASK_CHIP_PRIO);
	if ((app_env.app_chip.sem_chip = sem_create(TASK_CHIP_PRIO)) == NULL)
		KE_PANIC();

	task_create(rtc_second_show, NULL, TASK_RTC_PRIO);
	if ((app_env.app_rtc.sem_rtc = sem_create(TASK_RTC_PRIO)) == NULL)
		KE_PANIC();

	task_create(rgb_task_func, NULL, TASK_RGB_PRIO);
	if ((app_env.app_rgb.sem_rgb = sem_create(TASK_RGB_PRIO)) == NULL)
		KE_PANIC();

	task_create(adc_task_func, NULL, TASK_ADC_PRIO);
	if ((app_env.app_adc.sem_adc = sem_create(TASK_ADC_PRIO)) == NULL)
		KE_PANIC();

	task_create(gyro_task_func, NULL, TASK_GYRO_PRIO);
	if ((app_env.app_gyro.sem_gyro = sem_create(TASK_GYRO_PRIO)) == NULL)
		KE_PANIC();

	task_create(task_key_func, NULL, TASK_KEY_PRIO);
	if ((app_env.app_key.que_key = que_create(TASK_KEY_PRIO)) == NULL)
		KE_PANIC();

	task_create(eth_task, NULL, TASK_ETH_PRIO);
	if ((app_env.app_eth.sem_eth = sem_create(TASK_ETH_PRIO)) == NULL)
		KE_PANIC();

	/* Enable Interrupt */
	ald_mcu_irq_config(EXTI4_7_IRQn, 3, ENABLE);

	/* power up check */
	power_up_check();

	/* Start kernel */
	ke_start();
	return 0;
}
