/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Nov 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Nov 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
uart_handle_t h_uart;

BYTE work[FF_MAX_SS]; 
BYTE ReadBuffer[1024]= {0}; 
BYTE WriteBuffer[] = "EastSoft IC";

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart0_init(void)
{
	gpio_init_t x;
	
	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_10, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_11, &x);

	/* Initialize uart */
	h_uart.perh             = UART0;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);
}

/**
  * @brief  SdCard test function
  * @retval None
  */
void sdcard_test_entry(void);

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	/* Enable peripheral clock */ 
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	/* Enable uart interrupt */
	ald_mcu_irq_config(UART0_IRQn, 3, ENABLE);

	uart0_init();
	
	while(1) {
		ald_delay_ms(1500);
		sdcard_test_entry();
	}
}

/**
  * @brief  SdCard test function
  * @retval None
  */
void sdcard_test_entry(void)
{
	FIL fil;            
	FATFS fs;           
	FRESULT rs;
	UINT num;
	
	rs = f_mount(&fs, "0:", 1);
	if (rs == FR_NO_FILESYSTEM) {
		printf("\r\n SDCard have no filesystem.\r\n");
		rs = f_mkfs("0:", 0, 0, work, sizeof(work));

		if (rs == FR_OK) {
			printf("SDCard format successfully.\r\n");
			rs = f_mount(NULL,"0:",1);
			rs = f_mount(&fs,"0:",1);
		} else {
			printf("SDCard format fail.\r\n");
		}
	} else if (rs != FR_OK) {
		printf("!! SD mount fail(%d)\r\n",rs);
	} else {
		printf("SDCard mount successfully.\r\n");
	}
	printf("\r\n------------------ File write test ------------------\r\n");
	rs = f_open(&fil,"fileWR.txt", FA_CREATE_ALWAYS|FA_WRITE);
	if ( rs == FR_OK ) {
		printf("Open test file succuss.\r\n");
		rs = f_write(&fil, WriteBuffer, sizeof(WriteBuffer), &num);
		if (rs == FR_OK) {
			printf("File write success. Write number: %d\r\n", num);
			printf("Write content: \r\n%s\r\n", WriteBuffer);
		} else {
			printf("!!File write fail: (%d)\n",rs);
		}
		
		f_close(&fil);
	} else {
		printf("!!Open/Create file fail. \r\n");
	}
	printf("\r\n------------------ File read test -------------------\r\n");
	rs = f_open(&fil,"fileWR.txt",FA_OPEN_EXISTING|FA_READ);
	if (rs == FR_OK) {
		printf("Open file success.\r\n");
		rs = f_read(&fil, ReadBuffer, sizeof(ReadBuffer), &num);
		if (rs == FR_OK) {
			printf("File read success. Read number: %d\r\n",num);
			printf("Read content: \r\n%s \r\n", ReadBuffer);
		} else {
			printf("!!File read fail: (%d)\n",rs);
		}
	} else {
		printf("!!Open file fail\r\n");
	}
	f_close(&fil);
	f_mount(NULL,"0:",1);
}

/**
  * @brief  Reallocate the printf fputc
  * @param  ch: promotion of the character to be written
  * @param  f : pointer to a FILE object
  * @retval Write state
  */
int fputc(int ch, FILE *f)
{
	h_uart.perh->TBR = (uint8_t)ch;
	while((ald_uart_get_status(&h_uart, UART_STATUS_TEM)) != SET);

	return ch;
}

/**
  * @}
  */
/**
  * @}
  */
