/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 


#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
uint32_t lcd_buf[8] = {0};

/**
  * @brief  Initializate lcd pin.
  * @retval None
  */
void lcd_pin_init(void)
{
	md_gpio_init_t ald_gpio_init;
	/* LCD  	com1 com2 com3 com4 seg1  seg2  seg3  seg4  seg5  seg6  seg7  seg8
	 * MCU_LCD      com4 com3 com2 com1 seg16 seg21 seg10 seg15 seg14 seg13 seg12 seg11
	 */	
		
	/* COM4 */
	ald_gpio_init.pupd = MD_GPIO_FLOATING;
	ald_gpio_init.odrv = MD_GPIO_OUT_DRIVE_NORMAL;
	ald_gpio_init.type = MD_GPIO_TYPE_CMOS;
	ald_gpio_init.odos = MD_GPIO_PUSH_PULL;
	ald_gpio_init.mode = MD_GPIO_MODE_CLOSE;
	ald_gpio_init.func = MD_GPIO_FUNC_0;
	md_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM4, &ald_gpio_init);
	
	/* COM3 */	
	md_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM3, &ald_gpio_init);
	/* COM2 */	
	md_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM2, &ald_gpio_init);
	/* COM1 */	
	md_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM1, &ald_gpio_init);
	/* SEG1 */
	md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG1, &ald_gpio_init);
	/* SEG2 */
	md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG2, &ald_gpio_init);
	/* SEG3 */
	md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG3, &ald_gpio_init);
	/* SEG4 */
	md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG4, &ald_gpio_init);
	/* SEG5*/
	md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG5, &ald_gpio_init);	
	/* SEG6 */
	md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG6, &ald_gpio_init);
	/* SEG7 */
	md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG7, &ald_gpio_init);
	/* SEG8 */
	md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG8, &ald_gpio_init);
}

/**
  * @brief  Initializate lcd.
  * @retval None
  */
void lcd_init(void)
{
	md_lcd_init_t init;
	
	lcd_pin_init();

	/* Initialize lcd */
	md_lcd_struct_init(&init);
	md_lcd_init(LCD ,&init);

	/* seg16 seg21 seg10 seg15 seg14 seg13 seg12 seg11 */
	md_lcd_set_segcr0_value(0x0021FC00);
	md_lcd_set_segcr1_value(0x00000000);

	/* Enable lcd */
	md_lcd_enable_oe();
	md_lcd_enable_en();
}

/**
 * @brief 1~4 Digital Position COM And SEG MAP.
 */
const uint8_t  dig8[16]  = {
	0xED, 0x44, 0xD9, 0xD5, 0x74, 0xB5, 0xBD, 0xC4,
	0xFD, 0xF5, 0xFC, 0x3D, 0xA9, 0x5D, 0xB9, 0xB8
};

/**
 * @brief  LCD First Volume COM And SEG MAP.
 */
const uint32_t lcd_map[8] = {
	(1 << 16), (1 << 21), (1 << 10), (1 << 15),
	(1 << 14), (1 << 13), (1 << 12), (1 << 11)
};

/**
  * @brief  LCD fill display buffer.
  * @param  flushbuf:buffer start address.
  * @param  idx: display digital index
  * @param  number: number 0x0 ~ 0x0F
  * @retval None.
  */
void lcd_display(uint32_t *flushbuf, uint8_t idx, uint8_t number)
{
	uint8_t i = 0;
	uint8_t temp = 0;
	uint8_t offset = 0x3;
	if (idx > 4 || idx < 1) return;
	for (i = 0; i < 4; i++) {				
		temp = (dig8[number] & offset) >> 2 * i;		
		switch (temp) {
		case 1:
			flushbuf[3 - i] |= lcd_map[idx * 2 - 1];
			break;
		case 2:
			flushbuf[3 - i] |= lcd_map[idx * 2 - 2];
			break;
		case 3:
			flushbuf[3 - i] |= lcd_map[idx * 2 - 1] | lcd_map[idx * 2 - 2];
			break;
		default:
			break;
		}
		offset <<= 2;
	}
}

/**
  * @brief  Reflush the used com_seg buffer.
  * @param  srcbuf:the source data start address.
  * @retval None.
  */
void lcd_reflush(uint32_t *flushbuf)
{
	uint8_t i = 0;
	uint8_t tmp[4][2] = {{2,3},{4,5},{6,7},{8,9}};

	for (i = 0; i < 4; ++i)
		md_lcd_set_seg_buf_data(tmp[i][0], flushbuf[i]);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint8_t i = 0;
	uint8_t j = 0;
	
	md_init_1ms_tick();
	
	SYSCFG_UNLOCK();
	md_cmu_set_lcd_clock_source(0x4);
	SYSCFG_LOCK();
	lcd_init();

	md_delay_1ms(1000);
	while (1) {
		for (i = 0; i < 16; ++i) {
			md_delay_1ms(200);
			lcd_display(lcd_buf, 1, i);
			lcd_reflush(lcd_buf);

			for( j = 0; j < 8; ++j)
				lcd_buf[j] = 0;
		}		
	}
}

/**
  * @}
  */
/**
  * @}
  */
