/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */ 

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples_PMU
  * @{
  */

/* Only one macro can be opened at a time */
#define LP_MODE_STOP1
//#define LP_MODE_STOP2
//#define LP_MODE_STANDBY

uart_handle_t h_uart;
char uart_buf[64];
uint32_t _cnt = 0;

void uart_stdio_init(void);
void printf_e(const char *fmt, ...);

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void exti_pin_init(void)
{
	gpio_init_t x;

	/* Initialize PA1 as input */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &x);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	exti_init_t exti;

	/* Initialize ALD */
	ald_cmu_init();
	/* Enable external interrupt */
	ald_mcu_irq_config(EXTI0_3_IRQn, 3, ENABLE);

	exti_pin_init();
	uart_stdio_init();

	printf_e("System start...\r\n");

	/* Initialize external interrupt */
	exti.filter      = ENABLE;
	exti.cks         = EXTI_FILTER_CLOCK_10K;
	exti.filter_time = 10;
	ald_gpio_exti_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &exti);

	/* Clear interrupt flag */
	ald_gpio_exti_clear_flag_status(PMU_EXTI_PIN);
	/* Configure interrupt */
	ald_gpio_exti_interrupt_config(PMU_EXTI_PIN, EXTI_TRIGGER_RISING_EDGE, ENABLE);

#ifdef LP_MODE_STOP1
	ald_pmu_stop1_enter();
#endif
#ifdef LP_MODE_STOP2
	ald_pmu_stop2_enter();
#endif
#ifdef LP_MODE_STANDBY
	ald_pmu_standby_enter(PMU_STANDBY_PORT_SEL_PA1, PMU_STANDBY_LEVEL_HIGH);
#endif

	while (1) {
		printf_e("Count: 0x%x\r\n", ++_cnt);
		ald_delay_ms(1000);
	}
}

/**
  * @brief  GPIO IRQ handler
  * @retval None
  */
__isr__ void gpio_irq_handler(void)
{
	/* Check and clear interrupt flag */
	ald_gpio_exti_get_flag_status(PMU_EXTI_PIN);
	ald_gpio_exti_clear_flag_status(PMU_EXTI_PIN);

	return;
}

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);

	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_0, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_1, &x);

	/* Initialize uart */
	h_uart.perh             = UART1;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	return;
}
/**
  * @}
  */

/**
  * @}
  */
