/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uart_handle_t h_uart;
adc_timer_config_t config;
char uart_buf[64];
env_t _env;

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = UART_FUNC_TX;
	ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = UART_FUNC_RX;
	ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

	/* Initialize uart */
	h_uart.perh             = UART0;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	return;
}

/**
  * @brief  ADC convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void complete_cbk(adc_handle_t *arg)
{
	_env.flag = 1;
	return;
}

/**
  * @brief  Initializate pin of adc module.
  * @retval None.
  */
void adc_pin_init(void)
{
	gpio_init_t x;

	memset(&x, 0, sizeof(gpio_init_t));

	/* Initialize ADC pin */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;
	ald_gpio_init(GPIO_CH0_PORT, ADC_CH0_PIN, &x);

	return;
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
	config.time     = 200;
	config.size     = 100;
	config.buf      = _env.buf;
	config.n_ref    = ADC_NEG_REF_VSS;
	config.p_ref    = ADC_POS_REF_VDD;
	config.adc_ch   = ADC_CHANNEL_0;
	config.dma_ch   = 0;
	config.p_timer  = GP16C4T0;
	config.p_adc    = ADC0;
	config.cplt_cbk = complete_cbk;

	return;
}

/**
  * @brief  Get the average value of the normal channel.
  * @retval None
  */
void get_normal_average_voltage(void)
{
	/* Start ADC */ 
	ald_adc_timer_trigger_adc_by_dma(&config);
	return;
}

/**
  * @brief  Hnadle result data.
  * @param  buf: Pointer to the result pool
  * @param  size: Number of the data
  * @retval Average voltage.
  */
uint32_t handle_result(uint16_t *buf, uint16_t size)
{
	uint32_t i, min = 0xFFFF, max = 0, vol, sum = 0;

	for (i = 0; i < USR_ADC_COUNT; ++i) {
		max  = buf[i] > max ? buf[i] : max;
		min  = buf[i] < min ? buf[i] : min;
		sum += buf[i];
	}

	sum = sum - max - min;
	vol = sum / (USR_ADC_COUNT - 2);
	vol = (vol * config.h_adc.vdd_value) >> 12;

	return vol;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	/* PCLK2 bus divide by 2 */ 
	ald_cmu_div_config(CMU_PCLK_2, CMU_DIV_2);
	/* Clear env_t structure */
	memset(&_env, 0x0, sizeof(env_t));

	/* Initialize stdio */
	uart_stdio_init();
	/* Initialize ADC pin */
	adc_pin_init();
	/* Initialize ADC */
	adc_init();

	printf_e("System start...\r\n");
	printf_e("Index Normal Insert\r\n");
	printf_e("--------------------------\r\n");
	get_normal_average_voltage();
	while (!_env.flag);
	_env.flag = 0;
	_env.value = handle_result(_env.buf, USR_ADC_COUNT);
	printf_e("%-5d %-6d\r\n", ++_env.cnt, _env.value);

	while (1) {
		ald_delay_ms(1000);
		get_normal_average_voltage();
		while (!_env.flag);
		_env.flag = 0;
		_env.value = handle_result(_env.buf, USR_ADC_COUNT);
		printf_e("%-5d %-6d\r\n", ++_env.cnt, _env.value);
	}
}

/**
  * @}
  */

/**
  * @}
  */
