/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    17 Apr 2017
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          17 Apr 2017     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */


adc_handle_t h_adc;
uart_handle_t h_uart;
char uart_buf[64];
channel_env_t ch_env[USR_ADC_CH_NR];
env_t _env;

void get_normal_average_voltage(void);

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = UART_FUNC_TX;
	ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = UART_FUNC_RX;
	ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

	/* Initialize uart */
	h_uart.perh             = UART0;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	return;
}

/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void normal_complete_cbk(adc_handle_t *arg)
{
	int i;

	ald_adc_stop_by_dma(&h_adc, 0);

	for (i = 0; i < USR_ADC_CH_NR; ++i) {
		ch_env[i].max  = _env.buf[i] > ch_env[i].max ? _env.buf[i] : ch_env[i].max;
		ch_env[i].min  = _env.buf[i] < ch_env[i].min ? _env.buf[i] : ch_env[i].min;
		ch_env[i].sum += _env.buf[i];
	}

	if (++_env.idx == USR_ADC_COUNT) {
		for (i = 0; i < USR_ADC_CH_NR; ++i) {
			ch_env[i].sum = ch_env[i].sum - ch_env[i].max - ch_env[i].min;
			ch_env[i].vol = ch_env[i].sum / (USR_ADC_COUNT - 2);
			ch_env[i].vol = (ch_env[i].vol * h_adc.vdd_value) >> 12;
		}

		_env.flag = 1;
		return;
	}

	get_normal_average_voltage();
	return;
}

/**
  * @brief  Adc insert convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void insert_complete_cbk(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc level out of window complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void wdg_cbk(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc error complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void error_cbk(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Adc over complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void ovr_cbk(adc_handle_t *arg)
{
	return;
}

/**
  * @brief  Initializate pin of adc module.
  * @retval None.
  */
void adc_pin_init(void)
{
	gpio_init_t x;

	memset(&x, 0, sizeof(gpio_init_t));

	/* Initialize ADC pin */
	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_UP;
	x.odrv = GPIO_OUT_DRIVE_NORMAL;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_0;

	ald_gpio_init(GPIO_CH0_PORT, ADC_CH0_PIN, &x);
	ald_gpio_init(GPIO_CH2_PORT, ADC_CH2_PIN, &x);
	ald_gpio_init(GPIO_CH3_PORT, ADC_CH3_PIN, &x);
	ald_gpio_init(GPIO_CH9_PORT, ADC_CH9_PIN, &x);
	ald_gpio_init(GPIO_CH10_PORT, ADC_CH10_PIN, &x);

	return;
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
	adc_nch_conf_t config;

	/* clear structures */
	memset(&h_adc, 0x0, sizeof(adc_handle_t));
	memset(&config, 0x0, sizeof(adc_nch_conf_t));

	/* Initialize adc */
	h_adc.perh            = ADC0;
	h_adc.init.align      = ADC_DATAALIGN_RIGHT;
	h_adc.init.scan       = ENABLE;
	h_adc.init.cont       = ENABLE;
	h_adc.init.nch_nr     = ADC_NCH_NR_5;
	h_adc.init.disc       = ADC_ALL_DISABLE;
	h_adc.init.disc_nr    = ADC_DISC_NR_1;
	h_adc.init.data_bit   = ADC_CONV_BIT_12;
	h_adc.init.div        = ADC_CKDIV_128;
	h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ONE;
	h_adc.init.n_ref      = ADC_NEG_REF_VSS;
	h_adc.init.p_ref      = ADC_POS_REF_VDD;
	h_adc.normal_cplt_cbk = normal_complete_cbk;
	h_adc.insert_cplt_cbk = insert_complete_cbk;
	h_adc.wdg_cbk         = wdg_cbk;
	h_adc.error_cbk       = error_cbk;
	h_adc.ovr_cbk         = ovr_cbk;
	ald_adc_init(&h_adc);

	/* Initialize normal convert channel */
	config.ch   = ADC_CHANNEL_0;
	config.idx  = ADC_NCH_IDX_1;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_2;
	config.idx  = ADC_NCH_IDX_2;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_3;
	config.idx  = ADC_NCH_IDX_3;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_9;
	config.idx  = ADC_NCH_IDX_4;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	config.ch   = ADC_CHANNEL_10;
	config.idx  = ADC_NCH_IDX_5;
	config.samp = ADC_SAMPLETIME_4;
 	ald_adc_normal_channel_config(&h_adc, &config);

	return;
}

/**
  * @brief  Get the average value of the normal channel.
  * @retval None
  */
void get_normal_average_voltage(void)
{
	/* Get VDD value */
	ald_adc_get_vdd_value(&h_adc);
	/* Start normal convert */
	ald_adc_start_by_dma(&h_adc, _env.buf, USR_ADC_CH_NR, 0);
	return;
}

/**
  * @brief  Initialize channel environment
  * @retval None.
  */
void channel_env_init(void)
{
	int i;

	for (i = 0; i < USR_ADC_CH_NR; ++i) {
		ch_env[i].max = 0x0;
		ch_env[i].min = 0xFFF;
		ch_env[i].vol = 0x0;
		ch_env[i].sum = 0x0;
	}

	return;
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	/* PCLK2 bus divide by 2 */ 
	ald_cmu_div_config(CMU_PCLK_2, CMU_DIV_2);
	/* Clear env_t structure */
	memset(&_env, 0x0, sizeof(env_t));

	/* Initialize stdio */
	uart_stdio_init();
	/* Initialize ADC pin */
	adc_pin_init();
	/* Initialize ADC */
	adc_init();
	/* Initialize channel environment */
	channel_env_init();

	printf_e("System start...\r\n");
	printf_e("Index CH0    CH2    CH3    CH9    CH10\r\n");
	printf_e("----------------------------------------------\r\n");
	get_normal_average_voltage();
	while (!_env.flag);
	_env.flag = 0;
	_env.idx  = 0;
	printf_e("%-5d %-6d %-6d %-6d %-6d %-6d\r\n", ++_env.cnt, ch_env[0].vol,
	                 ch_env[1].vol, ch_env[2].vol, ch_env[3].vol, ch_env[4].vol);
	channel_env_init();

	while (1) {
		ald_delay_ms(1000);
		get_normal_average_voltage();
		while (!_env.flag);
		_env.flag = 0;
		_env.idx  = 0;
		printf_e("%-5d %-6d %-6d %-6d %-6d %-6d\r\n", ++_env.cnt, ch_env[0].vol,
				ch_env[1].vol, ch_env[2].vol, ch_env[3].vol, ch_env[4].vol);
		channel_env_init();
	}
}

/**
  * @}
  */

/**
  * @}
  */
