/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov  2021    Ginger          the first version
 *          28 Dec  2021    Ginger          Modify the example for PDS board
 *          24 Mar  2022    AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

#ifdef LEFT_BUTTON_OPENTIMER
md_gpio_inittypedef GPIOB_PIN1_Init =    /**< GPIOB Pin1(Button Left) GP16C2T1_CH2 init structure  */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};
#endif


md_gpio_inittypedef GPIOA_PIN2_Init =    /**< GP16C2T1_CH1 init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF5
};


md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    bsp_led_init();
    bsp_key_init();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    TimerInit();
    __enable_irq();


    printf("Press the left button to send a pulse\r\n");

    while (1)
    {
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_gp16c2t1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PA2: PWM GP16C2T1_CH1.
  *         PB1: input GP16C2T1_CH2(left buttom).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN2_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);
}


/**
  * @brief  Timer Init
  * @note   None
  * @param  None
  * @retval None
  */
void TimerInit(void)
{
    md_timer_set_auto_reload_value_arrv(GP16C2T1, 2400);
    md_timer_set_output_compare1_mode_ch1mod(GP16C2T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_cc1_output_polarity_cc1pol(GP16C2T1, MD_TIMER_OUTPUTPOLARITY_LOW);
    md_timer_set_cc2_func_cc2ssel(GP16C2T1, MD_TIMER_CHMODE_INPUT_DIRECT);
    md_timer_set_cc2_input_edge_cc2pol(GP16C2T1, MD_TIMER_OUTPUTPOLARITY_LOW);
    md_timer_set_slave_trigger_tssel(GP16C2T1, MD_TIMER_TRIGGER_I2F);
    md_timer_set_slave_mode_smods(GP16C2T1, MD_TIMER_SLAVEMODE_TRIGGER);
    md_timer_enable_onepulse_spmen(GP16C2T1);

    md_timer_enable_cc1_output_cc1en(GP16C2T1);
    md_timer_enable_cc2_output_cc2en(GP16C2T1);

    md_timer_enable_main_output_goen(GP16C2T1);

    md_timer_set_capture_compare1_value_ccrv1(GP16C2T1, 240);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


