/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    3 Mar 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          3 Mar   2022    Ginger          the first version
 *          24 Mar  2022    AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOA_PIN8_Init =    /**< AD16C4T1_CH1 init structure */
{
    MD_GPIO_PIN_8,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOA_PIN9_Init =    /**< AD16C4T1_CH2 init structure */
{
    MD_GPIO_PIN_9,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOA_PIN10_Init =    /**< AD16C4T1_CH3 init structure */
{
    MD_GPIO_PIN_10,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOB_PIN15_Init =    /**< AD16C4T1_CH3N init structure */
{
    MD_GPIO_PIN_15,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOB_PIN14_Init =    /**< AD16C4T1_CH2N init structure */
{
    MD_GPIO_PIN_14,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOB_PIN13_Init =    /**< AD16C4T1_CH1N init structure */
{
    MD_GPIO_PIN_13,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOA_PIN15_Init =    /**< GP32C4T1_CH1 init structure */
{
    MD_GPIO_PIN_15,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF4
};

md_gpio_inittypedef GPIOA_PIN0_Init =    /**< gpio init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_adc_initial adc_initStruct =    /**< ADC init structure */
{
    MD_ADC_CFG_ALIGN_RIGHT,    //Data alignment
    MD_ADC_CFG_RSEL_12BIT,     //Data resolution
    MD_ADC_MODE_NCHS,          //Regular or Injected
    MD_ADC_CFG_CM_SINGLE,      //Single mode
    MD_ADC_NCHS1_NSL_1CON,     //sample count
    MD_ADC_SMPT1_CKDIV_DIV6,   //ADC prescale
};

/* Private define -------------------------------------------------------------*/
//INJECT 0:close 1:open
#define INJECT 1
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
volatile uint8_t ADC_DONE = 0;
volatile uint32_t NSE, NCHE = 0;

extern uint32_t ADC_interrupt;

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint16_t adc_data;
    uint32_t sample_seq_channel[] =
    {
        MD_ADC_NCHS1_NS1_CH15
    };

    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    TimerInit();
    bsp_led_init();
    bsp_key_init();
    md_adc_calibration(ADC, &adc_initStruct);
    md_adc_set_sampletime_channel_13(ADC, 0xFF);

    md_adc_init(ADC, &adc_initStruct);

    __enable_irq();

    printf("PWM high side PIN: PA8 PA9 PA10\r\n");
    printf("PWM low  side PIN: PB13 PB14 PB15\r\n");
    md_tick_waitms(10, 50);

    while ((ADC->RIF & 0x1) == 0);

    md_adc_sequence_conversion(ADC, &adc_initStruct, sample_seq_channel);

#if INJECT==1
    //setting inject channel
    adc_initStruct.Regular_Injected = MD_ADC_MODE_ICHS;
    md_adc_sequence_conversion(ADC, &adc_initStruct, sample_seq_channel);
#endif

    printf("Use GP32C4T1_TRGOUT Trigger ADC\r\n");
    md_tick_waitms(10, 50);
    md_adc_set_normal_external_trigger_mode(ADC, MD_ADC_CFG_NETS_RISING);
    md_adc_set_normal_external_trigger_source(ADC, MD_ADC_CFG_NEXTSEL_GP32C4T1_TRGOUT);

#if INJECT==1
    md_adc_set_inj_external_trigger_mode(ADC, MD_ADC_CFG_NETS_RISING);
    md_adc_set_inj_external_trigger_source(ADC, MD_ADC_ICHS_IEXTSEL_GP32C4T1_TRGOUT);
    md_adc_set_start_inj(ADC, MD_ADC_CON_ISTART_START_INJECTED);
#endif
    md_adc_set_start_normal(ADC, MD_ADC_CON_NSTART_START_REGULAR);




    while (1)
    {
        md_tick_waitms(10, 50);

//    while(md_adc_is_active_flag_normal_status(ADC));
        if (md_adc_is_active_flag_inj_conversion_done(ADC))
        {
            adc_data = md_adc_get_inj_data_1(ADC);
            printf("inject_conversion_adc_data %d\r\n", adc_data);
        }

        if (md_adc_is_active_flag_normal_conversion_done(ADC))
        {
            adc_data = md_adc_get_normal_data(ADC);
            printf("normal_conversion_adc_data %d\r\n", adc_data);
        }

    }

}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_ad16c4t1(RCU);
    md_rcu_enable_gp32c4t1(RCU);
    md_rcu_enable_adc(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PA8: PWM AD16C4T1_CH1.
  *         PA9: PWM AD16C4T1_CH2.
  *         PA10: PWM AD16C4T1_CH3.
  *         PB13: PWM AD16C4T1_CH1N.
  *         PB14: PWM AD16C4T1_CH2N.
  *         PB15: PWM AD16C4T1_CH3N.
  *         PA15: PWM GP32C4T1_CH1. Use trigger ADC
  *         PA3 : ADC sample pin
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN8_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN9_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN10_Init);

    md_gpio_init(GPIOB, &GPIOB_PIN13_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN14_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN15_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN15_Init);
}

/**
  * @brief  Timer Init
  * @note   None
  * @param  None
  * @retval None
  */
void TimerInit(void)
{
    md_timer_set_auto_reload_value_arrv(AD16C4T1, 2400);
    md_timer_set_counter_aligned_mode_cmsel(AD16C4T1, MD_TIMER_CENTERALIGNED_BOTH);
    md_timer_set_master_trgout_select_mmsel(AD16C4T1, MD_TIMER_MASTERMODE_ENABLE);
    md_timer_enable_master_mode_mscfg(AD16C4T1);
    md_timer_set_output_compare1_mode_ch1mod(AD16C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare2_mode_ch2mod(AD16C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare3_mode_ch3mod(AD16C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare1_value_ccrv1(AD16C4T1, (md_timer_get_auto_reload_value_arrv(AD16C4T1)) >> 1);
    md_timer_set_capture_compare2_value_ccrv2(AD16C4T1, (md_timer_get_auto_reload_value_arrv(AD16C4T1)) >> 2);
    md_timer_set_capture_compare3_value_ccrv3(AD16C4T1, (md_timer_get_auto_reload_value_arrv(AD16C4T1)) >> 3);

    md_timer_enable_cc1_output_cc1en(AD16C4T1);
    md_timer_enable_cc2_output_cc2en(AD16C4T1);
    md_timer_enable_cc3_output_cc3en(AD16C4T1);
    md_timer_enable_cc1_complementary_cc1nen(AD16C4T1);
    md_timer_enable_cc2_complementary_cc2nen(AD16C4T1);
    md_timer_enable_cc3_complementary_cc3nen(AD16C4T1);
    md_timer_enable_main_output_goen(AD16C4T1);

    md_timer_set_auto_reload_value_arrv(GP32C4T1, 2400);
    md_timer_set_counter_aligned_mode_cmsel(GP32C4T1, MD_TIMER_CENTERALIGNED_BOTH);
    md_timer_set_output_compare1_mode_ch1mod(GP32C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare1_value_ccrv1(GP32C4T1, 1200);
    md_timer_enable_cc1_output_cc1en(GP32C4T1);
    md_timer_set_slave_trigger_tssel(GP32C4T1, MD_TIMER_TRIGGER_IT0);
    md_timer_set_slave_mode_smods(GP32C4T1, MD_TIMER_SLAVEMODE_TRIGGER);
    md_timer_set_master_trgout_select_mmsel(GP32C4T1, MD_TIMER_MASTERMODE_COMPARE1);
    md_timer_enable_counter_cnten(AD16C4T1);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


