/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov  2021    Ginger          the first version
 *          28 Dec  2021    Ginger          Modify the example for PDS board
 *          24 Mar  2022    AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOA_PIN9_Init =    /**< AD16C4T1_CH2 init structure */
{
    MD_GPIO_PIN_9,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/

/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
uint16_t count_box[COUNT_BOX_LEN] = {120, 240, 480, 720, 960, 1200, 1440, 1680, 1920, 2160, 2280};
md_dma_init_typedef dma_initStruct =
{
    MD_DMA_PRIMARY,
    MD_DMA_CHANNEL0,
    MD_DMA_CHANNEL_CFG_MODE_BASIC,
    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1,
    MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD,
    MD_DMA_CHANNEL_CFG_SRCINC_HALF_WORD,
    MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD,
    MD_DMA_CHANNEL_CFG_DSTINC_NO_INC,
    COUNT_BOX_LEN
};
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);
void DMAInit(void);
/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    bool flag = 0;
    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    bsp_led_init();
    bsp_key_init();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    TimerInit();
    __enable_irq();


    printf("After pressing any key, use DMA to set the DUTYCYCLE of CH2(PA9)\r\n");
    md_timer_set_capture_compare2_value_ccrv2(AD16C4T1, 0);

    while (1)
    {
        if (md_uart_is_active_flag_rfnempty(UART1) && flag == 0)
        {
            md_uart_receive(UART1);
            DMAInit();
            flag = 1;
            printf("dma send\r\n");
        }

        if (md_dma_is_active_flag_channel_done(DMA1, MD_DMA_CHANNEL0))
        {
            md_timer_clear_it_upd(AD16C4T1);

            while (md_timer_is_active_it_upd(AD16C4T1) == 0);

            flag = 0;
            md_timer_set_capture_compare2_value_ccrv2(AD16C4T1, 0);
            md_dma_clear_it_channel_done(DMA1, MD_DMA_CHANNEL0);
            printf("dma done\r\n");
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_ad16c4t1(RCU);
    md_rcu_enable_dma1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PA9: PWM AD16C4T1_CH2.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN9_Init);

}


/**
  * @brief  Timer Init
  * @note   None
  * @param  None
  * @retval None
  */
void TimerInit(void)
{
    md_timer_set_auto_reload_value_arrv(AD16C4T1, 2400);
    md_timer_set_output_compare2_mode_ch2mod(AD16C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare2_value_ccrv2(AD16C4T1, (md_timer_get_auto_reload_value_arrv(AD16C4T1)) >> 2);
    md_timer_enable_cc2_output_cc2en(AD16C4T1);
    md_timer_enable_main_output_goen(AD16C4T1);


    md_timer_enable_output_compare2_preload_ch2pen(AD16C4T1);
    md_timer_enable_dma_upd(AD16C4T1);
    md_timer_enable_counter_cnten(AD16C4T1);
}

/**
  * @brief  DMA init
  * @param  count_box index
  * @retval None
  */

void DMAInit()
{
    md_dma_set_configuration(DMA1, ENABLE);

    dma_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(count_box), COUNT_BOX_LEN, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    dma_initStruct.destination_data_end_address = (uint32_t)(&(AD16C4T1->CCVAL2));

    md_dma_init(DMA1, dma_initStruct);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL0, MD_DMA_PRS_AD16C4T1_UP);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


