/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    13 Jan 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          13 Jan  2022    AE Team         the first version
 *          24 Mar  2022    AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

#ifdef PWM_Control_LED2
md_gpio_inittypedef GPIOC_PIN1_Init =    /**< GPIOC Pin1(LED Red )& GP16C4T3_CH3 init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};
#endif


#ifdef PWM_Control_LED2
md_gpio_inittypedef GPIOC_PIN3_Init =    /**< GPIOC Pin3(LED Red) & GP16C4T3_ETR init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF3
};
#endif

md_gpio_inittypedef GPIOD_PIN3_Init =    /**< AD16C4T1_CH1 init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOA_PIN9_Init =    /**< AD16C4T1_CH2 init structure */
{
    MD_GPIO_PIN_9,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOA_PIN10_Init =    /**< AD16C4T1_CH3 init structure */
{
    MD_GPIO_PIN_10,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOB_PIN3_Init =    /**< AD16C4T1_CH3N init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOB_PIN4_Init =    /**< AD16C4T1_CH2N init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_gpio_inittypedef GPIOB_PIN5_Init =    /**< AD16C4T1_CH1N init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF6
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
#ifdef PWM_Control_LED2
    //disable bsp_Led2 function
    #define bsp_Led2_On()   (void)(0)
    #define bsp_Led2_Off()  (void)(0)
#endif
#ifdef PWM_Control_LED2
    //disable bsp_Led4 function
    #define bsp_Led4_On()   (void)(0)
    #define bsp_Led4_Off()  (void)(0)
#endif
void TimerInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    int32_t count = 0;
    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    bsp_led_init();
    bsp_key_init();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    TimerInit();
    __enable_irq();

    printf("PWM high side PIN: PD3 PA9 PA10\r\n");
    printf("PWM low  side PIN: PB5 PB4 PB3\r\n");
    md_tick_waitms(10, 500);

#ifdef PWM_Control_LED2
    printf("\nAdjust different PWM duty to control Red LED(PC1) brightness\r\n");
    printf("After inputting 0V voltage to PC3, you can turn off the LED flashing and make the LED continue to light up\r\n");
#endif

    while (1)
    {
        for (count = 0 ; count <= 2400 ; count++)
        {
            md_timer_set_capture_compare1_value_ccrv1(AD16C4T1, count);
            md_timer_set_capture_compare3_value_ccrv3(GP16C4T3, count);
            md_tick_waitms(1, 1);
        }

        md_tick_waitms(1, 10);

        for (count = 2400 ; count >= 0 ; count--)
        {
            md_timer_set_capture_compare1_value_ccrv1(AD16C4T1, count);
            md_timer_set_capture_compare3_value_ccrv3(GP16C4T3, count);
            md_tick_waitms(1, 1);
        }

        md_tick_waitms(1, 10);
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_ad16c4t1(RCU);
#ifdef PWM_Control_LED2
    md_rcu_enable_gp16c4t3(RCU);
#endif
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PD3: PWM AD16C4T1_CH1(LED RED).
  *         PA9: PWM AD16C4T1_CH2.
  *         PA10: PWM AD16C4T1_CH3.
  *         PB5: PWM AD16C4T1_CH1N.
  *         PB4: PWM AD16C4T1_CH2N.
  *         PB3: PWM AD16C4T1_CH3N.
  *         PC1: GPIO LED RED & PWM GP16C4T3_CH3.
  *         PC0: GPIO LED RED.
  *         PC3: GPIO LED RED & PWM GP16C4T3_ETR.
  *         PC2: GPIO LED RED.
  *         PB1: GPIO Button Mid.
  *         PC10: GPIO Button Left.
  *         PC13: GPIO Button Up.
  *         PD0: GPIO Button Right.
  *         PD1: GPIO Button Down.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOD, &GPIOD_PIN3_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN9_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN10_Init);

    md_gpio_init(GPIOB, &GPIOB_PIN3_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN4_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN5_Init);

#ifdef PWM_Control_LED2
    md_gpio_init(GPIOC, &GPIOC_PIN1_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN3_Init);
#endif
}

/**
  * @brief  Timer Init
  * @note   None
  * @param  None
  * @retval None
  */
void TimerInit(void)
{
    md_timer_set_auto_reload_value_arrv(AD16C4T1, 2400);
#ifdef PWM_Control_LED2
    md_timer_set_master_trgout_select_mmsel(AD16C4T1, MD_TIMER_MASTERMODE_ENABLE);
    md_timer_enable_master_mode_mscfg(AD16C4T1);
#endif
    md_timer_set_output_compare1_mode_ch1mod(AD16C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare2_mode_ch2mod(AD16C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare3_mode_ch3mod(AD16C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_output_compare4_mode_ch4mod(AD16C4T1, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare1_value_ccrv1(AD16C4T1, (md_timer_get_auto_reload_value_arrv(AD16C4T1)) >> 1);
    md_timer_set_capture_compare2_value_ccrv2(AD16C4T1, (md_timer_get_auto_reload_value_arrv(AD16C4T1)) >> 2);
    md_timer_set_capture_compare3_value_ccrv3(AD16C4T1, (md_timer_get_auto_reload_value_arrv(AD16C4T1)) >> 3);
    md_timer_set_capture_compare4_value_ccrv4(AD16C4T1, (md_timer_get_auto_reload_value_arrv(AD16C4T1)) >> 4);

//  md_timer_set_external_trigger_select(AD16C4T1, MD_TIMER_ETRRMP_ETR);
//  md_timer_set_external_trigger_polarity_etpol(AD16C4T1, MD_TIMER_ETRPOLARITY_NONINVERTED);
    md_timer_enable_output_compare1_clear_ch1oclren(AD16C4T1);
    md_timer_disable_output_compare2_clear_ch2oclren(AD16C4T1);
    md_timer_enable_output_compare3_clear_ch3oclren(AD16C4T1);
    md_timer_disable_output_compare4_clear_ch4oclren(AD16C4T1);

    md_timer_enable_cc1_output_cc1en(AD16C4T1);
    md_timer_enable_cc2_output_cc2en(AD16C4T1);
    md_timer_enable_cc3_output_cc3en(AD16C4T1);
    md_timer_enable_cc4_output_cc4en(AD16C4T1);
    md_timer_enable_cc1_complementary_cc1nen(AD16C4T1);
    md_timer_enable_cc2_complementary_cc2nen(AD16C4T1);
    md_timer_enable_cc3_complementary_cc3nen(AD16C4T1);
    md_timer_enable_main_output_goen(AD16C4T1);

#ifdef PWM_Control_LED2
    md_timer_set_auto_reload_value_arrv(GP16C4T3, 2400);
    md_timer_set_output_compare3_mode_ch3mod(GP16C4T3, MD_TIMER_OUTPUTMODE_PWMMODE1);
    md_timer_set_capture_compare3_value_ccrv3(GP16C4T3, 1200);
    md_timer_enable_cc3_output_cc3en(GP16C4T3);
    md_timer_set_slave_trigger_tssel(GP16C4T3, MD_TIMER_TRIGGER_IT0);
    md_timer_set_slave_mode_smods(GP16C4T3, MD_TIMER_SLAVEMODE_TRIGGER);
    md_timer_set_external_trigger_select(GP16C4T3, MD_TIMER_ETRRMP_ETR);
    md_timer_set_external_trigger_polarity_etpol(GP16C4T3, MD_TIMER_ETRPOLARITY_INVERTED);
    md_timer_enable_output_compare3_clear_ch3oclren(GP16C4T3);
#endif


    md_timer_enable_counter_cnten(AD16C4T1);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


