/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          30 Dec 2021     Ginger          Modify the example for PDS board and function name
 *          25 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOA_PIN4_Init =    /**< SPI1_NSS/I2S1_WS init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN5_Init =    /**< SPI1_SCK/I2S1_CK init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN6_Init =    /**< SPI1_MISO/I2S1_SDI init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN7_Init =    /**< SPI1_MOSI/I2S1_SDO init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_spi_inittypedef SPI1_InitStruct =    /**< SPI init structure */
{
    MD_SPI_MODE_MASTER,
    MD_SPI_PHASE_1EDGE,
    MD_SPI_POLARITY_LOW,
    MD_SPI_BAUDRATEPRESCALER_DIV16,
    MD_SPI_MSB_FIRST,
    MD_SPI_FULL_DUPLEX,
    MD_SPI_FRAME_FORMAT_8BIT,
    MD_SPI_NSS_HARD,
};

/* Private define -------------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/

/**
  * @brief  main.
  * @param  None
  * @retval None
  * @note   Please connect your own MISO and MOSI together to use
  */
int main(void)
{
    uint8_t data[32];
    int i, j = 0;

    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();
    md_spi_init(SPI1, &SPI1_InitStruct);

    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);

    while (1)
    {
        printf("Press to start master Tx\r\n");

        while (!md_uart_is_active_flag_rfnempty(UART1));

        *data = md_uart_get_recv_data8(UART1);

        for (i = 0; i < 5; i++)
        {
            md_spi_send_byte(SPI1, 0x00 + j);
            data[i] = md_spi_recv_byte(SPI1);
        }

        for (i = 0; i < 5; i++)
            printf("data[%d] = %x\r\n", i, data[i]);

        j++;
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_dma1(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_spi1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PA4: SPI1_NSS/I2S1_WS.
  *         PA5: SPI1_SCK/I2S1_CK.
  *         PA6: SPI1_MISO/I2S1_SDI.
  *         PA7: SPI1_MOSI/I2S1_SDO.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN4_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN5_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN6_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN7_Init);
}


/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


