/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          30 Dec 2021     Ginger          Modify the example for PDS board
 *          28 Mar 2022     AE Team         Modify MD Driver
 *          22 Apr 2022     AE Team         Modify SPI Setting
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};


md_gpio_inittypedef GPIOA_PIN4_Init =    /**< SPI1_NSS/I2S1_WS init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN5_Init =    /**< SPI1_SCK/I2S1_CK init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN6_Init =    /**< SPI1_MISO/I2S1_SDI init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN7_Init =    /**< SPI1_MOSI/I2S1_SDO init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_spi_inittypedef SPI1_InitStruct =    /**< SPI init structure */
{
    MD_SPI_MODE_MASTER,
    MD_SPI_PHASE_1EDGE,
    MD_SPI_POLARITY_LOW,
    MD_SPI_BAUDRATEPRESCALER_DIV8,
    MD_SPI_MSB_FIRST,
    MD_SPI_FULL_DUPLEX,
    MD_SPI_FRAME_FORMAT_8BIT,
    MD_SPI_NSS_HARD,
};
#define BUFFER_SIZE  16

md_dma_init_typedef dma_initStruct[2] =
{
    {
        MD_DMA_PRIMARY,
        MD_DMA_CHANNEL0,
        MD_DMA_CHANNEL_CFG_MODE_BASIC,
        MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1,
        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_BYTE,
        MD_DMA_CHANNEL_CFG_SRCINC_BYTE,
        MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_BYTE,
        MD_DMA_CHANNEL_CFG_DSTINC_NO_INC,
        BUFFER_SIZE
    },
    {
        MD_DMA_PRIMARY,
        MD_DMA_CHANNEL1,
        MD_DMA_CHANNEL_CFG_MODE_BASIC,
        MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1,
        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_BYTE,
        MD_DMA_CHANNEL_CFG_SRCINC_NO_INC,
        MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_BYTE,
        MD_DMA_CHANNEL_CFG_DSTINC_BYTE,
        BUFFER_SIZE
    }
};

/* Private define -------------------------------------------------------------*/


/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
__IO uint32_t DMA_INTR0, DMA_INTR12;

/* Tx data buffer */
uint8_t SPI_TxBuf[BUFFER_SIZE] = {0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10};

/* Rx data buffer */
uint8_t SPI_RxBuf[BUFFER_SIZE] = {0};

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/


void md_spi_send_recv_by_dma()
{
    md_dma_set_configuration(DMA1, ENABLE);

    dma_initStruct[0].source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(SPI_TxBuf), BUFFER_SIZE, MD_DMA_CHANNEL_SIZE_BYTE);
    dma_initStruct[0].destination_data_end_address = (uint32_t)&SPI1->DATA;
    md_dma_init(DMA1, dma_initStruct[0]);

    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL0);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL0, MD_DMA_PRS_SPI1_TX);

    dma_initStruct[1].source_data_end_address = (uint32_t)&SPI1->DATA;
    dma_initStruct[1].destination_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(SPI_RxBuf), BUFFER_SIZE, MD_DMA_CHANNEL_SIZE_BYTE);
    md_dma_init(DMA1, dma_initStruct[1]);

    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL1);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL1, MD_DMA_PRS_SPI1_RX);

    md_spi_enable_txdma(SPI1);
    md_spi_enable_rxdma(SPI1);

    while (DMA_INTR0 == 0);

    while (DMA_INTR12 == 0);
}

void irq_set()
{
    NVIC_SetPriority((IRQn_Type) SPI1_IRQn,  0);      //priority 0
    NVIC_SetPriority((IRQn_Type) DMA1_CH0_IRQn,  0);  //priority 1
    NVIC_SetPriority((IRQn_Type) DMA1_CH12_IRQn,  0);  //priority 1
    NVIC_EnableIRQ((IRQn_Type) SPI1_IRQn);            // SPI1_IRQn interrupt
    NVIC_EnableIRQ((IRQn_Type) DMA1_CH0_IRQn);        // DMA1_CH0_IRQn interrupt
    NVIC_EnableIRQ((IRQn_Type) DMA1_CH12_IRQn);       // DMA1_CH12_IRQn interrupt
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    int i;

    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    md_spi_init(SPI1, &SPI1_InitStruct);
    irq_set();
    __enable_irq();

    DMA_INTR0 = 0;
    DMA_INTR12 = 0;

    printf("SPI_Send_Recv_By_DMA Demo\r\n\r\n");

    for (i = 0; i < BUFFER_SIZE; i++)
    {
        printf("SPI_TxBuf[%d]:0x%02x\r\n", i, SPI_TxBuf[i]);
    }

    printf("---------------------------\r\n");
    md_spi_send_recv_by_dma();
    printf("sending ...\r\n");
    printf("---------------------------\r\n");

    for (i = 0; i < BUFFER_SIZE; i++)
        printf("SPI_RxBuf[%d]:0x%02X\r\n", i, SPI_RxBuf[i]);

    while (1);
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_dma1(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_spi1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PA4: SPI1_NSS/I2S1_WS.
  *         PA5: SPI1_SCK/I2S1_CK.
  *         PA6: SPI1_MISO/I2S1_SDI.
  *         PA7: SPI1_MOSI/I2S1_SDO.

  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN4_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN5_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN6_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN7_Init);
}



/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


