/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          25 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOA_PIN4_Init =    /**< SPI1_NSS/I2S1_WS init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

//md_gpio_inittypedef GPIOA_PIN4_Init =    /**< SPI1_NSS/I2S1_WS init structure */
//{
//  MD_GPIO_PIN_4,
//  MD_GPIO_MODE_FUNCTION,
//  MD_GPIO_OUTPUT_PUSHPULL,
//  MD_GPIO_PULL_UP,
//  MD_GPIO_DRIVING_8MA,
//  MD_GPIO_AF0
//};

md_gpio_inittypedef GPIOA_PIN5_Init =    /**< SPI1_SCK/I2S1_CK init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN6_Init =    /**< SPI1_MISO/I2S1_SDI init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN7_Init =    /**< SPI1_MOSI/I2S1_SDO init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_spi_inittypedef SPI1_InitStruct =    /**< SPI init structure */
{
    MD_SPI_MODE_MASTER,
    MD_SPI_PHASE_1EDGE,
    MD_SPI_POLARITY_LOW,
    MD_SPI_BAUDRATEPRESCALER_DIV2,
    MD_SPI_MSB_FIRST,
    MD_SPI_FULL_DUPLEX,
    MD_SPI_FRAME_FORMAT_8BIT,
    MD_SPI_NSS_HARD,
};

/* Private define -------------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
#define SPI4KERASE  0x20    //4K block erase
#define SPIERASE    0x60    //chip erase
#define SPIPAGEWR   0x2     //page write
#define SPIPAGERD   0xb     //page read
#define SPIWEN      0x6     //write enable
#define SPIWDIS     0x4     //write disable
#define SPISTSRD    0x5     //read status
#define SPISTSWR    0x1     //write status

/* Private variables ----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
/* Private functions ----------------------------------------------------------*/

void ClearRxFIFO(SPI_TypeDef *SPIx)
{
    uint8_t result;
    uint8_t dummy;

    UNUSED(dummy);
    result = 1;

    while (result)
    {
        dummy = md_spi_get_data_reg(SPIx);
        result = md_spi_is_active_flag_rxne(SPIx);
    }
}

void flash_write_status(SPI_TypeDef *SPIx, uint8_t status)
{
    uint8_t dummy;

    UNUSED(dummy);
    md_spi_set_data_reg(SPIx, SPISTSWR);
    md_spi_set_data_reg(SPIx, status);

    while (!md_spi_is_active_flag_busy(SPIx));

    while (md_spi_is_active_flag_busy(SPIx));

    dummy = md_spi_get_data_reg(SPIx);
    dummy = md_spi_get_data_reg(SPIx);
}

uint8_t flash_read_status(SPI_TypeDef *SPIx)
{
    uint8_t status;
    uint8_t dummy;

    UNUSED(dummy);
    md_gpio_set_pin_low(GPIOA, MD_GPIO_PIN_4);
    md_spi_set_data_reg(SPIx, SPISTSRD);
    md_spi_set_data_reg(SPIx, 0xFF);

    while (!md_spi_is_active_flag_busy(SPIx));

    while (md_spi_is_active_flag_busy(SPIx));

    dummy   = md_spi_get_data_reg(SPIx);
    status  = md_spi_get_data_reg(SPIx);
    md_gpio_set_pin_high(GPIOA, MD_GPIO_PIN_4);
    return status;
}

void flash_wait_wren_latch(SPI_TypeDef *SPIx)
{
    while (!(flash_read_status(SPIx) & 0x2));
}

void flash_write_enable(SPI_TypeDef *SPIx)
{
    uint8_t dummy;

    UNUSED(dummy);
    md_gpio_set_pin_low(GPIOA, MD_GPIO_PIN_4);
    md_spi_set_data_reg(SPIx, SPIWEN);

    while (md_spi_is_active_flag_busy(SPIx));

    dummy = md_spi_get_data_reg(SPIx);

    while (md_spi_is_active_flag_rxne(SPIx));

    md_gpio_set_pin_high(GPIOA, MD_GPIO_PIN_4);
    flash_wait_wren_latch(SPIx);
}

void flash_wait_ready(SPI_TypeDef *SPIx)
{
    while ((flash_read_status(SPIx) & 0x1));
}

bool flash_check_error(SPI_TypeDef *SPIx)
{
    if ((flash_read_status(SPIx) & 0x20) != 0)
        return TRUE;
    else
        return FALSE;
}

void flash_chip_erase(SPI_TypeDef *SPIx)
{
    uint8_t dummy;

    UNUSED(dummy);
    md_gpio_set_pin_low(GPIOA, MD_GPIO_PIN_4);
    md_spi_set_data_reg(SPIx, SPIERASE);

    while (!md_spi_is_active_flag_busy(SPIx));

    while (md_spi_is_active_flag_busy(SPIx));

    dummy = md_spi_get_data_reg(SPIx);
    md_gpio_set_pin_high(GPIOA, MD_GPIO_PIN_4);
    flash_wait_ready(SPIx);
}

void flash_byte_write(SPI_TypeDef *SPIx, uint8_t page, uint8_t offset, uint8_t data)
{
    uint8_t  dummy;
    uint8_t  spicmd[5];
    uint16_t ii;
    uint32_t spiaddr;

    UNUSED(dummy);

    spiaddr = ((page & 0xff) << 8) | offset;

    spicmd[0] = SPIPAGEWR;
    spicmd[1] = ((spiaddr >> 16) & 0xff);
    spicmd[2] = ((spiaddr >> 8) & 0xff);
    spicmd[3] = (spiaddr & 0xff);
    spicmd[4] = data;
    md_gpio_set_pin_low(GPIOA, MD_GPIO_PIN_4);

    for (ii = 0; ii < 5; ii++)
    {
        md_spi_set_data_reg(SPIx, spicmd[ii]);
    }

    while (!md_spi_is_active_flag_txe(SPIx));

    while (md_spi_is_active_flag_rxne(SPIx))
    {
        dummy = md_spi_get_data_reg(SPIx);
    }

    md_gpio_set_pin_high(GPIOA, MD_GPIO_PIN_4);
    flash_wait_ready(SPIx);
}

uint8_t flash_byte_read(SPI_TypeDef *SPIx, uint8_t page, uint8_t offset)
{
    uint8_t  dummy, data;
    uint8_t  spicmd[6];
    uint16_t ii;
    uint32_t spiaddr;

    UNUSED(dummy);
    spiaddr = ((page & 0xff) << 8) | offset; // for read command
    spicmd[0] = SPIPAGERD;  //page read
    spicmd[1] = ((spiaddr >> 16) & 0xff);
    spicmd[2] = ((spiaddr >> 8) & 0xff);
    spicmd[3] = (spiaddr & 0xff);
    spicmd[4] = 0;          //don't care byte
    spicmd[5] = 0;          //don't care byte
    md_gpio_set_pin_low(GPIOA, MD_GPIO_PIN_4);

    for (ii = 0; ii < 6; ii++)
    {
        md_spi_set_data_reg(SPIx, spicmd[ii]);
    }

    while (!md_spi_is_active_flag_txe(SPIx));

    while (md_spi_is_active_flag_rxne(SPIx))
    {
        data = md_spi_get_data_reg(SPIx);
    }

    md_gpio_set_pin_high(GPIOA, MD_GPIO_PIN_4);
    return data;
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t data;

    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();

    md_spi_init(SPI1, &SPI1_InitStruct);

    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);
    md_gpio_set_outputport(GPIOA, MD_GPIO_PIN_4);
    //Global Unprotect
    printf("flash unprotect\r\n");
    flash_write_enable(SPI1);
    flash_write_status(SPI1, 0x0);

    printf("flash chip erase\r\n");
    flash_write_enable(SPI1);
    flash_chip_erase(SPI1);

    if (flash_check_error(SPI1))
        printf("chip erase fail\r\n");
    else
        printf("chip erase success\r\n");

    printf("flash write\r\n");
    flash_write_enable(SPI1);
    flash_byte_write(SPI1, 0, 0, 0x5A);

    if (flash_check_error(SPI1))
        printf("flash write fail\r\n");
    else
        printf("flash write success\r\n");

    printf("flash read\r\n");
    data = flash_byte_read(SPI1, 0, 0);
    printf("data:%x \r\n", data);

    if (data == 0x5A)
        printf("flash read success\r\n");
    else
        printf("flash read fail\r\n");

    while (1);
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_dma1(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_spi1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PA4: SPI1_NSS/I2S1_WS.
  *         PA5: SPI1_SCK/I2S1_CK.
  *         PA6: SPI1_MISO/I2S1_SDI.
  *         PA7: SPI1_MOSI/I2S1_SDO.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN4_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN5_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN6_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN7_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));   // Tx FIFO empty

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
