/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          24 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
    RCU_LCON_LRCON
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
volatile extern uint8_t RTC_1HZ;

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions -----------------------------------------------------------*/

/**
  * @brief   RTC init
  * @param  None
  * @retval None
  */
void RTCInit(uint8_t week, uint8_t hour, uint8_t min, uint8_t sec)
{
    md_rtc_disable_rtc(RTC);

    md_rtc_set_scale(RTC, 255);
    md_rtc_set_prescale(RTC, 127);
    md_rtc_set_ref_clock(RTC, MD_RTC_CTRL_CKSEL_LRC);

    md_rtc_set_date_weekday(RTC, week);
    md_rtc_set_time_hour(RTC, hour);
    md_rtc_set_time_minute(RTC, min);
    md_rtc_set_time_second(RTC, sec);

    md_rtc_enable_it_freq1hz(RTC);
    md_rtc_enable_rtc(RTC);

}

/**
  * @brief  IrqInit
  * @param  None
  * @retval None
  */
void IrqInit(void)
{
    NVIC_EnableIRQ((IRQn_Type) RTC_IRQn);       // RTC_IRQn interrupt
}

/**
  * @brief  set irq priority
  * @param  None
  * @retval None
  */
void IrqPriority(void)
{
    NVIC_SetPriority((IRQn_Type) RTC_IRQn,  0);  //priority 1
}

uint16_t  key_scan()
{
    uint8_t   rdat;
    uint16_t  keyin;

    keyin   = 0;
    rdat    = 0;

    while (rdat != 0xD) //Enter key ASCII 0xD
    {
        rdat = md_uart_receive(UART1);

        if ((rdat >= '0') & (rdat <= '9'))
        {
            keyin   = (keyin * 10) + (rdat & 0xf);
            printf("%d\r", keyin);
        }
        else if (rdat != 0xD)
            return (255);
    }

    if (keyin > 255)
        return (255);
    else
        return ((keyin & 0xff));
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint16_t u16RtcTime;
    uint16_t Week, Hour, Min, Sec;
    uint8_t temp = 0;

    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    IrqPriority();
    IrqInit();
    __enable_irq();

//  printf("On %s at %s \r\n", __DATE__, __TIME__);
    printf("SystemCoreClock :%d\r\n", SystemCoreClock);

    while (1)
    {
        printf("Enter Week:\r\n");
        Week = key_scan();
        printf("Week=%d\r\n", Week);
        printf("Enter Hour:\r\n");
        Hour = key_scan();
        printf("Hour=%d\r\n", Hour);
        printf("Enter Min:\r\n");
        Min = key_scan();
        printf("Min=%d\r\n", Min);
        printf("Enter Sec:\r\n");
        Sec = key_scan();
        printf("Sec=%d\r\n", Sec);
        break;
    }

    Hour = (Hour / 10) * (16) + (Hour % 10) * 1;
    Min = (Min / 10) * (16) + (Min % 10) * 1;
    Sec = (Sec / 10) * (16) + (Sec % 10) * 1;

    RTCInit(Week, Hour, Min, Sec);

    while (1)
    {
        if (temp)
        {
            bsp_Led1_Off();
            bsp_Led2_On();
        }
        else
        {
            bsp_Led1_On();
            bsp_Led2_Off();
        }

        RTC_1HZ = 0;

        while (!RTC_1HZ);

        while (!md_rtc_is_active_flag_sync_done(RTC));

        u16RtcTime = md_rtc_get_time_second(RTC);
        printf("Sec : %02x  ", u16RtcTime);
        u16RtcTime = md_rtc_get_time_minute(RTC);
        printf("Min : %02x  ", u16RtcTime);
        u16RtcTime = md_rtc_get_time_hour(RTC);
        printf("Hour : %02x  ", u16RtcTime);
        u16RtcTime = md_rtc_get_date_weekday(RTC);
        printf("Week : %02x \r\n", u16RtcTime);

        temp ^= 1;
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_rtc(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


