/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov  2021    Ginger          the first version
 *          29 Mar  2022    AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_48M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< GPIOB init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< GPIOB init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOB_PIN2_Init =    /**< GPIOB init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOA_PIN0_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOA_PIN1_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOA_PIN2_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOA_PIN3_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOA_PIN4_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOA_PIN5_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOA_PIN6_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOA_PIN7_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOA_PIN8_Init =    /**< GPIOA init structure */
{
    MD_GPIO_PIN_8,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOC_PIN0_Init =    /**< GPIOC init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOC_PIN1_Init =    /**< GPIOC init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_gpio_inittypedef GPIOC_PIN2_Init =    /**< GPIOC init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF9
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
uint32_t LED_MEM[3 * 15] =
{
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
    0x0, 0x0, 0x0,
};

static uint32_t g_sStartAddr;
uint8_t frameflag;
uint32_t *pdma_src;

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/




/**
* @brief  IRQ Init.
* @param  None
* @retval None
*/
void irq_init(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
    NVIC_EnableIRQ((IRQn_Type) UART4_KBCU_IRQn);
    NVIC_EnableIRQ((IRQn_Type) DMA1_CH0_IRQn);
}


/**
* @brief  DMA Channel 0 Config.
* @param  None
* @retval None
*/void dma_pri_ch00_cfg(void)
{
    md_dma_set_primary_channel_cycle_mode(DMA1, MD_DMA_CHANNEL0, MD_DMA_CHANNEL_CFG_MODE_BASIC);
    md_dma_disable_primary_channel_next_burst_request(DMA1, MD_DMA_CHANNEL0);
    md_dma_set_primary_channel_transfer_number(DMA1, MD_DMA_CHANNEL0, 3 - 1);
    md_dma_set_primary_channel_r_power(DMA1, MD_DMA_CHANNEL0, MD_DMA_CHANNEL_CFG_RPOWER_SIZE_8);
    md_dma_set_primary_channel_destination_address_incremental(DMA1, MD_DMA_CHANNEL0, MD_DMA_CHANNEL_CFG_DSTINC_WORD);
    md_dma_set_primary_channel_destination_address_size(DMA1, MD_DMA_CHANNEL0, MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_WORD);
    md_dma_set_primary_channel_source_address_incremental(DMA1, MD_DMA_CHANNEL0, MD_DMA_CHANNEL_CFG_SRCINC_WORD);
    md_dma_set_primary_channel_source_address_size(DMA1, MD_DMA_CHANNEL0, MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_WORD);
}

/**
* @brief  DMA Init.
* @param  None
* @retval None
*/
void dma_init(void)
{
    g_sStartAddr = 0;
    md_dma_set_configuration(DMA1, 0xE1);
    md_dma_disable_burst_request(DMA1, MD_DMA_CHANNEL0);
    md_dma_disable_burst_request(DMA1, MD_DMA_CHANNEL1);
    md_dma_disable_burst_request(DMA1, MD_DMA_CHANNEL2);
    md_dma_disable_burst_request(DMA1, MD_DMA_CHANNEL3);
    md_dma_disable_burst_request(DMA1, MD_DMA_CHANNEL4);
    md_dma_disable_burst_request(DMA1, MD_DMA_CHANNEL5);

    md_dma_enable_request(DMA1, MD_DMA_CHANNEL0);
    md_dma_enable_request(DMA1, MD_DMA_CHANNEL1);
    md_dma_enable_request(DMA1, MD_DMA_CHANNEL2);
    md_dma_enable_request(DMA1, MD_DMA_CHANNEL3);
    md_dma_enable_request(DMA1, MD_DMA_CHANNEL4);
    md_dma_enable_request(DMA1, MD_DMA_CHANNEL5);


    md_dma_disable_channel(DMA1, MD_DMA_CHANNEL0);
    md_dma_disable_channel(DMA1, MD_DMA_CHANNEL1);
    md_dma_disable_channel(DMA1, MD_DMA_CHANNEL2);
    md_dma_disable_channel(DMA1, MD_DMA_CHANNEL3);
    md_dma_disable_channel(DMA1, MD_DMA_CHANNEL4);
    md_dma_disable_channel(DMA1, MD_DMA_CHANNEL5);

    md_dma_enable_primary_mode(DMA1, MD_DMA_CHANNEL0);
    md_dma_enable_primary_mode(DMA1, MD_DMA_CHANNEL1);
    md_dma_enable_primary_mode(DMA1, MD_DMA_CHANNEL2);
    md_dma_enable_primary_mode(DMA1, MD_DMA_CHANNEL3);
    md_dma_enable_primary_mode(DMA1, MD_DMA_CHANNEL4);
    md_dma_enable_primary_mode(DMA1, MD_DMA_CHANNEL5);

    md_dma_set_channel_priority_default(DMA1, MD_DMA_CHANNEL0);
    md_dma_set_channel_priority_default(DMA1, MD_DMA_CHANNEL1);
    md_dma_set_channel_priority_default(DMA1, MD_DMA_CHANNEL2);
    md_dma_set_channel_priority_default(DMA1, MD_DMA_CHANNEL3);
    md_dma_set_channel_priority_default(DMA1, MD_DMA_CHANNEL4);
    md_dma_set_channel_priority_default(DMA1, MD_DMA_CHANNEL5);


    md_dma_enable_burst_request(DMA1, MD_DMA_CHANNEL0);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL0, MD_DMA_PRS_KBCU);
    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL0);

    dma_pri_ch00_cfg();

    pdma_src = &LED_MEM[0] + 3 * 1 - 1;
    md_dma_set_primary_channel_destination_data_end_address(DMA1, MD_DMA_CHANNEL0, (uint32_t) &KBCU->LED2);
    md_dma_set_primary_channel_source_data_end_address(DMA1, MD_DMA_CHANNEL0, (uint32_t) pdma_src);

    md_dma_enable_channel(DMA1, MD_DMA_CHANNEL0);
}

/**
* @brief  DMA IRQHandler.
* @param  None
* @retval None
*/
void dma_ch0_int_handler(void)
{
    uint8_t ii;

    for (int i = MD_DMA_CHANNEL0; i <= MD_DMA_CHANNEL5; i++)
    {
        if (md_dma_is_masked_it_channel_done(DMA1, i))
        {
            md_dma_clear_it_channel_done(DMA1, i);
        }
    }

    g_sStartAddr += 3;                 //start address of SRAM shift 3 words

    if (g_sStartAddr == 3 * 15)        // out of range
    {
        g_sStartAddr = 0;                // reset to 0
    }

    for (ii = 0; ii < 3; ii++)
    {
        if (*(pdma_src - 2 + ii + g_sStartAddr) != 0)
        {
            switch (ii)
            {
                case 0:
                    (*(pdma_src - 2 + ii + g_sStartAddr))++;

                    if ((*(pdma_src - 2 + ii + g_sStartAddr) & 0x7) > 5)
                    {
                        *(pdma_src - 2 + ii + g_sStartAddr) &= ~(0x7);

                        if ((*(pdma_src - 2 + ii + g_sStartAddr) & 0xF8) != 0xF8)
                        {
                            *(pdma_src - 2 + ii + g_sStartAddr) += 0x8000008;
                        }
                        else
                        {
                            *(pdma_src - 2 + ii + g_sStartAddr) = 0;
                        }
                    }

                    break;

                case 1:
                    *(pdma_src - 2 + ii + g_sStartAddr) = (*(pdma_src - 2 + ii + g_sStartAddr)) + 0x100;

                    if ((*(pdma_src - 2 + ii + g_sStartAddr) & 0x700) > 0x500)
                    {
                        *(pdma_src - 2 + ii + g_sStartAddr) &= ~(0x700);

                        if ((*(pdma_src - 2 + ii + g_sStartAddr) & 0xF800) != 0xF800)
                        {
                            *(pdma_src - 2 + ii + g_sStartAddr) += 0x8000800;
                        }
                        else
                        {
                            *(pdma_src - 2 + ii + g_sStartAddr) = 0;
                        }
                    }

                    break;

                case 2:
                    *(pdma_src - 2 + ii + g_sStartAddr) = (*(pdma_src - 2 + ii + g_sStartAddr)) + 0x10000;

                    if ((*(pdma_src - 2 + ii + g_sStartAddr) & 0x70000) > 0x50000)
                    {
                        *(pdma_src - 2 + ii + g_sStartAddr) &= ~(0x70000);

                        if ((*(pdma_src - 2 + ii + g_sStartAddr) & 0xF80000) != 0xF80000)
                        {
                            *(pdma_src - 2 + ii + g_sStartAddr) += 0x8080000;
                        }
                        else
                        {
                            *(pdma_src - 2 + ii + g_sStartAddr) = 0;
                        }
                    }

                    break;
            }
        }
    }

    dma_pri_ch00_cfg();
    md_dma_set_primary_channel_destination_data_end_address(DMA1, MD_DMA_CHANNEL0, (uint32_t) &KBCU->LED2);
    md_dma_set_primary_channel_source_data_end_address(DMA1, MD_DMA_CHANNEL0, (uint32_t)(pdma_src + g_sStartAddr));

    md_dma_enable_channel(DMA1, MD_DMA_CHANNEL0);
}

/**
* @brief  KBCU Init.
* @param  None
* @retval None
*/
void kbcu_init(void)
{
    md_kbcu_set_con1_column_sel(KBCU, MD_KBCU_CON1_COLUMN_SEL_15);
    md_kbcu_set_con2_arvalue(KBCU, 255);
    md_kbcu_set_con2_frdiv(KBCU, 0);
    md_kbcu_set_con2_dtvalue(KBCU, MD_KBCU_CON2_DTVALUE_7);
    md_kbcu_set_con2_colvalue(KBCU, MD_KBCU_CON2_COLVALUE_32);

    md_kbcu_enable_it_frame(KBCU);
}

/**
* @brief  Get the pressed key.
* @param  keybuf: Pointer that contains the row of the pressed keys.
* @retval None
*/
void kbcu_keydata(uint8_t *keybuf)
{
    uint32_t key_scan, col_bitmap;
    uint8_t ii, *pkeyscan;

    key_scan = md_kbcu_get_scan0(KBCU);

    pkeyscan = (uint8_t *)&key_scan;

    col_bitmap = md_kbcu_get_stat(KBCU);

    for (ii = 0; ii < 3; ii++)
    {
        if (col_bitmap & (1 << ii))
        {
            *(keybuf + ii) = *(pkeyscan + ii);
        }
    }
}

/**
* @brief  KBCU IRQHandler.
* @param  None
* @retval None
*/
void kbcu_int_handler(void)
{
    uint32_t kbcu_intr;

    kbcu_intr = md_kbcu_get_ifm(KBCU);
    md_kbcu_set_icr(KBCU, kbcu_intr);

    if (kbcu_intr & KBCU_IFM_FRAME_MSK)
    {
        kbcu_intr &= ~KBCU_IFM_FRAME_MSK;
        frameflag = 1;
    }

    if (kbcu_intr & KBCU_IFM_COLUMN_MSK)
    {
        kbcu_intr &= ~KBCU_IFM_COLUMN_MSK;
    }

    if (kbcu_intr & KBCU_IFM_KEY_MSK)
    {
        kbcu_intr &= ~KBCU_IFM_KEY_MSK;
    }
}

/**
* @brief  main.
* @param  None
* @retval None
*/
int main(void)
{
    uint8_t ii, aa, keybitmap[3] = {0};
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    irq_init();
    kbcu_init();
    dma_init();
    md_kbcu_enable_con1_kben(KBCU);
    printf("SysFosc=%dHz\r\n", SystemFrequency_SysClk);

    while (1)
    {
        if (frameflag)
        {
            frameflag = 0;
            //Capture the COLUMN data and store it in the key bitmap
            kbcu_keydata(keybitmap);

            for (ii = 0; ii < 3; ii++)
            {
                //Determine which column the value is not 0 (not 0 means that the button of a row in that column is pressed)
                if (keybitmap[ii] != 0)
                {
                    for (aa = 0; aa < 3; aa++)
                    {
                        /*
                          shift1 uses aa and the and operation to take out the row in the column data for judgment
                          (when the judgment is 1, ii is the column, aa is the row, you can know which button is pressed)
                        */
                        if (keybitmap[ii] & (1 << aa))
                        {
                            switch (aa)
                            {
                                case 0:
                                    LED_MEM[ii * 3] = 0x1000000;
                                    break;

                                case 1:
                                    LED_MEM[ii * 3 + 1] = 0x2000000;
                                    break;

                                case 2:
                                    LED_MEM[ii * 3 + 2] = 0x4000000;
                                    break;
                            }

                            printf("%d\r\n", 3 * ii + aa);
                        }
                    }

                    keybitmap[ii] = 0;
                }
            }
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_kbcu(RCU);
    md_rcu_enable_dma1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOA, &GPIOA_PIN0_Init);   // Row0 R
    md_gpio_init(GPIOA, &GPIOA_PIN1_Init);   // Row0 G
    md_gpio_init(GPIOA, &GPIOA_PIN2_Init);   // Row0 B
    md_gpio_init(GPIOA, &GPIOA_PIN3_Init);   // Row1 R
    md_gpio_init(GPIOA, &GPIOA_PIN4_Init);   // Row1 G
    md_gpio_init(GPIOA, &GPIOA_PIN5_Init);   // Row1 B
    md_gpio_init(GPIOA, &GPIOA_PIN6_Init);   // Row2 R
    md_gpio_init(GPIOA, &GPIOA_PIN7_Init);   // Row2 G
    md_gpio_init(GPIOA, &GPIOA_PIN8_Init);   // Row2 B
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);   // UART1 Tx
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);   // UART1 Rx
    md_gpio_init(GPIOC, &GPIOC_PIN0_Init);   // Column0
    md_gpio_init(GPIOC, &GPIOC_PIN1_Init);   // Column1
    md_gpio_init(GPIOC, &GPIOC_PIN2_Init);   // Column2
    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);   // KEY0
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);   // KEY1
    md_gpio_init(GPIOB, &GPIOB_PIN2_Init);   // KEY2
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY_MSK)));  /* Tx FIFO full */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
