/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          30 Dec 2021     Ginger          Modify the example for PDS board
 *          28 Mar 2022     AE Team         Modify MD Driver
 *          22 Apr 2022     AE Team         Modify I2S Setting
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOA_PIN4_Init =    /**< SPI1_NSS/I2S1_WS init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN5_Init =    /**< SPI1_SCK/I2S1_CK init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN6_Init =    /**< SPI1_MISO/I2S1_SDI init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN7_Init =    /**< SPI1_MOSI/I2S1_SDO init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_i2s_inittypedef I2S1_InitStruct =    /**< SPI init structure */
{
    MD_SPI_MODE_MASTER,
    MD_SPI_PHASE_1EDGE,
    MD_SPI_POLARITY_LOW,
    MD_SPI_BAUDRATEPRESCALER_DIV8,
    MD_SPI_MSB_FIRST,
    MD_SPI_FULL_DUPLEX,
    MD_SPI_FRAME_FORMAT_8BIT,
    MD_SPI_NSS_HARD,
};

/* Private define -------------------------------------------------------------*/
#define BUFFER_SIZE  16

/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Tx data buffer */
//uint16_t I2S_TxBuf[BUFFER_SIZE] = {0x5555,0xAAAA,0x5555,0xAAAA,0x5555,0xAAAA,0x5555,0xAAAA,
//                                   0x5555,0xAAAA,0x5555,0xAAAA,0x5555,0xAAAA,0x5555,0xAAAA};


uint16_t I2S_TxBuf[BUFFER_SIZE] = {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
                                   0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
                                  };

/* Rx data buffer */
uint16_t I2S_RxBuf[BUFFER_SIZE] = {0};

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/
void md_i2s1_init()
{
    md_i2s_disable(I2S1);                                         //I2S disable
    md_i2s_set_mode(I2S1, MD_I2S_MASTER_FULL_DUPLEX);             //Master transmit / Receive
    md_i2s_set_standard(I2S1, MD_I2S_MSB_STANDARD);               //MSB(Left Justified)
    md_i2s_set_channel_length(I2S1, MD_I2S_CHANNEL_LENGTH_16BIT); //16 bits
    md_i2s_set_data_length(I2S1, MD_I2S_DATA_LENGTH_16BIT);       //16 bits
    md_i2s_set_div(I2S1, 0x2);                                    //prescale 2
    md_i2s_set_div_odd(I2S1, MD_I2S_EVEN);                        //i2sdiv/2
    md_i2s_enable_mclock(I2S1);                                   //MCLK output enable
    md_i2s_enable(I2S1);                                          //I2S enable
}

void irq_set()
{
    NVIC_SetPriority((IRQn_Type) I2S1_IRQn,  0);      //priority 0
    NVIC_SetPriority((IRQn_Type) DMA1_CH0_IRQn,  1);  //priority 1
    NVIC_EnableIRQ((IRQn_Type) I2S1_IRQn);            // SPI1_IRQn interrupt
    NVIC_EnableIRQ((IRQn_Type) DMA1_CH0_IRQn);        // DMA1_CH0_IRQn interrupt
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  * @note   Please connect your own SDI and SDO together to use
  */
int main(void)
{
    int i;

    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    md_i2s1_init();
    __enable_irq();

    for (i = 0; i < 16; i++)
    {
        while (md_i2s_is_active_flag_txf(I2S1));

        md_i2s_set_data_reg(I2S1, I2S_TxBuf[i]);
    }

    while (md_i2s_is_active_flag_busy(I2S1));

    printf("I2S Transmit_Receive \r\n");

    for (i = 0; i < 16; i++)
    {
        I2S_RxBuf[i] = md_i2s_get_data_reg(I2S1);
        printf("I2S_RxBuf[%d] = 0x%02x\r\n", i, I2S_RxBuf[i]);
    }

    while (1);
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_i2s1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PA4: SPI1_NSS/I2S1_WS.
  *         PA5: SPI1_SCK/I2S1_CK.
  *         PA6: SPI1_MISO/I2S1_SDI.
  *         PA7: SPI1_MOSI/I2S1_SDO.

  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN4_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN5_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN6_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN7_Init);
}


/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


