/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          21 Mar 2022     AE Team         Modify MD Driver
 *          17 Nov 2022     AE Team         Modify main code led/button set
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOC_PIN0_Init =    /**< GPIOC Pin0(LED1) init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN1_Init =    /**< GPIOC Pin1(LED2) init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN2_Init =    /**< GPIOC Pin2(LED3) init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN3_Init =    /**< GPIOC Pin3(LED4) init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOD_PIN0_Init =    /**< GPIOD Pin0(Button Up) init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOD_PIN1_Init =    /**< GPIOD Pin1(Button Down) init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOB_PIN1_Init =    /**< GPIOB Pin1(Button Left) init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOB_PIN2_Init =    /**< GPIOB Pin2(Button Right) init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOB_PIN0_Init =    /**< GPIOB Pin(Button Middle) init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void Led1_On(void);
void Led1_Off(void);
void Led2_On(void);
void Led2_Off(void);
void Led3_On(void);
void Led3_Off(void);
void Led4_On(void);
void Led4_Off(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t u32GpioC_Sts;

    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    __enable_irq();

    printf("On %s at %s \r\n", __DATE__, __TIME__);
    printf("SystemFrequency_SysClk:%d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk:%d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk:%d\r\n", SystemFrequency_APBClk);
    printf("\r\n");

    u32GpioC_Sts = 0x00000001;

    while (1)
    {
        u32GpioC_Sts ^= 0x00000001;
        md_tick_waitms(10, 50);

        if (u32GpioC_Sts)
        {
            Led1_On();
            Led2_Off();
            Led3_On();
            Led4_Off();
        }
        else
        {
            Led1_Off();
            Led2_On();
            Led3_Off();
            Led4_On();
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_gp32c4t1(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PC0: LED1.
  *         PC1: LED2.
  *         PC2: LED3.
  *         PC3: LED4.
  *         PB0: GPIO Button Mid.
  *         PD0: GPIO Button Up.
  *         PD1: GPIO Button Down.
  *         PB1: GPIO Button Left.
  *         PB2: GPIO Button Right.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOC, &GPIOC_PIN0_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN1_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN2_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN3_Init);

    md_gpio_init(GPIOB, &GPIOB_PIN0_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN0_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN1_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN1_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN2_Init);
}

/**
  * @brief  Turn-on LED1.
  * @param  None
  * @retval None
  */
void Led1_On()
{
    md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_0);
}

/**
  * @brief  Turn-off LED1.
  * @param  None
  * @retval None
  */
void Led1_Off()
{
    md_gpio_set_pin_high(GPIOC, MD_GPIO_PIN_0);
}

/**
  * @brief  Turn-on LED2.
  * @param  None
  * @retval None
  */
void Led2_On()
{
    md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_1);
}

/**
  * @brief  Turn-off LED2.
  * @param  None
  * @retval None
  */
void Led2_Off()
{
    md_gpio_set_pin_high(GPIOC, MD_GPIO_PIN_1);
}

/**
  * @brief  Turn-on LED3.
  * @param  None
  * @retval None
  */
void Led3_On()
{
    md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_2);
}

/**
  * @brief  Turn-off LED3.
  * @param  None
  * @retval None
  */
void Led3_Off()
{
    md_gpio_set_pin_high(GPIOC, MD_GPIO_PIN_2);
}

/**
  * @brief  Turn-on LED4.
  * @param  None
  * @retval None
  */
void Led4_On()
{
    md_gpio_set_pin_low(GPIOC, MD_GPIO_PIN_3);
}

/**
  * @brief  Turn-off LED4.
  * @param  None
  * @retval None
  */
void Led4_Off()
{
    md_gpio_set_pin_high(GPIOC, MD_GPIO_PIN_3);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


