/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          23 Dec 2021     Ginger          Modify the example for PDS board
 *          25 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
uint32_t  EFPattern0[MD_FC_PC_EF_MPAGESZ / 4] =
{
    0x55005501, 0x55025503, 0x55045505, 0x55065507, 0x55085509, 0x550a550b, 0x550c550d, 0x550e550f,
    0x55105511, 0x55125513, 0x55145515, 0x55165517, 0x55185519, 0x551a551b, 0x551c551d, 0x551e551f,
    0x55205521, 0x55225523, 0x55245525, 0x55265527, 0x55285529, 0x552a552b, 0x552c552d, 0x552e552f,
    0x55305531, 0x55325533, 0x55345535, 0x55365537, 0x55385539, 0x553a553b, 0x553c553d, 0x553e553f,
    0x55405541, 0x55425543, 0x55445545, 0x55465547, 0x55485549, 0x554a554b, 0x554c554d, 0x554e554f,
    0x55505551, 0x55525553, 0x55545555, 0x55565557, 0x55585559, 0x555a555b, 0x555c555d, 0x555e555f,
    0x55605561, 0x55625563, 0x55645565, 0x55665567, 0x55685569, 0x556a556b, 0x556c556d, 0x556e556f,
    0x55705571, 0x55725573, 0x55745575, 0x55765577, 0x55785579, 0x557a557b, 0x557c557d, 0x557e557f,
    0x55805581, 0x55825583, 0x55845585, 0x55865587, 0x55885589, 0x558a558b, 0x558c558d, 0x558e558f,
    0x55905591, 0x55925593, 0x55945595, 0x55965597, 0x55985599, 0x559a559b, 0x559c559d, 0x559e559f,
    0x55a055a1, 0x55a255a3, 0x55a455a5, 0x55a655a7, 0x55a855a9, 0x55aa55ab, 0x55ac55ad, 0x55ae55af,
    0x55b055b1, 0x55b255b3, 0x55b455b5, 0x55b655b7, 0x55b855b9, 0x55ba55bb, 0x55bc55bd, 0x55be55bf,
    0x55c055c1, 0x55c255c3, 0x55c455c5, 0x55c655c7, 0x55c855c9, 0x55ca55cb, 0x55cc55cd, 0x55ce55cf,
    0x55d055d1, 0x55d255d3, 0x55d455d5, 0x55d655d7, 0x55d855d9, 0x55da55db, 0x55dc55dd, 0x55de55df,
    0x55e055e1, 0x55e255e3, 0x55e455e5, 0x55e655e7, 0x55e855e9, 0x55ea55eb, 0x55ec55ed, 0x55ee55ef,
    0x55f055f1, 0x55f255f3, 0x55f455f5, 0x55f655f7, 0x55f855f9, 0x55fa55fb, 0x55fc55fd, 0x55fe55ff,
};

uint32_t  EFPattern1[MD_FC_PC_EF_MPAGESZ / 4] =
{
    0xaa00aa01, 0xaa02aa03, 0xaa04aa05, 0xaa06aa07, 0xaa08aa09, 0xaa0aaa0b, 0xaa0caa0d, 0xaa0eaa0f,
    0xaa10aa11, 0xaa12aa13, 0xaa14aa15, 0xaa16aa17, 0xaa18aa19, 0xaa1aaa1b, 0xaa1caa1d, 0xaa1eaa1f,
    0xaa20aa21, 0xaa22aa23, 0xaa24aa25, 0xaa26aa27, 0xaa28aa29, 0xaa2aaa2b, 0xaa2caa2d, 0xaa2eaa2f,
    0xaa30aa31, 0xaa32aa33, 0xaa34aa35, 0xaa36aa37, 0xaa38aa39, 0xaa3aaa3b, 0xaa3caa3d, 0xaa3eaa3f,
    0xaa40aa41, 0xaa42aa43, 0xaa44aa45, 0xaa46aa47, 0xaa48aa49, 0xaa4aaa4b, 0xaa4caa4d, 0xaa4eaa4f,
    0xaa50aa51, 0xaa52aa53, 0xaa54aa55, 0xaa56aa57, 0xaa58aa59, 0xaa5aaa5b, 0xaa5caa5d, 0xaa5eaa5f,
    0xaa60aa61, 0xaa62aa63, 0xaa64aa65, 0xaa66aa67, 0xaa68aa69, 0xaa6aaa6b, 0xaa6caa6d, 0xaa6eaa6f,
    0xaa70aa71, 0xaa72aa73, 0xaa74aa75, 0xaa76aa77, 0xaa78aa79, 0xaa7aaa7b, 0xaa7caa7d, 0xaa7eaa7f,
    0xaa80aa81, 0xaa82aa83, 0xaa84aa85, 0xaa86aa87, 0xaa88aa89, 0xaa8aaa8b, 0xaa8caa8d, 0xaa8eaa8f,
    0xaa90aa91, 0xaa92aa93, 0xaa94aa95, 0xaa96aa97, 0xaa98aa99, 0xaa9aaa9b, 0xaa9caa9d, 0xaa9eaa9f,
    0xaaa0aaa1, 0xaaa2aaa3, 0xaaa4aaa5, 0xaaa6aaa7, 0xaaa8aaa9, 0xaaaaaaab, 0xaaacaaad, 0xaaaeaaaf,
    0xaab0aab1, 0xaab2aab3, 0xaab4aab5, 0xaab6aab7, 0xaab8aab9, 0xaabaaabb, 0xaabcaabd, 0xaabeaabf,
    0xaac0aac1, 0xaac2aac3, 0xaac4aac5, 0xaac6aac7, 0xaac8aac9, 0xaacaaacb, 0xaaccaacd, 0xaaceaacf,
    0xaad0aad1, 0xaad2aad3, 0xaad4aad5, 0xaad6aad7, 0xaad8aad9, 0xaadaaadb, 0xaadcaadd, 0xaadeaadf,
    0xaae0aae1, 0xaae2aae3, 0xaae4aae5, 0xaae6aae7, 0xaae8aae9, 0xaaeaaaeb, 0xaaecaaed, 0xaaeeaaef,
    0xaaf0aaf1, 0xaaf2aaf3, 0xaaf4aaf5, 0xaaf6aaf7, 0xaaf8aaf9, 0xaafaaafb, 0xaafcaafd, 0xaafeaaff,
};

uint32_t  EFData[MD_FC_PC_EF_MPAGESZ / 4];

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  IrqInit
  * @param  None
  * @retval None
  */
void IrqInit(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
}

/**
  * @brief  set irq priority
  * @param  None
  * @retval None
  */
void IrqPriority(void)
{
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    md_fc_ControlTypeDef  FCControlPara;
    uint16_t  page, ii;
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);

    printf("Flash Dump Example\r\n");
    printf("SysFosc Sys=%dHz\r\n", SystemFrequency_SysClk);

    md_tick_waitms(10, 50);

    //fc initial setting
    page = 20;
    FCControlPara.SAddr = page * MD_FC_PC_EF_MPAGESZ;
    FCControlPara.SAddrC = ~(page * MD_FC_PC_EF_MPAGESZ);
    FCControlPara.BCnt = MD_FC_PC_EF_MPAGESZ;
    FCControlPara.pU32Buf = EFPattern0;


    md_fc_unlock();
    md_fc_sector_erase(&FCControlPara);
    md_fc_program(&FCControlPara);

    md_fc_lock();


    FCControlPara.pU32Buf = EFData; //EFData;
    md_fc_read(&FCControlPara);
    printf("\r\nEFPage=%d\r\n", page);

    for (ii = 0; ii < MD_FC_PC_EF_MPAGESZ / 4; ii++)
    {
        printf("0x%08x ", EFData[ii]);

        if ((ii & 0x7) == 0x7)
            printf("\r\n");
    }


    printf("\r\n");

    md_tick_waitms(10, 50);

    page = 48;
    FCControlPara.SAddr = page * MD_FC_PC_EF_MPAGESZ;
    FCControlPara.SAddrC = ~(page * MD_FC_PC_EF_MPAGESZ);
    FCControlPara.BCnt = MD_FC_PC_EF_MPAGESZ;
    FCControlPara.pU32Buf = EFPattern1;
    md_fc_unlock();
    md_fc_sector_erase(&FCControlPara);
    md_fc_program(&FCControlPara);
    md_fc_lock();
    FCControlPara.pU32Buf = EFData;
    md_fc_read(&FCControlPara);
    printf("EFPage=%d\r\n", page);

    for (ii = 0; ii < MD_FC_PC_EF_MPAGESZ / 4; ii++)
    {
        printf("0x%08x ", EFData[ii]);

        if ((ii & 0x7) == 0x7)
            printf("\r\n");
    }

    printf("\r\n");

    while (1);
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


