/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    6 Jan 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          6 Jan  2021     Ginger          the first version
 *          25 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};
#define UART_BUF_SIZE 8
md_dma_init_typedef dma_initStruct =
{
    MD_DMA_PRIMARY,
    MD_DMA_CHANNEL1,
    MD_DMA_CHANNEL_CFG_MODE_BASIC,
    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1,
    MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_BYTE,
    MD_DMA_CHANNEL_CFG_SRCINC_NO_INC,
    MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_BYTE,
    MD_DMA_CHANNEL_CFG_DSTINC_BYTE,
    UART_BUF_SIZE
};

/* Private define -------------------------------------------------------------*/


/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
//uint8_t tx_buf[UART_BUF_SIZE] = {0x48, 0x69, 0x20, 0x47, 0x69, 0x6E, 0x67, 0x65, 0x72};
uint8_t rx_buf[UART_BUF_SIZE];
volatile uint8_t dma12_interrupt = 0;
__IO uint32_t DMA_INTR0, DMA_INTR12, UART1_INTR;

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void DMAInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    volatile uint32_t i;
    uint32_t temp = 0;
    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    NVIC_EnableIRQ((IRQn_Type) DMA1_CH12_IRQn);        // DMA1_CH12_IRQn interrupt

    __enable_irq();

    DMAInit();
    printf("Please enter %d data\r\n", UART_BUF_SIZE);

    while (!dma12_interrupt);

    dma12_interrupt = 0;
    printf("\r\n");


    for (i = 0; i < UART_BUF_SIZE; i++)
        printf("rx_buf[%d]:%c\r\n", i, rx_buf[i]);

    printf("UART_RX_DMA OK\r\n");


    while (1)
    {
        temp ^= 1;

        for (i = 0; i < 100000; i++);

        if (temp)
        {
            bsp_Led1_On();
            bsp_Led2_Off();
        }
        else
        {
            bsp_Led1_Off();
            bsp_Led2_On();
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_dma1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PC1: GPIO LED RED.
  *         PC0: GPIO LED RED.
  *         PC3: GPIO LED RED.
  *         PC2: GPIO LED RED.
  *         PB1: GPIO Button Mid.
  *         PC10: GPIO Button Left.
  *         PC13: GPIO Button Up.
  *         PD0: GPIO Button Right.
  *         PD1: GPIO Button Down.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}


/**
  * @brief  DMA init
  * @param  None
  * @retval None
  */

void DMAInit(void)
{
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL1, MD_DMA_PRS_UART1_RX);
    md_uart_enable_rxdma(UART1);

    md_dma_set_configuration(DMA1, ENABLE);
    dma_initStruct.source_data_end_address = (uint32_t)&UART1->RXDATA;
    dma_initStruct.destination_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)rx_buf, UART_BUF_SIZE, MD_DMA_CHANNEL_SIZE_BYTE);
    md_dma_init(DMA1, dma_initStruct);

    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL1);

}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


