/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          29 Dec 2021     Ginger          Modify the example for PDS board
 *          25 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
#define TOTAL_SIZE  0x100

md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_dma_init_typedef dma_PRIMARY_initStruct =
{
    MD_DMA_PRIMARY,
    MD_DMA_CHANNEL0,
    MD_DMA_CHANNEL_CFG_MODE_PING_PONG,
    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_64,
    MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_BYTE,
    MD_DMA_CHANNEL_CFG_SRCINC_BYTE,
    MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_BYTE,
    MD_DMA_CHANNEL_CFG_DSTINC_BYTE,
    TOTAL_SIZE / 4
};

md_dma_init_typedef dma_ALTERNATE_initStruct =
{
    MD_DMA_ALTERNATE,
    MD_DMA_CHANNEL0,
    MD_DMA_CHANNEL_CFG_MODE_PING_PONG,
    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_64,
    MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_BYTE,
    MD_DMA_CHANNEL_CFG_SRCINC_BYTE,
    MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_BYTE,
    MD_DMA_CHANNEL_CFG_DSTINC_BYTE,
    TOTAL_SIZE / 4
};


/* Private define -------------------------------------------------------------*/


uint8_t source[TOTAL_SIZE] = {0};
uint8_t destination[TOTAL_SIZE] = {0};

#define SOURCE_START_ADDR       source
#define DESTINATION_START_ADDR  destination

#define DEBUG

/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);
void MemoryInit(void);
void DMAInit(void);

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    volatile uint32_t i;
    uint32_t temp = 0;

    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();
    MemoryInit();
    DMAInit();
    printf("DMA MtoM demo\r\n");

#ifdef DEBUG
    uint8_t *ptrSource = (uint8_t *)SOURCE_START_ADDR;
    uint8_t *ptrDestination = (uint8_t *)DESTINATION_START_ADDR;

    for (int j = 0; j < TOTAL_SIZE; j++)
    {
        printf("Source      0x%x:0x%x\r\n", (unsigned int)(ptrSource + j), (unsigned int) * (ptrSource + j));
        printf("Destination 0x%x:0x%x\r\n", (unsigned int)(ptrDestination + j), (unsigned int) * (ptrDestination + j));
        printf("\r\n");
    }

#endif

    while (1)
    {
        temp ^= 1;

        for (i = 0; i < 100000; i++);

        if (temp)
        {
            bsp_Led1_On();
            bsp_Led2_Off();
        }
        else
        {
            bsp_Led1_Off();
            bsp_Led2_On();
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_dma1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}

/**
  * @brief  Memory init
  * @param  None
  * @retval None
  * @note   Erase SOURCE and DESTINATION data int the sram
  */

void MemoryInit(void)
{
    uint8_t *ptr;
    int i;

    ptr = (uint8_t *) SOURCE_START_ADDR;

    for (i = 0; i < TOTAL_SIZE; i++)
    {
        *(ptr++) = i;
    }

    ptr = (uint8_t *) DESTINATION_START_ADDR;

    for (i = 0; i < TOTAL_SIZE; i++)
    {
        *(ptr++) = 0;
    }
}

/**
  * @brief  DMA init
  * @param  None
  * @retval None
  */

void DMAInit()
{
    int i = 0;
    md_dma_set_configuration(DMA1, ENABLE);
    i = 0; //PRIMARY setting
    dma_PRIMARY_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(SOURCE_START_ADDR + i * (TOTAL_SIZE / 4)), TOTAL_SIZE / 4, MD_DMA_CHANNEL_SIZE_BYTE);
    dma_PRIMARY_initStruct.destination_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(DESTINATION_START_ADDR + i * (TOTAL_SIZE / 4)), TOTAL_SIZE / 4, MD_DMA_CHANNEL_SIZE_BYTE);
    md_dma_init(DMA1, dma_PRIMARY_initStruct);
    i = 1; //ALTERNATE setting
    dma_ALTERNATE_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(SOURCE_START_ADDR + i * (TOTAL_SIZE / 4)), TOTAL_SIZE / 4, MD_DMA_CHANNEL_SIZE_BYTE);
    dma_ALTERNATE_initStruct.destination_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(DESTINATION_START_ADDR + i * (TOTAL_SIZE / 4)), TOTAL_SIZE / 4, MD_DMA_CHANNEL_SIZE_BYTE);
    md_dma_init(DMA1, dma_ALTERNATE_initStruct);

    md_dma_enable_soft_request(DMA1, MD_DMA_CHANNEL0);

    while (md_dma_is_active_flag_channel_done(DMA1, MD_DMA_CHANNEL0) == 0);

    //PRIMARY send end
    md_dma_clear_it_channel_done(DMA1, MD_DMA_CHANNEL0);
    i = 2; //PRIMARY setting
    dma_PRIMARY_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(SOURCE_START_ADDR + i * (TOTAL_SIZE / 4)), TOTAL_SIZE / 4, MD_DMA_CHANNEL_SIZE_BYTE);
    dma_PRIMARY_initStruct.destination_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(DESTINATION_START_ADDR + i * (TOTAL_SIZE / 4)), TOTAL_SIZE / 4, MD_DMA_CHANNEL_SIZE_BYTE);
    md_dma_init(DMA1, dma_PRIMARY_initStruct);

    md_dma_enable_soft_request(DMA1, MD_DMA_CHANNEL0);

    while (md_dma_is_active_flag_channel_done(DMA1, MD_DMA_CHANNEL0) == 0);

    //ALTERNATE send end
    md_dma_clear_it_channel_done(DMA1, MD_DMA_CHANNEL0);
    i = 3; //ALTERNATE setting
    dma_ALTERNATE_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(SOURCE_START_ADDR + i * (TOTAL_SIZE / 4)), TOTAL_SIZE / 4, MD_DMA_CHANNEL_SIZE_BYTE);
    dma_ALTERNATE_initStruct.destination_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(DESTINATION_START_ADDR + i * (TOTAL_SIZE / 4)), TOTAL_SIZE / 4, MD_DMA_CHANNEL_SIZE_BYTE);
    md_dma_init(DMA1, dma_ALTERNATE_initStruct);

    md_dma_enable_soft_request(DMA1, MD_DMA_CHANNEL0);

    while (md_dma_is_active_flag_channel_done(DMA1, MD_DMA_CHANNEL0) == 0);

    //PRIMARY send end
    md_dma_clear_it_channel_done(DMA1, MD_DMA_CHANNEL0);

    md_dma_enable_soft_request(DMA1, MD_DMA_CHANNEL0);

    while (md_dma_is_active_flag_channel_done(DMA1, MD_DMA_CHANNEL0) == 0);

    //ALTERNATE send end
    md_dma_clear_it_channel_done(DMA1, MD_DMA_CHANNEL0);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  /* Tx FIFO empty */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


