/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          29 Dec 2021     Ginger          Modify the example for PDS board
 *          25 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
#define TOTAL_SIZE  0x1000

md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOA_PIN4_Init =    /**< SPI1_NSS/I2S1_WS init structure */
{
    MD_GPIO_PIN_4,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN5_Init =    /**< SPI1_SCK/I2S1_CK init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN6_Init =    /**< SPI1_MISO/I2S1_SDI init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN7_Init =    /**< SPI1_MOSI/I2S1_SDO init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_dma_init_typedef dma_I2STx_initStruct =
{
    MD_DMA_PRIMARY,
    MD_DMA_CHANNEL0,
    MD_DMA_CHANNEL_CFG_MODE_PERIPHERAL_PRIMARY,
    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_4,
    MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_WORD,
    MD_DMA_CHANNEL_CFG_SRCINC_WORD,
    MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_WORD,
    MD_DMA_CHANNEL_CFG_DSTINC_WORD,
    8
};

md_dma_init_typedef dma_I2SRx_initStruct =
{
    MD_DMA_PRIMARY,
    MD_DMA_CHANNEL1,
    MD_DMA_CHANNEL_CFG_MODE_PERIPHERAL_PRIMARY,
    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_4,
    MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_WORD,
    MD_DMA_CHANNEL_CFG_SRCINC_WORD,
    MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_WORD,
    MD_DMA_CHANNEL_CFG_DSTINC_WORD,
    8
};

/* Private define -------------------------------------------------------------*/

/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);
void MemoryInit(void);

void DMAInit_I2S(uint32_t);

__IO uint32_t DMA_INTR0, DMA_INTR1;
uint32_t I2S_Buf[1025] = {0};
md_dma_channel_config_typedef DMA_list_TX[2] = {0},DMA_list_RX[2] = {0};

void md_i2s1_init()
{
    md_i2s_disable(I2S1);                                             //I2S disable
    md_i2s_set_mode(I2S1, MD_I2S_MASTER_FULL_DUPLEX);                 //Master transmit / receive
    md_i2s_set_standard(I2S1, MD_I2S_MSB_STANDARD);                   //MSB(Left Justified)
    md_i2s_set_channel_length(I2S1, MD_I2S_CHANNEL_LENGTH_32BIT);     //32 bits
    md_i2s_set_data_length(I2S1, MD_I2S_DATA_LENGTH_32BIT);           //32 bits
    md_i2s_set_div(I2S1, 0x2);                                        //prescale 2
    md_i2s_set_div_odd(I2S1, MD_I2S_EVEN);                            //i2sdiv/2
    md_i2s_enable_mclock(I2S1);                                       //MCLK output enable
    md_i2s_enable(I2S1);                                              //I2S enable
}

/* Private functions ----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    volatile uint32_t i;
    uint32_t temp = 0;
    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    md_i2s1_init();    
    NVIC_EnableIRQ((IRQn_Type) DMA1_CH0_IRQn);        // DMA1_CH0_IRQn interrupt
    NVIC_EnableIRQ((IRQn_Type) DMA1_CH12_IRQn);       // DMA1_CH12_IRQn interrupt    
    __enable_irq(); 
    printf("DMA Scatter aggregate mode for I2S\r\n");
    
    for(i=0;i<TOTAL_SIZE/4;i++)
        I2S_Buf[i] = i;
  
    DMAInit_I2S(TOTAL_SIZE);   

    while (!DMA_INTR0);
    DMA_INTR0 = 0;

    while (!DMA_INTR1);    
    DMA_INTR1 = 0;    
    
    for (i = 0; i < (TOTAL_SIZE/4)+1; i++)
        printf("I2S_Buf[%d]:%d\r\n", i, I2S_Buf[i]);
    
    printf("done\r\n");
    
    md_dma_init(DMA1, dma_I2STx_initStruct);
    md_dma_init(DMA1, dma_I2SRx_initStruct);

    while (!DMA_INTR0);
    DMA_INTR0 = 0;

    while (!DMA_INTR1);
    DMA_INTR1 = 0;    
    
    for (i = 0; i < 1025; i++)
        printf("I2S_Buf[%d]:%d\r\n", i, I2S_Buf[i]);
        
    printf("done\r\n");        

    while (1)
    {
        temp ^= 1;

        for (i = 0; i < 100000; i++);

        if (temp)
        {
            bsp_Led1_On();
            bsp_Led2_Off();
        }
        else
        {
            bsp_Led1_Off();
            bsp_Led2_On();
        }
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_dma1(RCU);
    md_rcu_enable_i2s1(RCU);    
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
    
    md_gpio_init(GPIOA, &GPIOA_PIN4_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN5_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN6_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN7_Init);    
}

/**
  * @brief  DMA init
  * @param  None
  * @retval None
  */

void DMAInit(void)
{
    DMA_list_TX[0].source_data_end_address =            md_dma_set_channel_data_start_address_and_length((uint32_t)(I2S_Buf), 1024, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    DMA_list_TX[0].destination_data_end_address =       (uint32_t)&I2S1->DATA;
    DMA_list_TX[0].control.md_dma_config_typedef.cycle_mode =                 MD_DMA_CHANNEL_CFG_MODE_PERIPHERAL_ALTERNATE;
    DMA_list_TX[0].control.md_dma_config_typedef.transfer_number = 1024 - 1;
    DMA_list_TX[0].control.md_dma_config_typedef.r_power =                    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1;
    DMA_list_TX[0].control.md_dma_config_typedef.source_address_size =        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD;
    DMA_list_TX[0].control.md_dma_config_typedef.source_address_incremental = MD_DMA_CHANNEL_CFG_SRCINC_HALF_WORD;
    DMA_list_TX[0].control.md_dma_config_typedef.destination_address_size =   MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD;
    DMA_list_TX[0].control.md_dma_config_typedef.destination_address_incremental = MD_DMA_CHANNEL_CFG_DSTINC_NO_INC;

    DMA_list_TX[1].source_data_end_address =            md_dma_set_channel_data_start_address_and_length((uint32_t)(I2S_Buf), 2048, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    DMA_list_TX[1].destination_data_end_address =       (uint32_t)&I2S1->DATA;
    DMA_list_TX[1].control.md_dma_config_typedef.cycle_mode =                 MD_DMA_CHANNEL_CFG_MODE_BASIC;
    DMA_list_TX[1].control.md_dma_config_typedef.transfer_number = 1024 - 1;
    DMA_list_TX[1].control.md_dma_config_typedef.r_power =                    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1;
    DMA_list_TX[1].control.md_dma_config_typedef.source_address_size =        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD;
    DMA_list_TX[1].control.md_dma_config_typedef.source_address_incremental = MD_DMA_CHANNEL_CFG_SRCINC_HALF_WORD;
    DMA_list_TX[1].control.md_dma_config_typedef.destination_address_size =   MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD;
    DMA_list_TX[1].control.md_dma_config_typedef.destination_address_incremental = MD_DMA_CHANNEL_CFG_DSTINC_NO_INC;
    
    md_dma_set_configuration(DMA1, ENABLE);
    dma_I2STx_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)DMA_list_TX, 8, MD_DMA_CHANNEL_SIZE_WORD);
    dma_I2STx_initStruct.destination_data_end_address = ((uint32_t)(&(DMA1->ALT_CH00_SRC_DATA_END_PTR))) + (sizeof(md_dma_channel_config_typedef)) - 4;
    md_dma_init(DMA1, dma_I2STx_initStruct);

    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL0);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL0, MD_DMA_PRS_I2S1_TX);

    DMA_list_RX[0].source_data_end_address =            (uint32_t)&I2S1->DATA;
    DMA_list_RX[0].destination_data_end_address =       md_dma_set_channel_data_start_address_and_length((uint32_t)(I2S_Buf)+4, 1024, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    DMA_list_RX[0].control.md_dma_config_typedef.cycle_mode =                 MD_DMA_CHANNEL_CFG_MODE_PERIPHERAL_ALTERNATE;
    DMA_list_RX[0].control.md_dma_config_typedef.transfer_number = 1024 - 1;
    DMA_list_RX[0].control.md_dma_config_typedef.r_power =                    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1;
    DMA_list_RX[0].control.md_dma_config_typedef.source_address_size =        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD;
    DMA_list_RX[0].control.md_dma_config_typedef.source_address_incremental = MD_DMA_CHANNEL_CFG_SRCINC_NO_INC;
    DMA_list_RX[0].control.md_dma_config_typedef.destination_address_size =   MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD;
    DMA_list_RX[0].control.md_dma_config_typedef.destination_address_incremental = MD_DMA_CHANNEL_CFG_DSTINC_HALF_WORD;

    DMA_list_RX[1].source_data_end_address =            (uint32_t)&I2S1->DATA;
    DMA_list_RX[1].destination_data_end_address =       md_dma_set_channel_data_start_address_and_length((uint32_t)(I2S_Buf)+4, 2048, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    DMA_list_RX[1].control.md_dma_config_typedef.cycle_mode =                 MD_DMA_CHANNEL_CFG_MODE_BASIC;
    DMA_list_RX[1].control.md_dma_config_typedef.transfer_number = 1024 - 1;
    DMA_list_RX[1].control.md_dma_config_typedef.r_power =                    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1;
    DMA_list_RX[1].control.md_dma_config_typedef.source_address_size =        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD;
    DMA_list_RX[1].control.md_dma_config_typedef.source_address_incremental = MD_DMA_CHANNEL_CFG_SRCINC_NO_INC;
    DMA_list_RX[1].control.md_dma_config_typedef.destination_address_size =   MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD;
    DMA_list_RX[1].control.md_dma_config_typedef.destination_address_incremental = MD_DMA_CHANNEL_CFG_DSTINC_HALF_WORD;

    md_dma_set_configuration(DMA1, ENABLE);
    dma_I2SRx_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)DMA_list_RX, 8, MD_DMA_CHANNEL_SIZE_WORD);
    dma_I2SRx_initStruct.destination_data_end_address = ((uint32_t)(&(DMA1->ALT_CH01_SRC_DATA_END_PTR))) + (sizeof(md_dma_channel_config_typedef)) - 4;
    md_dma_init(DMA1, dma_I2SRx_initStruct);

    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL1);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL1, MD_DMA_PRS_I2S1_RX);
    
    md_i2s_enable_txdma(I2S1);   
    md_i2s_enable_rxdma(I2S1); 
}


void DMAInit_I2S(uint32_t Total_Size_Byte)
{
    DMA_list_TX[0].source_data_end_address =            md_dma_set_channel_data_start_address_and_length((uint32_t)(I2S_Buf), Total_Size_Byte/4, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    DMA_list_TX[0].destination_data_end_address =       (uint32_t)&I2S1->DATA;
    DMA_list_TX[0].control.md_dma_config_typedef.cycle_mode =                 MD_DMA_CHANNEL_CFG_MODE_PERIPHERAL_ALTERNATE;
    DMA_list_TX[0].control.md_dma_config_typedef.transfer_number = (Total_Size_Byte/4) - 1;
    DMA_list_TX[0].control.md_dma_config_typedef.r_power =                    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1;
    DMA_list_TX[0].control.md_dma_config_typedef.source_address_size =        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD;
    DMA_list_TX[0].control.md_dma_config_typedef.source_address_incremental = MD_DMA_CHANNEL_CFG_SRCINC_HALF_WORD;
    DMA_list_TX[0].control.md_dma_config_typedef.destination_address_size =   MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD;
    DMA_list_TX[0].control.md_dma_config_typedef.destination_address_incremental = MD_DMA_CHANNEL_CFG_DSTINC_NO_INC;

    DMA_list_TX[1].source_data_end_address =            md_dma_set_channel_data_start_address_and_length((uint32_t)(I2S_Buf), Total_Size_Byte/2, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    DMA_list_TX[1].destination_data_end_address =       (uint32_t)&I2S1->DATA;
    DMA_list_TX[1].control.md_dma_config_typedef.cycle_mode =                 MD_DMA_CHANNEL_CFG_MODE_BASIC;
    DMA_list_TX[1].control.md_dma_config_typedef.transfer_number = (Total_Size_Byte/4) - 1;
    DMA_list_TX[1].control.md_dma_config_typedef.r_power =                    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1;
    DMA_list_TX[1].control.md_dma_config_typedef.source_address_size =        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD;
    DMA_list_TX[1].control.md_dma_config_typedef.source_address_incremental = MD_DMA_CHANNEL_CFG_SRCINC_HALF_WORD;
    DMA_list_TX[1].control.md_dma_config_typedef.destination_address_size =   MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD;
    DMA_list_TX[1].control.md_dma_config_typedef.destination_address_incremental = MD_DMA_CHANNEL_CFG_DSTINC_NO_INC;
    
    md_dma_set_configuration(DMA1, ENABLE);
    dma_I2STx_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)DMA_list_TX, 8, MD_DMA_CHANNEL_SIZE_WORD);
    dma_I2STx_initStruct.destination_data_end_address = ((uint32_t)(&(DMA1->ALT_CH00_SRC_DATA_END_PTR))) + (sizeof(md_dma_channel_config_typedef)) - 4;
    md_dma_init(DMA1, dma_I2STx_initStruct);

    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL0);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL0, MD_DMA_PRS_I2S1_TX);

    DMA_list_RX[0].source_data_end_address =            (uint32_t)&I2S1->DATA;
    DMA_list_RX[0].destination_data_end_address =       md_dma_set_channel_data_start_address_and_length((uint32_t)(I2S_Buf)+4, Total_Size_Byte/4, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    DMA_list_RX[0].control.md_dma_config_typedef.cycle_mode =                 MD_DMA_CHANNEL_CFG_MODE_PERIPHERAL_ALTERNATE;
    DMA_list_RX[0].control.md_dma_config_typedef.transfer_number = (Total_Size_Byte/4) - 1;
    DMA_list_RX[0].control.md_dma_config_typedef.r_power =                    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1;
    DMA_list_RX[0].control.md_dma_config_typedef.source_address_size =        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD;
    DMA_list_RX[0].control.md_dma_config_typedef.source_address_incremental = MD_DMA_CHANNEL_CFG_SRCINC_NO_INC;
    DMA_list_RX[0].control.md_dma_config_typedef.destination_address_size =   MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD;
    DMA_list_RX[0].control.md_dma_config_typedef.destination_address_incremental = MD_DMA_CHANNEL_CFG_DSTINC_HALF_WORD;

    DMA_list_RX[1].source_data_end_address =            (uint32_t)&I2S1->DATA;
    DMA_list_RX[1].destination_data_end_address =       md_dma_set_channel_data_start_address_and_length((uint32_t)(I2S_Buf)+4, Total_Size_Byte/2, MD_DMA_CHANNEL_SIZE_HALF_WORD);
    DMA_list_RX[1].control.md_dma_config_typedef.cycle_mode =                 MD_DMA_CHANNEL_CFG_MODE_BASIC;
    DMA_list_RX[1].control.md_dma_config_typedef.transfer_number = (Total_Size_Byte/4) - 1;
    DMA_list_RX[1].control.md_dma_config_typedef.r_power =                    MD_DMA_CHANNEL_CFG_RPOWER_SIZE_1;
    DMA_list_RX[1].control.md_dma_config_typedef.source_address_size =        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_HALF_WORD;
    DMA_list_RX[1].control.md_dma_config_typedef.source_address_incremental = MD_DMA_CHANNEL_CFG_SRCINC_NO_INC;
    DMA_list_RX[1].control.md_dma_config_typedef.destination_address_size =   MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_HALF_WORD;
    DMA_list_RX[1].control.md_dma_config_typedef.destination_address_incremental = MD_DMA_CHANNEL_CFG_DSTINC_HALF_WORD;

    md_dma_set_configuration(DMA1, ENABLE);
    dma_I2SRx_initStruct.source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)DMA_list_RX, 8, MD_DMA_CHANNEL_SIZE_WORD);
    dma_I2SRx_initStruct.destination_data_end_address = ((uint32_t)(&(DMA1->ALT_CH01_SRC_DATA_END_PTR))) + (sizeof(md_dma_channel_config_typedef)) - 4;
    md_dma_init(DMA1, dma_I2SRx_initStruct);

    md_dma_enable_it_channel_done(DMA1, MD_DMA_CHANNEL1);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL1, MD_DMA_PRS_I2S1_RX);
    
    md_i2s_enable_txdma(I2S1);   
    md_i2s_enable_rxdma(I2S1); 
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  /* Tx FIFO empty */

    UART1->TXDATA = ch;            /* Sent byte */
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


