/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          22 Dec 2021     Ginger          Modify the example for PDS board
 *          24 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/

#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);

/* Private functions ----------------------------------------------------------*/

uint16_t  key_scan()
{
    uint8_t    rdat;
    uint16_t  keyin;

    keyin   = 0;
    rdat    = 0;

    while (rdat != 0xD) //Enter key ASCII 0xD
    {
        while (!md_uart_is_active_flag_rfnempty(UART1));

        rdat = md_uart_get_recv_data8(UART1);

        if ((rdat >= '0') & (rdat <= '9'))
        {
            keyin   = (keyin * 10) + (rdat & 0xf);
        }
        else if (rdat != 0xD)
            return (255);
    }

    if (keyin > 255)
        return (255);
    else
        return ((keyin & 0xff));
}


/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t rxdata;

    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    bsp_led_init();
    bsp_key_init();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);

    __enable_irq();

    printf("=============================================\r\n");
    printf("CRC SDK verification On %s, %s\r\n", __DATE__, __TIME__);
    printf("=============================================\r\n");

    printf("Please Enter CRC 8/16/32: ");
    rxdata = key_scan();
    printf("\r\n");

    uint32_t data[5] = {0x12345678, 0x87654321, 0x13572468, 0x24681357, 0x99999999};
    uint32_t ans[3][5] =
    {
        0x1C, 0xD5, 0x8A, 0xC8, 0xF9,                           //CRC-8 standard answer
        0x1E83, 0xE138, 0xA15A, 0xBCDE, 0x2A91,                 //CRC-16/BUYPASS standard answer
        0x4A090E98, 0x88EB54CD, 0xB89A1794, 0xDC30F828, 0x3A9EE8FE //CRC-32 standard answer
    };

    switch (rxdata)
    {
        case 8:


            for (int i = 0; i < 5; i++)
            {
                md_crc_trigger_reset(CRC);
                //CRC-8
                md_crc_set_polynomial_coef(CRC, 0x07);
                md_crc_set_initial_data(CRC, 0x00000000);
                md_crc_set_remainder_data(CRC, 0x00000000);
                md_crc_set_polynomial_size(CRC, MD_CRC_CON_SIZE_POLY8);
                md_crc_set_compare_mode(CRC, MD_CRC_CON_MODE_COMP_ENABLE);
                md_crc_set_input_data_msb(CRC, MD_CRC_CON_MSB_ON);
                md_crc_set_compare_data(CRC, ans[0][i]^md_crc_get_remainder_data(CRC));
                md_crc_set_input_data32(CRC, data[i]);

                while (!md_crc_is_active_flag_done(CRC));

                printf("input data:               0x%x\r\n", data[i]);
                printf("standard answer:          0x%x\r\n", ans[0][i]);
                printf("Answer calculated by CRC: 0x%x\r\n", md_crc_get_output_data_xor(CRC));
                printf("md_crc_compare_stat(%s)\r\n\r\n", md_crc_is_active_flag_compare_fail(CRC) == 0 ? "correct" : "fail");
            }


            break;

        case 16:

            for (int i = 0; i < 5; i++)
            {
                md_crc_trigger_reset(CRC);
                //CRC-16/BUYPASS
                md_crc_set_polynomial_coef(CRC, 0x8005);
                md_crc_set_initial_data(CRC, 0x00000000);
                md_crc_set_remainder_data(CRC, 0x00000000);
                md_crc_set_polynomial_size(CRC, MD_CRC_CON_SIZE_POLY16);
                md_crc_set_compare_mode(CRC, MD_CRC_CON_MODE_COMP_ENABLE);
                md_crc_set_input_data_msb(CRC, MD_CRC_CON_MSB_ON);
                md_crc_set_compare_data(CRC, ans[1][i]^md_crc_get_remainder_data(CRC));
                md_crc_set_input_data32(CRC, data[i]);

                while (!md_crc_is_active_flag_done(CRC));

                printf("input data:               0x%x\r\n", data[i]);
                printf("standard answer:          0x%x\r\n", ans[1][i]);
                printf("Answer calculated by CRC: 0x%x\r\n", md_crc_get_output_data_xor(CRC));
                printf("md_crc_compare_stat(%s)\r\n\r\n", md_crc_is_active_flag_compare_fail(CRC) == 0 ? "correct" : "fail");
            }

            break;

        case 32:

            for (int i = 0; i < 5; i++)
            {
                md_crc_trigger_reset(CRC);
                //CRC-32
                md_crc_set_polynomial_coef(CRC, 0x04c11db7);
                md_crc_set_initial_data(CRC, 0xFFFFFFFF);
                md_crc_set_remainder_data(CRC, 0xFFFFFFFF);
                md_crc_enable_output_data_reverse(CRC);
                md_crc_enable_input_data_reverse(CRC);
                md_crc_set_polynomial_size(CRC, MD_CRC_CON_SIZE_POLY32);
                md_crc_set_compare_mode(CRC, MD_CRC_CON_MODE_COMP_ENABLE);
                md_crc_set_input_data_msb(CRC, MD_CRC_CON_MSB_ON);
                md_crc_set_compare_data(CRC, ans[2][i]^md_crc_get_remainder_data(CRC));
                md_crc_set_input_data32(CRC, data[i]);

                while (!md_crc_is_active_flag_done(CRC));

                printf("input data:               0x%x\r\n", data[i]);
                printf("standard answer:          0x%x\r\n", ans[2][i]);
                printf("Answer calculated by CRC: 0x%x\r\n", md_crc_get_output_data_xor(CRC));
                printf("md_crc_compare_stat(%s)\r\n\r\n", md_crc_is_active_flag_compare_fail(CRC) == 0 ? "correct" : "fail");
            }

            break;

        default:
            break;
    }

    printf("=============================================\r\n");
    printf("CRC SDK verification finish\r\n");
    printf("=============================================\r\n");

    while (1);
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_crc(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}


/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


