/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          29 Dec 2021     Ginger          Modify the example for PDS board
 *          24 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/


/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOD_PIN3_Init =    /**< GPIOD Pin3(LED Red) init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOB_PIN8_Init =    /**< GPIOB Pin8(LED Green) init structure */
{
    MD_GPIO_PIN_8,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOB_PIN5_Init =    /**< GPIOB Pin5(Button Mid) init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN10_Init =    /**< GPIOC Pin10(Button Left) init structure */
{
    MD_GPIO_PIN_10,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN13_Init =    /**< GPIOC Pin13(Button Up) init structure */
{
    MD_GPIO_PIN_13,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOD_PIN0_Init =    /**< GPIOD Pin0(Button Right) init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOD_PIN1_Init =    /**< GPIOD Pin1(Button Down) init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN6_Init =    /**< CMP1_OUT init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF7
};

md_gpio_inittypedef GPIOA_PIN7_Init =    /**< CMP2_OUT init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF7
};

md_gpio_inittypedef GPIOA_PIN1_Init =    /**< CMP1_INP init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_ANALOG,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN3_Init =    /**< CMP2_INP init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_ANALOG,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN0_Init =    /**< CMP1_INN init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_ANALOG,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOA_PIN2_Init =    /**< CMP2_INN init structure */
{
    MD_GPIO_PIN_2,
    MD_GPIO_MODE_ANALOG,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void Led_Red_On(void);
void Led_Red_Off(void);
void Led_Green_On(void);
void Led_Green_Off(void);

/* Private functions ----------------------------------------------------------*/

uint8_t UARTByteRd(UART_TypeDef *UARTx)
{
    while (!md_uart_is_active_flag_rfnempty(UARTx));           // Rx FIFO empty

    return (md_uart_get_recv_data8(UARTx));
}

void cmp_init(uint8_t cmpx)
{
    md_rcu_enable_cmp(RCU);

    printf("\r\n");

    switch (cmpx)
    {
        case 1:
            md_cmp1_enable(CMP);
            md_cmp1_set_blanking_source(CMP, MD_CMP_CFG1_BLANKSEL_CLOSE);
            md_cmp1_disable_filter(CMP);
            md_cmp1_disable_hysteresis(CMP);
            md_cmp1_disable_polarity(CMP);
            md_cmp1_set_output_trigger_select(CMP, MD_CMP_CFG1_OUTSEL_NO);
            md_cmp1_set_input_negative_select(CMP, MD_CMP_CFG1_INNSEL_PA00);
            break;

        case 2:
            md_cmp2_enable(CMP);
            md_cmp2_set_blanking_source(CMP, MD_CMP_CFG2_BLANKSEL_CLOSE);
            md_cmp2_disable_filter(CMP);
            md_cmp2_disable_hysteresis(CMP);
            md_cmp2_disable_polarity(CMP);
            md_cmp2_set_output_trigger_select(CMP, MD_CMP_CFG2_OUTSEL_NO);
            md_cmp2_set_input_negative_select(CMP, MD_CMP_CFG2_INNSEL_PA02);
            break;

        default:
            printf("Enter the wrong number\r\n");
            Led_Red_Off();
            Led_Green_Off();
            break;
    }
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t rxdata;

    __disable_irq();
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    __enable_irq();

    printf("SystemFrequency_SysClk = %d\r\n", SystemFrequency_SysClk);

    while (1)
    {

        printf("Please enter the number 1 or 2 to select CMPx:\r\n");
        rxdata = UARTByteRd(UART1);

        cmp_init(rxdata - '0');

        if (rxdata - '0' == 1)
        {
            printf("cmp1 output\r\n");

            if (md_cmp1_get_output_level(CMP) == MD_CMP_CFG1_VALUE_OUTPUT_HIGH_LEVEL)
            {
                printf("RULT1 = High\r\n");
                Led_Red_On();
                Led_Green_Off();
            }
            else
            {
                printf("RULT1 = Low\r\n");
                Led_Red_Off();
                Led_Green_On();
            }
        }
        else if (rxdata - '0' == 2)
        {
            printf("cmp2 output\r\n");

            if (md_cmp2_get_output_level(CMP) == MD_CMP_CFG2_VALUE_OUTPUT_HIGH_LEVEL)
            {
                printf("RULT2 = High\r\n");
                Led_Red_On();
                Led_Green_Off();
            }
            else
            {
                printf("RULT2 = Low\r\n1");
                Led_Red_Off();
                Led_Green_On();
            }
        }

        printf("================================================\r\n");
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_cmp(RCU);
    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PD3: GPIO LED RED.
  *         PB8: GPIO LED GREEN.
  *         PB5: GPIO Button Mid.
  *         PC10: GPIO Button Left.
  *         PC13: GPIO Button Up.
  *         PD0: GPIO Button Right.
  *         PD1: GPIO Button Down.
  *         PA0: CMP1_INN.
  *         PA1: CMP1_INP.
  *         PA2: CMP2_INN.
  *         PA3: CMP2_INP.
  *         PA6: CMP1_OUT.
  *         PA7: CMP2_OUT.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOA, &GPIOA_PIN0_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN1_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN2_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN3_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN6_Init);
    md_gpio_init(GPIOA, &GPIOA_PIN7_Init);

    md_gpio_init(GPIOD, &GPIOD_PIN3_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN8_Init);

    md_gpio_init(GPIOB, &GPIOB_PIN5_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN10_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN13_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN0_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN1_Init);
}

/**
  * @brief  Turn-on LED Red.
  * @param  None
  * @retval None
  */
void Led_Red_On()
{
    md_gpio_set_pin_low(GPIOD, MD_GPIO_PIN_3);
}

/**
  * @brief  Turn-off LED Red.
  * @param  None
  * @retval None
  */
void Led_Red_Off()
{
    md_gpio_set_pin_high(GPIOD, MD_GPIO_PIN_3);
}

/**
  * @brief  Turn-on LED Green.
  * @param  None
  * @retval None
  */
void Led_Green_On()
{
    md_gpio_set_pin_low(GPIOB, MD_GPIO_PIN_8);
}

/**
  * @brief  Turn-off LED Green.
  * @param  None
  * @retval None
  */
void Led_Green_Off()
{
    md_gpio_set_pin_high(GPIOB, MD_GPIO_PIN_8);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


