/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov 2021     Ginger          the first version
 *          22 Dec 2021     Ginger          Modify the example for PDS board
 *          24 Mar 2022     AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include "main.h"
# include <string.h>
/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_aes_init_typedef aes_data[2];

md_dma_init_typedef dma_initStruct[2] =
{
    //dma entry data in aes
    {
        MD_DMA_PRIMARY,
        MD_DMA_CHANNEL0,
        MD_DMA_CHANNEL_CFG_MODE_BASIC,
        MD_DMA_CHANNEL_CFG_RPOWER_SIZE_128,
        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_WORD,
        MD_DMA_CHANNEL_CFG_SRCINC_WORD,
        MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_WORD,
        MD_DMA_CHANNEL_CFG_SRCINC_NO_INC,
        4
    },
    //dma get data form aes
    {
        MD_DMA_PRIMARY,
        MD_DMA_CHANNEL1,
        MD_DMA_CHANNEL_CFG_MODE_BASIC,
        MD_DMA_CHANNEL_CFG_RPOWER_SIZE_128,
        MD_DMA_CHANNEL_CFG_SRCDATA_SIZE_WORD,
        MD_DMA_CHANNEL_CFG_SRCINC_NO_INC,
        MD_DMA_CHANNEL_CFG_DSTDATA_SIZE_WORD,
        MD_DMA_CHANNEL_CFG_DSTINC_WORD,
        4
    }
};

/* Private define -------------------------------------------------------------*/
//#define AES_CBC_ENCRYPT
//#define AES_CBC_DECRYPT
//#define AES_CBF_ENCRYPT
//#define AES_CBF_DECRYPT
//#define AES_OFB_ENCRYPT
//#define AES_OFB_DECRYPT
//#define AES_CTR_ENCRYPT
//#define AES_CTR_DECRYPT
#define AES_ECB_ENCRYPT
//#define AES_ECB_DECRYPT

/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void TimerInit(void);
void DMAInit(void);
/* Private functions ----------------------------------------------------------*/

bool verification(volatile const unsigned int a[],
                  const unsigned int d3,
                  const unsigned int d2,
                  const unsigned int d1,
                  const unsigned int d0)
{
    if (a[0] != d0)
    {
        printf("[%08x] != [%08x]\r\n", a[0], d0);
        return false;
    }

    if (a[1] != d1)
    {
        printf("[%08x] != [%08x]\r\n", a[1], d1);
        return false;
    }

    if (a[2] != d2)
    {
        printf("[%08x] != [%08x]\r\n", a[2], d2);
        return false;
    }

    if (a[3] != d3)
    {
        printf("[%08x] != [%08x]\r\n", a[3], d3);
        return false;
    }

    return true;
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{

    bool pass = false;

    __disable_irq();

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    bsp_key_init();
    md_uart_init(UART1, &uart_initStruct);
    DMAInit();
    __enable_irq();

    printf("=============================================\r\n");
    printf("AES SDK verification On %s, %s\r\n", __DATE__, __TIME__);
    printf("=============================================\r\n");

    printf("md_rcu_enable_ahben_aesen\r\n");
    md_rcu_enable_aes(RCU);
    md_aes_set_dma_en(AES, MD_AES_CON_DMA_EN_OUTPUT_ENABLE | MD_AES_CON_DMA_EN_INPUT_ENABLE);

#ifdef AES_ECB_ENCRYPT

    memset(aes_data, 0, sizeof(md_aes_init_typedef) * 2);

    aes_data[0].aes_key[3] = 0x2b7e1516;
    aes_data[0].aes_key[2] = 0x28aed2a6;
    aes_data[0].aes_key[1] = 0xabf71588;
    aes_data[0].aes_key[0] = 0x09cf4f3c;

    aes_data[0].aes_iv[3] = 0x0;
    aes_data[0].aes_iv[2] = 0x0;
    aes_data[0].aes_iv[1] = 0x0;
    aes_data[0].aes_iv[0] = 0x0;

    aes_data[0].aes_din[3] = 0x6bc1bee2;
    aes_data[0].aes_din[2] = 0x2e409f96;
    aes_data[0].aes_din[1] = 0xe93d7e11;
    aes_data[0].aes_din[0] = 0x7393172a;

    printf("=============================================\r\n");
    printf("ECB mode encryption verification\r\n");
    printf("=============================================\r\n");

    printf("md_aes_trigger_initial(%d)\r\n", ENABLE);
    md_aes_trigger_initial(AES);

    printf("md_aes_set_block_mode(%d)\r\n", MD_AES_CON_BL_ECB);
    md_aes_set_block_mode(AES, MD_AES_CON_BL_ECB);
    printf("md_aes_get_block_mode(%d)\r\n\r\n", md_aes_get_block_mode(AES));

    printf("md_aes_set_direction(%d)\r\n", MD_AES_CON_MODE_ENCRYPT);
    md_aes_set_direction(AES, MD_AES_CON_MODE_ENCRYPT);
    printf("md_aes_get_direction(%d)\r\n\r\n", md_aes_get_direction(AES));

    printf("md_aes_set_key3(%08x)\r\n", aes_data[0].aes_key[3]);
    printf("md_aes_set_key2(%08x)\r\n", aes_data[0].aes_key[2]);
    printf("md_aes_set_key1(%08x)\r\n", aes_data[0].aes_key[1]);
    printf("md_aes_set_key0(%08x)\r\n\r\n", aes_data[0].aes_key[0]);

    md_aes_set_key2(AES, aes_data[0].aes_key[2]);
    md_aes_set_key3(AES, aes_data[0].aes_key[3]);
    md_aes_set_key1(AES, aes_data[0].aes_key[1]);
    md_aes_set_key0(AES, aes_data[0].aes_key[0]);

    printf("md_aes_get_key3(%08x)\r\n", md_aes_get_key3(AES));
    printf("md_aes_get_key2(%08x)\r\n", md_aes_get_key2(AES));
    printf("md_aes_get_key1(%08x)\r\n", md_aes_get_key1(AES));
    printf("md_aes_get_key0(%08x)\r\n\r\n", md_aes_get_key0(AES));






    printf("md_aes_set_data(%08x)\r\n",      aes_data[0].aes_din[0]);
    printf("md_aes_set_data(%08x)\r\n",      aes_data[0].aes_din[1]);
    printf("md_aes_set_data(%08x)\r\n",      aes_data[0].aes_din[2]);
    printf("md_aes_set_data(%08x)\r\n\r\n",  aes_data[0].aes_din[3]);






    printf("md_aes_get_output_data_depth(%d)\r\n\r\n", md_aes_get_output_data_depth(AES));

    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[0]);
    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[1]);
    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[2]);
    printf("md_aes_get_data(%08x)\r\n\r\n",  aes_data[0].aes_dout[3]);


    printf("md_aes_trigger_start(%d)\r\n", ENABLE);
    md_aes_trigger_start(AES);

    printf("md_aes_is_active_it_enc(%d)\r\n", md_aes_is_active_it_enc(AES));

    while (!md_aes_is_active_it_enc(AES));

    printf("md_aes_is_active_it_enc(%d)\r\n", md_aes_is_active_it_enc(AES));







    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[0]);
    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[1]);
    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[2]);
    printf("md_aes_get_data(%08x)\r\n\r\n",  aes_data[0].aes_dout[3]);

    pass = verification(aes_data[0].aes_dout,
                        0x3ad77bb4,
                        0x0d7a3660,
                        0xa89ecaf3,
                        0x2466ef97);
    printf("verification[%s]\r\n", pass ? "PASS" : "FAIL");
#endif

#ifdef AES_ECB_DECRYPT

    md_aes_set_control(AES, 0x0);

    aes_data[0].aes_key[3] = 0x2b7e1516;
    aes_data[0].aes_key[2] = 0x28aed2a6;
    aes_data[0].aes_key[1] = 0xabf71588;
    aes_data[0].aes_key[0] = 0x09cf4f3c;

    aes_data[0].aes_iv[3] = 0x0;
    aes_data[0].aes_iv[2] = 0x0;
    aes_data[0].aes_iv[1] = 0x0;
    aes_data[0].aes_iv[0] = 0x0;

    aes_data[0].aes_din[3] = 0x3ad77bb4;
    aes_data[0].aes_din[2] = 0x0d7a3660;
    aes_data[0].aes_din[1] = 0xa89ecaf3;
    aes_data[0].aes_din[0] = 0x2466ef97;

    printf("=============================================\r\n");
    printf("ECB mode decryption verification\r\n");
    printf("=============================================\r\n");

    printf("md_aes_trigger_initial(%d)\r\n", ENABLE);
    md_aes_trigger_initial(AES);

    printf("md_aes_set_block_mode(%d)\r\n", MD_AES_CON_BL_ECB);
    md_aes_set_block_mode(AES, MD_AES_CON_BL_ECB);
    printf("md_aes_get_block_mode(%d)\r\n\r\n", md_aes_get_block_mode(AES));

    printf("md_aes_set_direction(%d)\r\n", MD_AES_CON_MODE_DECRYPT);
    md_aes_set_direction(AES, MD_AES_CON_MODE_DECRYPT);
    printf("md_aes_get_direction(%d)\r\n\r\n", md_aes_get_direction(AES));

    printf("md_aes_set_key3(%08x)\r\n", aes_data[0].aes_key[3]);
    printf("md_aes_set_key2(%08x)\r\n", aes_data[0].aes_key[2]);
    printf("md_aes_set_key1(%08x)\r\n", aes_data[0].aes_key[1]);
    printf("md_aes_set_key0(%08x)\r\n\r\n", aes_data[0].aes_key[0]);

    md_aes_set_key2(AES, aes_data[0].aes_key[2]);
    md_aes_set_key3(AES, aes_data[0].aes_key[3]);
    md_aes_set_key1(AES, aes_data[0].aes_key[1]);
    md_aes_set_key0(AES, aes_data[0].aes_key[0]);

    printf("md_aes_get_key3(%08x)\r\n", md_aes_get_key3(AES));
    printf("md_aes_get_key2(%08x)\r\n", md_aes_get_key2(AES));
    printf("md_aes_get_key1(%08x)\r\n", md_aes_get_key1(AES));
    printf("md_aes_get_key0(%08x)\r\n\r\n", md_aes_get_key0(AES));

    md_aes_set_data(AES, aes_data[0].aes_din[0]);
    md_aes_set_data(AES, aes_data[0].aes_din[1]);
    md_aes_set_data(AES, aes_data[0].aes_din[2]);
    md_aes_set_data(AES, aes_data[0].aes_din[3]);
    printf("md_aes_set_data(%08x)\r\n",      aes_data[0].aes_din[0]);
    printf("md_aes_set_data(%08x)\r\n",      aes_data[0].aes_din[1]);
    printf("md_aes_set_data(%08x)\r\n",      aes_data[0].aes_din[2]);
    printf("md_aes_set_data(%08x)\r\n\r\n",  aes_data[0].aes_din[3]);

    aes_data[0].aes_dout[0] = md_aes_get_data(AES);
    aes_data[0].aes_dout[1] = md_aes_get_data(AES);
    aes_data[0].aes_dout[2] = md_aes_get_data(AES);
    aes_data[0].aes_dout[3] = md_aes_get_data(AES);
    printf("md_aes_get_output_data_depth(%d)\r\n\r\n", md_aes_get_output_data_depth(AES));

    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[0]);
    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[1]);
    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[2]);
    printf("md_aes_get_data(%08x)\r\n\r\n",  aes_data[0].aes_dout[3]);


    printf("md_aes_trigger_start(%d)\r\n", ENABLE);
    md_aes_trigger_start(AES);

    printf("md_aes_is_active_it_dec(%d)\r\n", md_aes_is_active_it_dec(AES));

    while (!md_aes_is_active_it_dec(AES));

    printf("md_aes_is_active_it_dec(%d)\r\n", md_aes_is_active_it_dec(AES));


    aes_data[0].aes_dout[0] = md_aes_get_data(AES);
    aes_data[0].aes_dout[1] = md_aes_get_data(AES);
    aes_data[0].aes_dout[2] = md_aes_get_data(AES);
    aes_data[0].aes_dout[3] = md_aes_get_data(AES);

    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[0]);
    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[1]);
    printf("md_aes_get_data(%08x)\r\n",      aes_data[0].aes_dout[2]);
    printf("md_aes_get_data(%08x)\r\n\r\n",  aes_data[0].aes_dout[3]);

    pass = verification(aes_data[0].aes_dout,
                        0x6bc1bee2,
                        0x2e409f96,
                        0xe93d7e11,
                        0x7393172a);
    printf("verification[%s]\r\n", pass ? "PASS" : "FAIL");
#endif


    printf("=============================================\r\n");
    printf("AES SDK verification finish\r\n");
    printf("=============================================\r\n");

    while (1);
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_dma1(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @brief  DMA init
  * @param  None
  * @retval None
  */

void DMAInit(void)
{
    md_dma_set_configuration(DMA1, ENABLE);

//  md_dma_set_primary_channel_source_data_end_address(DMA1, MD_DMA_CHANNEL0, ((uint32_t)(aes_data[0].aes_din+3)));
    dma_initStruct[0].source_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(aes_data[0].aes_din), 4, MD_DMA_CHANNEL_SIZE_WORD);
    dma_initStruct[0].destination_data_end_address = (uint32_t)&AES->DIO;
    md_dma_init(DMA1, dma_initStruct[0]);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL0, MD_DMA_PRS_AES_IN);


    dma_initStruct[1].source_data_end_address = (uint32_t)&AES->DIO;
    dma_initStruct[1].destination_data_end_address = md_dma_set_channel_data_start_address_and_length((uint32_t)(aes_data[0].aes_dout), 4, MD_DMA_CHANNEL_SIZE_WORD);
    md_dma_init(DMA1, dma_initStruct[1]);
    md_dma_set_request_peripherals(DMA1, MD_DMA_CHANNEL1, MD_DMA_PRS_AES_OUT);


}


/**
  * @} Public_Function
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


