/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    7 Jun 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          7 Jun 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <stdio.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */

uint8_t g_adc_temperature_hrc4m_handle_flag = 0U;

const md_rcu_init_typedef g_rcu_initstruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV128,
    MD_RCU_MSW_MCO_HRC,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

const md_gpio_inittypedef g_pin_init_uart_tx =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

const md_gpio_inittypedef g_pin_init_uart_rx =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

const md_gpio_inittypedef g_pin_init_clk_out =    /**< GPIOA Pin8(CLK_OUT) init structure */
{
    MD_GPIO_PIN_8,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF7
};

const md_uart_init_typedef g_uart_initstruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  main.
  * @param  None
  * @retval None
  */

int main(void)
{
    __disable_irq();

    md_rcu_pll0_init(RCU, (md_rcu_init_typedef *)(&g_rcu_initstruct));
    md_rcu_sys_init(RCU, (md_rcu_init_typedef *)(&g_rcu_initstruct));

    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_uart1(RCU);

    md_gpio_init(GPIOB, (md_gpio_inittypedef *)(&g_pin_init_uart_tx)); /*PB6: UART1_TX*/
    md_gpio_init(GPIOB, (md_gpio_inittypedef *)(&g_pin_init_uart_rx)); /*PB7: UART1_RX(Internal weak pull-up)*/
    md_gpio_init(GPIOA, (md_gpio_inittypedef *)(&g_pin_init_clk_out)); /*PA8: CLK OUT*/

    md_uart_init(UART1, (md_uart_init_typedef *)(&g_uart_initstruct));
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);

    __enable_irq();

    adc_temperature_hrc4m_handle_init();
               
    /*定时器初始化*/
    md_rcu_enable_bs16t1(RCU);
    md_timer_set_prescaler_value_pscv(BS16T1, 720);
    md_timer_set_auto_reload_value_arrv(BS16T1, 10000);
    md_timer_enable_it_upd(BS16T1);
    md_timer_enable_counter_cnten(BS16T1);
    NVIC_SetPriority(BS16T1_IRQn, 3);
    NVIC_EnableIRQ(BS16T1_IRQn);
           
    while (1)
    {
        if(g_adc_temperature_hrc4m_handle_flag)
        {
            g_adc_temperature_hrc4m_handle_flag = 0;
            adc_temperature_hrc4m_handle();
        }
    }
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  /*Tx FIFO empty*/

    UART1->TXDATA = ch;            /*Sent byte*/
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


