/**********************************************************************************
 *
 * @file    .c
 * @brief   main C file
 *
 * @date    7 Jun 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          7 Jun 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <stdio.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
#define HRC4M_DATA                     *(volatile uint32_t *)0x1FFFEE00  /*校准值*/
#define ADC_TSDATA                     *(volatile uint32_t *)0x1FFFEE04  /*温感ADC通道ADC值。条件：30C、5V*/

#define TREM_PARAM1                    (3.75)                      /*mV/C。条件：5V*/ 
#define TREM_PARAM2                    (((TREM_PARAM1)*4.096)/5)   /*ADC值/C。条件：5V*/

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */

const md_adc_initial g_adc_initstruct =    /**< ADC init structure */
{
    MD_ADC_CFG_ALIGN_RIGHT,    /*Data alignment*/
    MD_ADC_CFG_RSEL_12BIT,     /*Data resolution*/
    MD_ADC_MODE_NCHS,          /*Regular or Injected*/
    MD_ADC_CFG_CM_SINGLE,      /*Single mode*/
    MD_ADC_NCHS1_NSL_2CON,     /*sample count*/
    MD_ADC_SMPT1_CKDIV_DIV6,   /*ADC prescale*/
};

const uint32_t g_sample_seq_channel1[] =
{
    MD_ADC_NCHS1_NS1_CH17, MD_ADC_NCHS1_NS2_CH16
};

uint16_t g_chip_info_adcvrefint_cal, g_chip_info_adc30, g_chip_info_adc110, g_chip_info_hrc4m_default_trim_val, g_adc_data_buf[2];
float g_adc_voltage_avgslope, g_temperature_buf[16], g_temperature, g_last_temperature = 9999; /*保证第一次必定触发*/
uint8_t g_temperature_buf_index = 0U;

/*                   
+85C时，HRC4M温漂 = +2.0%
+30C时，无温漂  
-30C时，HRC4M温漂 = -2.5% 

校准值对HRC4M的影响：0.45%/bit

温度      校准值     与校准值的偏移
86~         -4       -1.80%
76~85       -3       -1.35%
61~75       -2       -0.90%
46~60       -1       -0.45%
15~45        0        0.00%
0~14        +1       +0.45%
-10~-1      +2       +0.90%
-20~-11     +3       +1.35%
-30~-21     +4       +1.80%
   ~-31     +5       +2.25%
*/
const float g_hrc4m_trim_temperature_table[10] = {-30, -20, -10, 0, 15, 45, 60, 75, 85, 95};  /*温度
                                 校准值        +5    +4   +3  +2  +1  +0  -1  -2  -3  -4*/
/* Private Function ---------------------------------------------------------- */

void adc_temperature_hrc4m_handle_init(void)
{
    md_rcu_enable_adc(RCU);

    g_chip_info_hrc4m_default_trim_val = md_syscfg_get_option_hrctrim(SYSCFG);
    md_rcu_hrc_software_trim(RCU, g_chip_info_hrc4m_default_trim_val);

    g_chip_info_adcvrefint_cal = md_syscfg_get_adc_reference(SYSCFG);/*5V时,内部电源通道的ADC值*/
    g_chip_info_adc30 = md_syscfg_get_adc_low_temperature_reference(SYSCFG);/*5V+30C时,温度通道的ADC值 */
    g_chip_info_adc110 = md_syscfg_get_adc_high_temperature_reference(SYSCFG);/*5V+110C时,温度通道的ADC值 */

    g_adc_voltage_avgslope = (g_chip_info_adc110 - g_chip_info_adc30) / 80.0;

    md_adc_calibration(ADC, (md_adc_initial *)(&g_adc_initstruct));
    md_adc_set_sampletime_channel_16(ADC, 0x33);
    md_adc_set_sampletime_channel_17(ADC, 0x33);
    md_adc_init(ADC, (md_adc_initial *)(&g_adc_initstruct));

    md_adc_sequence_conversion(ADC, (md_adc_initial *)(&g_adc_initstruct), (uint32_t *)g_sample_seq_channel1);
}

void adc_temperature_hrc4m_handle(void)
{
    uint32_t i;
    float temperature_diff;

    md_adc_set_start_normal(ADC, MD_ADC_CON_NSTART_START_REGULAR);

    /*g_adc_data_buf[0] = ADC_CH17:内部电压通道
      g_adc_data_buf[1] = ADC_CH16:温度通道*/
    for (i = 0; i < 2; i++)
    {
        while (md_adc_is_active_flag_normal_status(ADC));

        g_adc_data_buf[i] = md_adc_get_data(ADC);
    }

    if ((HRC4M_DATA) == 0xFFFFFFFF)
        g_temperature_buf[g_temperature_buf_index++] = ((((float)(g_chip_info_adcvrefint_cal * g_adc_data_buf[1]) / g_adc_data_buf[0]) - g_chip_info_adc30) / g_adc_voltage_avgslope) + 30;
    else
        g_temperature_buf[g_temperature_buf_index++] = ((((float)(g_chip_info_adcvrefint_cal * g_adc_data_buf[1]) / g_adc_data_buf[0]) - ((ADC_TSDATA) & 0xFFFFul)) / TREM_PARAM2) + 30; 


    if (g_temperature_buf_index == 16)
    {
        g_temperature_buf_index = 0;
        g_temperature = 0;

        for (i = 0; i < 16; i++)
            g_temperature += g_temperature_buf[i];

        g_temperature = (g_temperature / 16);

        temperature_diff = (g_last_temperature > g_temperature) ? (g_last_temperature - g_temperature) : (g_temperature - g_last_temperature);

        if (temperature_diff > 5)
        {
            for (i = 0; i < 9; i++)
            {
                if (g_temperature < g_hrc4m_trim_temperature_table[i])
                    break;
            }

            if ((HRC4M_DATA) == 0xFFFFFFFF)
                md_rcu_set_hrctrim(RCU, g_chip_info_hrc4m_default_trim_val + (5 - i));
            else
                md_rcu_set_hrctrim(RCU, g_chip_info_hrc4m_default_trim_val + (5 - i) - ((((HRC4M_DATA) & 0xFFFF) - 4000) / 20));

            g_last_temperature = g_temperature;
        }
    }
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


