/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    7 Jun 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          7 Jun 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

#define vrefint     0

#define temperature 1

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOD_PIN3_Init =    /**< GPIOD Pin3(LED Red) init structure */
{
    MD_GPIO_PIN_3,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOB_PIN8_Init =    /**< GPIOB Pin8(LED Green) init structure */
{
    MD_GPIO_PIN_8,
    MD_GPIO_MODE_OUTPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOB_PIN5_Init =    /**< GPIOB Pin5(Button Mid) init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN10_Init =    /**< GPIOC Pin10(Button Left) init structure */
{
    MD_GPIO_PIN_10,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOC_PIN13_Init =    /**< GPIOC Pin13(Button Up) init structure */
{
    MD_GPIO_PIN_13,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOD_PIN0_Init =    /**< GPIOD Pin0(Button Right) init structure */
{
    MD_GPIO_PIN_0,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_gpio_inittypedef GPIOD_PIN1_Init =    /**< GPIOD Pin1(Button Down) init structure */
{
    MD_GPIO_PIN_1,
    MD_GPIO_MODE_INPUT,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_adc_initial adc_initStruct =    /**< ADC init structure */
{
    MD_ADC_CFG_ALIGN_RIGHT,    //Data alignment
    MD_ADC_CFG_RSEL_12BIT,     //Data resolution
    MD_ADC_MODE_NCHS,          //Regular or Injected
    MD_ADC_CFG_CM_SINGLE,      //Single mode
    MD_ADC_NCHS1_NSL_2CON,     //sample count
    MD_ADC_SMPT1_CKDIV_DIV6,   //ADC prescale
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
#define SLOPEAVG    0.34
#define AVGSLOPE    3.75
#define VDDA        5.0
#define FULLSCALE   4095.0

/* Private variable -----------------------------------------------------------*/
volatile uint8_t ADC_DONE = 0;
volatile uint32_t NSE, NCHE = 0;
/* Private function prototypes ------------------------------------------------*/
void SysPeriInit(void);
void Iomux(void);
void Led_Red_On(void);
void Led_Red_Off(void);
void Led_Green_On(void);
void Led_Green_Off(void);

/* Private functions -----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint16_t adc_data[16], i = 0;
    uint16_t ADCts;
    uint16_t ADC30;
    float Vts;
    float V30;
    float Temperature;
    uint16_t ADCvrefint_cal, ADCvrefint;

    float Vrefp;
    uint32_t sample_seq_channel1[] =
    {
        MD_ADC_NCHS1_NS1_CH17, MD_ADC_NCHS1_NS2_CH16
    };

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);

    printf("SystemFrequency_SysClk = %d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk = %d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk = %d\r\n", SystemFrequency_APBClk);
    printf("\r\n");

    ADCvrefint_cal = md_syscfg_get_adc_reference(SYSCFG);
    ADC30 = md_syscfg_get_adc_low_temperature_reference(SYSCFG);
    V30 = ADC30 * (VDDA / FULLSCALE);
    md_adc_calibration(ADC, &adc_initStruct);
    md_adc_set_sampletime_channel_16(ADC, 0x33); //Sample Time Cycle = 51T
    md_adc_set_sampletime_channel_17(ADC,0x5);  //Sample Time Cycle = 5T
    md_adc_init(ADC, &adc_initStruct);

    while ((ADC->RIF & 0x1) == 0);

    md_adc_sequence_conversion(ADC, &adc_initStruct, sample_seq_channel1);
    md_adc_set_start_normal(ADC, MD_ADC_CON_NSTART_START_REGULAR);

    for (i = 0; i <= adc_initStruct.Cnt; i++)
    {
        while (md_adc_is_active_flag_normal_status(ADC));
        adc_data[i] = md_adc_get_data(ADC);
    }

    ADCvrefint = adc_data[0];

    Vrefp = VDDA * ADCvrefint_cal / ADCvrefint ;


    ADCts = adc_data[1];

    Temperature = ((Vts - V30) / AVGSLOPE * 1000) + 30;

    md_tick_waitms(100, 10);

    while (1)
    {
        i = 0;

        while ((ADC->RIF & 0x1) == 0);

        md_adc_sequence_conversion(ADC, &adc_initStruct, sample_seq_channel1);
        md_adc_set_start_normal(ADC, MD_ADC_CON_NSTART_START_REGULAR);

        for (i = 0; i <= adc_initStruct.Cnt; i++)
        {
            while (md_adc_is_active_flag_normal_status(ADC));
            adc_data[i] = md_adc_get_data(ADC);
        }

        ADCvrefint = adc_data[0];

        Vrefp = VDDA * ADCvrefint_cal / ADCvrefint ;

        ADCts = adc_data[1];
        Vts = Vrefp / FULLSCALE * ADCts;

        Temperature = ((Vts - V30) / AVGSLOPE * 1000) + 30;

        printf("current TS temp:%lf\r\n",Temperature);
        md_tick_waitms(100, 10);
    }

}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_adc(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PD3: GPIO LED RED.
  *         PB8: GPIO LED GREEN.
  *         PB5: GPIO Button Mid.
  *         PC10: GPIO Button Left.
  *         PC13: GPIO Button Up.
  *         PD0: GPIO Button Right.
  *         PD1: GPIO Button Down.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOD, &GPIOD_PIN3_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN8_Init);

    md_gpio_init(GPIOB, &GPIOB_PIN5_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN10_Init);
    md_gpio_init(GPIOC, &GPIOC_PIN13_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN0_Init);
    md_gpio_init(GPIOD, &GPIOD_PIN1_Init);
}

/**
  * @brief  Turn-on LED Red.
  * @param  None
  * @retval None
  */
void Led_Red_On()
{
    md_gpio_set_pin_low(GPIOD, MD_GPIO_PIN_3);
}

/**
  * @brief  Turn-off LED Red.
  * @param  None
  * @retval None
  */
void Led_Red_Off()
{
    md_gpio_set_pin_high(GPIOD, MD_GPIO_PIN_3);
}

/**
  * @brief  Turn-on LED Green.
  * @param  None
  * @retval None
  */
void Led_Green_On()
{
    md_gpio_set_pin_low(GPIOB, MD_GPIO_PIN_8);
}

/**
  * @brief  Turn-off LED Green.
  * @param  None
  * @retval None
  */
void Led_Green_Off()
{
    md_gpio_set_pin_high(GPIOB, MD_GPIO_PIN_8);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


