/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    23 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Nov  2021    Ginger          the first version
 *          28 Dec  2021    Ginger          Modify the example for PDS board
 *          28 Mar  2022    AE Team         Modify MD Driver
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC,
    MD_RCU_PLLCLK_72M,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_PLL0,
    (RCU_CON_HRCON | RCU_CON_PLL0ON),
};

md_gpio_inittypedef GPIOB_PIN6_Init =    /**< Uart Tx init structure */
{
    MD_GPIO_PIN_6,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_FLOATING,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOB_PIN7_Init =    /**< Uart Rx init structure */
{
    MD_GPIO_PIN_7,
    MD_GPIO_MODE_FUNCTION,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF2
};

md_gpio_inittypedef GPIOC_PIN5_Init =    /**< ADC ANIN15 init structure */
{
    MD_GPIO_PIN_5,
    MD_GPIO_MODE_ANALOG,
    MD_GPIO_OUTPUT_PUSHPULL,
    MD_GPIO_PULL_UP,
    MD_GPIO_DRIVING_8MA,
    MD_GPIO_AF0
};


md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_EVEN,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

md_adc_initial adc_initStruct =    /**< ADC init structure */
{
    MD_ADC_CFG_ALIGN_RIGHT,    //Data alignment
    MD_ADC_CFG_RSEL_12BIT,     //Data resolution
    MD_ADC_MODE_NCHS,          //Regular or Injected
    MD_ADC_CFG_CM_SINGLE,      //Single mode
    MD_ADC_NCHS1_NSL_16CON,    //sample count
    MD_ADC_SMPT1_CKDIV_DIV6,   //ADC prescale
};

/* Private define -------------------------------------------------------------*/
/* Private macro --------------------------------------------------------------*/
/* Private variable -----------------------------------------------------------*/
volatile uint8_t ADC_DONE = 0;
volatile uint32_t NSE, NCHE = 0;

/* Private function prototypes ------------------------------------------------*/
void SysPeriInit(void);
void Iomux(void);
void delay(int i);

/* Private functions -----------------------------------------------------------*/
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t i = 0;
    uint16_t adc_data[16];
    uint32_t sample_seq_channel[] =
    {
        MD_ADC_NCHS1_NS1_CH15, MD_ADC_NCHS1_NS2_CH15, MD_ADC_NCHS1_NS3_CH15, MD_ADC_NCHS1_NS4_CH15,
        MD_ADC_NCHS2_NS5_CH15, MD_ADC_NCHS2_NS6_CH15, MD_ADC_NCHS2_NS7_CH15, MD_ADC_NCHS2_NS8_CH15,
        MD_ADC_NCHS2_NS9_CH15, MD_ADC_NCHS3_NS10_CH15, MD_ADC_NCHS3_NS11_CH15, MD_ADC_NCHS3_NS12_CH15,
        MD_ADC_NCHS3_NS13_CH15, MD_ADC_NCHS3_NS14_CH15, MD_ADC_NCHS4_NS15_CH15, MD_ADC_NCHS4_NS16_CH15
    };

    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);
    SysPeriInit();
    Iomux();
    bsp_led_init();
    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);

    printf("SystemFrequency = %d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk = %d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk = %d\r\n", SystemFrequency_APBClk);

    md_adc_calibration(ADC, &adc_initStruct);
    md_adc_set_sampletime_channel_15(ADC, 0x40);

    md_adc_init(ADC, &adc_initStruct);

    while ((ADC->RIF & 0x1) == 0);

    //setting ADC Watch Dog
    md_adc_enable_normal_awd(ADC);
    md_adc_set_awdg_high_threshold(ADC, 2000);
    md_adc_set_awdg_low_threshold(ADC, 500);

    md_adc_sequence_conversion(ADC, &adc_initStruct, sample_seq_channel);
    md_adc_set_start_normal(ADC, MD_ADC_CON_NSTART_START_REGULAR);

    for (i = 0; i <= adc_initStruct.Cnt; i++)
    {
        while (md_adc_is_active_flag_normal_status(ADC));

        adc_data[i] = md_adc_get_data(ADC);
    }

    for (i = 0; i <= adc_initStruct.Cnt; i++)
    {
        printf("%dth conversion data is %d\r\n", i, adc_data[i]);
    }

    printf("\r\n");

    md_tick_waitms(100, 10);

    while (1)
    {

        md_adc_set_start_normal(ADC, MD_ADC_CON_NSTART_START_REGULAR);

        for (i = 0; i <= adc_initStruct.Cnt; i++)
        {
            while (md_adc_is_active_flag_normal_status(ADC));

            adc_data[i] = md_adc_get_data(ADC);
        }

        for (i = 0; i <= adc_initStruct.Cnt; i++)
        {
            printf("%dth conversion data is %d\r\n", i, adc_data[i]);
        }

        if (md_adc_is_active_flag_awd(ADC))
        {
            printf("\r\n\r\n--------------------------------------------\r\nTo trigger the watchdog\r\nplease adjust the ADC sampling value within 1000~2000 to stop triggering the watchdog\r\n--------------------------------------------\r\n\r\n");
            md_adc_clear_it_awd(ADC);
        }

        printf("Please rotate the knob to change the resistance value\r\n\r\n");
        i = 0;
        md_tick_waitms(100, 10);
    }
}

void delay(int i)
{
    while (i > 0)
    {
        i--;
    }
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);

    md_rcu_enable_uart1(RCU);
    md_rcu_enable_adc(RCU);
}

/**
  * @brief  Configure I/O Multiplexer
  * @note   PB6: UART1_TX.
  *         PB7: UART1_RX (Internal weak pull-up).
  *         PC5: ANIN15.
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    md_gpio_init(GPIOB, &GPIOB_PIN6_Init);
    md_gpio_init(GPIOB, &GPIOB_PIN7_Init);

    md_gpio_init(GPIOC, &GPIOC_PIN5_Init);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!(UART1->STAT & (UART_STAT_TFEMPTY)));  // Tx FIFO empty

    UART1->TXDATA = ch;            // Sent byte
    return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


