/**********************************************************************************
 *
 * @file    mscfs_host.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

//-------------------------------------------------------//
//----------------------   DRIVER    --------------------//
//-------------------------------------------------------//
#include "es32f0283.h"
#include "md_rcu.h"
#include "mscfat.h"
#include "usb_core.h"
#include "usb_host_enum.h"
#include "usb_host_msc.h"
#include "mscfs_host.h"
#include <stdio.h>

#define FAT16_CLUSTER       4085
#define FAT32_CLUSTER       65525

#define LFN_SEGMENT         13
#define LFN_CHAR_MAX        255

#define Cluster2LBA(x)  (((x - 2) * pLUNEntry->SectorsPerCluster) + pLUNEntry->FirstDataLBA)

static  uint8_t         DIRData[FAT_SECTOR_SZ], FATData[FAT_SECTOR_SZ];
static  uint32_t        LBAData[FAT_SECTOR_SZ / 4];
static  _pLUN_ENTRY     pLUNEntry;
static  _pSHORT_FILENAME    pSFN;
static  _pLONG_FILENAME     pLFN;
static  _LUN_ENTRY  LUNEntry;

static  uint32_t    FATCurrent = 0;
static  uint16_t    LFNData[LFN_CHAR_MAX];

/****
    * @brief    Dump LBA Data
    * @param    pHost       : Host Instance
    * @param    pBuf        : Pointer to LBA Data Buffer
    * @retval   None
****/
void    fshost_dump_lba(_pUSB_FS_HOST pHost, uint8_t *pBuf)
{
    uint16_t    ii;

    for (ii = 0; ii < FAT_SECTOR_SZ; ii++)
    {
        if ((ii & 0xf) == 0)
            printf("0x%04x : ", ii);

        printf("%02x ", *pBuf++);

        if ((ii & 0xf) == 0xf)
            printf("\r\n");
    }
}
/****
    * @brief    Read MSC LBA Data
    * @param    pHost       : Host Instance
    * @param    LBAIndex    : LBA Number
    * @param    pLBABuf     : Pointer to LBA Data Buffer
    * @retval   0x0:No Error, 0x1:Error
****/
uint8_t fshost_read_lba(_pUSB_FS_HOST pHost, uint32_t LBAIndex, uint8_t *pLBABuf)
{
    if (msc_read_lba(pHost, LBAIndex, 1, pLBABuf))
        return (1);

//  fshost_dump_lba(pHost, pLBABuf);
    return (0);
}
/****
    * @brief    Read MSC Cluster Data
    * @param    pHost       : Host Instance
    * @param    Cluster     : Cluster Number
    * @param    pLBABuf     : Pointer to LBA Data Buffer
    * @retval   0x0:No Error, 0x1:Error
****/
uint8_t fshost_read_cluster(_pUSB_FS_HOST pHost, uint32_t Cluster, uint8_t *pLBABuf)
{
    uint32_t    LBA;

    LBA = Cluster2LBA(Cluster);

    if (msc_read_lba(pHost, LBA, pLUNEntry->SectorsPerCluster, pLBABuf))
        return (1);

    return (0);
}
/****
    * @brief    Get Next Cluster from FAT Table
    * @param    pHost       : Host Instance
    * @param    Cluster     : Current Cluster Number
    * @retval   Next Cluster Number
****/
uint32_t    fshost_next_cluster(_pUSB_FS_HOST pHost, uint32_t Cluster)
{
    uint32_t    NextCluster;
    uint32_t    FATEntry, FATIndex;

    FATEntry = pLUNEntry->FAT0Entry;

    if (pLUNEntry->FATMedia == MEDIA_FAT16)
    {
        Cluster <<= 1;
        FATEntry = (FATEntry + (Cluster / FAT_SECTOR_SZ));

        if (FATEntry != FATCurrent)
        {
            FATCurrent = FATEntry;

            if (fshost_read_lba(pHost, FATEntry, FATData))
                return (ENDOFCHAIN16);      //ENDOFCHAIN
        }

        FATIndex = (Cluster & (FAT_SECTOR_SZ - 1));
        NextCluster = DATA16(FATData[FATIndex + 1], FATData[FATIndex]);
    }

    if (pLUNEntry->FATMedia == MEDIA_FAT32)
    {
        Cluster <<= 2;
        FATEntry = (FATEntry + (Cluster / FAT_SECTOR_SZ));

        if (FATEntry != FATCurrent)
        {
            FATCurrent = FATEntry;

            if (fshost_read_lba(pHost, FATEntry, FATData))
                return (ENDOFCHAIN32);      //ENDOFCHAIN
        }

        FATIndex = (Cluster & (FAT_SECTOR_SZ - 1));
        NextCluster = DATA32(FATData[FATIndex + 3], FATData[FATIndex + 2], FATData[FATIndex + 1], FATData[FATIndex]);
    }

    return (NextCluster);
}
/****
    * @brief    Dump File Cluster
    * @param    pHost       : Host Instance
    * @param    Cluster     : First Cluster Number
    * @retval   None
****/
void    fshost_dump_file(_pUSB_FS_HOST pHost, uint32_t Cluster)
{
    printf("Cluster:\r\n");

    if (pLUNEntry->FATMedia == MEDIA_FAT16)
    {
        while (Cluster < ENDOFCHAIN16)
        {
            printf("0x%04x-", Cluster);
            Cluster = fshost_next_cluster(pHost, Cluster);
        }

        printf("0x%04x\r\n", Cluster);
    }

    if (pLUNEntry->FATMedia == MEDIA_FAT32)
    {
        while (Cluster < ENDOFCHAIN32)
        {
            printf("0x%08x-", Cluster);
            Cluster = fshost_next_cluster(pHost, Cluster);
        }

        printf("0x%08x\r\n", Cluster);
    }
}
/****
    * @brief    Enumerate FAT Type
    * @param    pHost       : Host Instance
    * @retval   0x0:No Error, 0x1:Error
****/
uint8_t fshost_enumeration(_pUSB_FS_HOST pHost)
{
    _pMBR_ENTRY     pMBREntry;
    _pBPB_ENTRY     pBPBEntry;
    uint32_t        TotalCluster, TotalSectors;
    uint32_t        Root16Sectors, SectorsPerFAT;

    pLUNEntry = (_pLUN_ENTRY)&LUNEntry;
    pLUNEntry->BPBEntry = 0;    //BPB=LBA0
//Load MBR Entry
    printf("Read LBA : 0x%x\r\n", pLUNEntry->BPBEntry);

    if (fshost_read_lba(pHost, pLUNEntry->BPBEntry, (uint8_t *)LBAData))
        return (1);

    pBPBEntry = (_pBPB_ENTRY)LBAData;

    if (pBPBEntry->Signature != SIGNATUREID)
    {
        printf("Invalid Signature\r\n");
        return (1);
    }

    if (pBPBEntry->BytesPerSector != FAT_SECTOR_SZ)
    {
        printf("is MBR\r\n");
        pLUNEntry->MBREntry = 0;    //MBR=LBA0
        pMBREntry = (_pMBR_ENTRY)LBAData;
//Load BPB Entry
        pLUNEntry->BPBEntry = pMBREntry->Partition0.FirstLBA;
        printf("Read BPB : 0x%x\r\n", pLUNEntry->BPBEntry);

        if (fshost_read_lba(pHost, pLUNEntry->BPBEntry, (uint8_t *)LBAData))
            return (1);

        if (pBPBEntry->Signature != SIGNATUREID)
        {
            printf("Invalid Signature\r\n");
            return (1);
        }

        pBPBEntry = (_pBPB_ENTRY)LBAData;
    }
    else
        printf("is BPB\r\n");

//FAT Determination
//Load FAT Entry
    printf("ReservedSectors=0x%x\r\n", pBPBEntry->ReservedSectors);
    pLUNEntry->FAT0Entry = (pLUNEntry->BPBEntry + (pBPBEntry->ReservedSectors));
    printf("FAT0Entry=0x%x\r\n", pLUNEntry->FAT0Entry);
    pLUNEntry->SectorsPerCluster = pBPBEntry->SectorsPerCluster;
    printf("SectorsPerCluster=0x%x\r\n", pLUNEntry->SectorsPerCluster);
//Total Sectors
    Root16Sectors = (pBPBEntry->RootEntry16 * sizeof(_SHORT_FILENAME) / pBPBEntry->BytesPerSector); //Bytes of Root Entries
    printf("Root16Sectors=0x%x\r\n", Root16Sectors);
    SectorsPerFAT = pBPBEntry->SectorsPreFAT16;
    printf("SectorsPerFAT=0x%x\r\n", SectorsPerFAT);

    if (SectorsPerFAT == 0)
        SectorsPerFAT = pBPBEntry->Ext.Ext32.SectorsPreFAT32;

    printf("SectorsPerFAT=0x%x\r\n", SectorsPerFAT);
    pLUNEntry->FAT1Entry = (pLUNEntry->FAT0Entry + SectorsPerFAT);
    printf("FAT1Entry=0x%x\r\n", pLUNEntry->FAT1Entry);
    pLUNEntry->FirstDataLBA = (pLUNEntry->FAT1Entry + SectorsPerFAT);
    printf("FirstDataLBA=0x%x\r\n", pLUNEntry->FirstDataLBA);
    TotalSectors = pBPBEntry->TotalSectors16;

    if (TotalSectors == 0)
        TotalSectors = pBPBEntry->TotalSectors32;

    printf("TotalSectors=0x%x\r\n", TotalSectors);
//Total Clusters
    TotalCluster = ((TotalSectors - Root16Sectors) / pBPBEntry->SectorsPerCluster);     //TotalCluster Clusters

    if (TotalCluster < FAT16_CLUSTER)
    {
        pLUNEntry->FATMedia = MEDIA_FAT12;
        printf("FAT12\r\n");
        return (1);
    }

    if (TotalCluster < FAT32_CLUSTER)
    {
        printf("FAT16\r\n");
        pLUNEntry->FATMedia = MEDIA_FAT16;
        pLUNEntry->FirstDataLBA += Root16Sectors;
        pLUNEntry->RootCluster = 0;
        pLUNEntry->RootLBA = (pLUNEntry->FAT1Entry + SectorsPerFAT);
    }
    else
    {
        printf("FAT32\r\n");
        pLUNEntry->FATMedia = MEDIA_FAT32;
        pLUNEntry->RootCluster = pBPBEntry->Ext.Ext32.RootCluster;
        pLUNEntry->RootLBA = Cluster2LBA(pLUNEntry->RootCluster);
    }

    printf("FAT0Entry : 0x%x\r\n", pLUNEntry->FAT0Entry);
    printf("FAT1Entry : 0x%x\r\n", pLUNEntry->FAT1Entry);
    printf("Cluster2 : 0x%x\r\n", pLUNEntry->FirstDataLBA);
    printf("SectorsPerCluster : 0x%x\r\n", pLUNEntry->SectorsPerCluster);
    printf("RootCluster : 0x%x\r\n", pLUNEntry->RootCluster);

    return (0);
}
/****
    * @brief    Dump LFN Name
    * @param    pHost       : Host Instance
    * @param    pLFNData    : Pointer to LFN Data
    * @retval   None
****/
void    fshost_dump_lfn(_pUSB_FS_HOST pHost, uint16_t *pLFNData)
{
    while (*pLFNData)
    {
        if (*pLFNData >> 8)
            printf("%c%c", (*pLFNData >> 8), (*pLFNData & 0xff));
        else
            printf("%c", (*pLFNData & 0xff));

        pLFNData++;
    }

    printf("\t<LFN>\r\n");
}
/****
    * @brief    Dump Root File Name
    * @param    pHost       : Host Instance
    * @retval   0x0:No Error, 0x1:Error
****/
uint8_t fshost_root_dir(_pUSB_FS_HOST pHost)
{
    uint32_t    RootLBA, RootCluster;
    uint8_t     ii, DIRIndex, SectorOrder;

    RootLBA = pLUNEntry->RootLBA;
    RootCluster = pLUNEntry->RootCluster;
    printf("RootLBA=0x%x\r\n", RootLBA);
    fshost_read_lba(pHost, RootLBA, DIRData);
    pSFN = (_pSHORT_FILENAME)DIRData;
    DIRIndex = 0;
    SectorOrder = 0;

    while (pSFN->Name[0] != SFN_EMPTY)
    {
        if (pSFN->Attribute == SFN_ATTR_LFN)
        {
            pLFN = (_pLONG_FILENAME)pSFN;

            if (pLFN->Ordinal & 0x40)       //The Last Field
            {
                pLFN->Ordinal &= 0x3f;
                LFNData[pLFN->Ordinal * LFN_SEGMENT] = 0;
            }

            pLFN->Ordinal--;        //Order-1

            for (ii = 0; ii < 5; ii++)          //UniCode 0~4
                LFNData[(pLFN->Ordinal * LFN_SEGMENT) + ii] = pLFN->UniCode0[ii];

            for (; ii < 11; ii++)                   //UniCode 5~10
                LFNData[(pLFN->Ordinal * LFN_SEGMENT) + ii] = pLFN->UniCode1[ii - 5];

            for (; ii < 13; ii++)                   //UniCode 11~12
                LFNData[(pLFN->Ordinal * LFN_SEGMENT) + ii] = pLFN->UniCode2[ii - 11];

            if (pLFN->Ordinal == 0)
                fshost_dump_lfn(pHost, LFNData);
        }
        else
        {
            if (pSFN->Name[0] == SFN_DELETE)        //Deleted
                printf("?");
            else if (pSFN->Name[0] == SFN_E5)       //0xE5
                printf("%c", 0xe5);
            else
                printf("%c", pSFN->Name[0]);

            for (ii = 1; ii < 8; ii++)
                printf("%c", pSFN->Name[ii]);

            if (!(pSFN->Attribute & SFN_ATTR_VOLUME))
                printf(".");

            for (ii = 0; ii < 3; ii++)
                printf("%c", pSFN->Extension[ii]);

            if (pSFN->Attribute & SFN_ATTR_VOLUME)
            {
                printf("\t<Volnme>\r\n");
            }

            if (pSFN->Attribute & SFN_ATTR_SUBDIR)
            {
                printf("\t<DIR>\r\n");
            }

            if (pSFN->Attribute & SFN_ATTR_ARCHIVE)
            {
                printf("\t<Archive>\r\n");
            }
        }

        pSFN++;
        DIRIndex++;

        if (DIRIndex == (FAT_SECTOR_SZ / sizeof(_SHORT_FILENAME)))
        {
            if (pLUNEntry->FATMedia == MEDIA_FAT16)
                RootLBA++;

            if (pLUNEntry->FATMedia == MEDIA_FAT32)
            {
                RootLBA++;
                SectorOrder++;

                if (SectorOrder == pLUNEntry->SectorsPerCluster)
                {
                    RootCluster = fshost_next_cluster(pHost, RootCluster);
                    RootLBA = Cluster2LBA(RootCluster);
                    SectorOrder = 0;
                }
            }

            printf("RootLBA=0x%x\r\n", RootLBA);
            fshost_read_lba(pHost, RootLBA, DIRData);
            pSFN = (_pSHORT_FILENAME)DIRData;
            DIRIndex = 0;
        }
    }

    return (0);
}
/****
    * @brief    Check if MSC filename match Target filename
    * @param    pHost       : Host Instance
    * @param    Size        : Character Size
    * @param    pDirName    : MSC Filename
    * @param    pUserName   : Target Filename
    * @retval   0x0:No Error, 0x1:Error
****/
uint8_t fshost_identify_name(_pUSB_FS_HOST pHost, uint8_t Size, uint8_t *pDirName, uint8_t *pUserName)
{
    uint8_t ii;

    while ((*pDirName) && (*pUserName))
    {
        for (ii = 0; ii < Size; ii++)
        {
            if ((*pDirName++) != (*pUserName++))
                return (1);
        }
    }

    if (*pUserName)
        return (1);

    return (0);
}
/****
    * @brief    Identify if there is matched file on Root
    * @param    pSFName : Target Filename
    * @param    pExt    : Target Extension Name
    * @retval   0x0:No Error, 0x1:Error
****/
uint8_t fshost_app_identify(_pUSB_FS_HOST pHost, uint8_t *pSFName, uint8_t *pExt)
{
    uint32_t    RootLBA, RootCluster;
    uint8_t     ii, DIRIndex, SectorOrder;

    RootLBA = pLUNEntry->RootLBA;
    RootCluster = pLUNEntry->RootCluster;
    printf("RootLBA=0x%x\r\n", RootLBA);
    fshost_read_lba(pHost, RootLBA, DIRData);
    pSFN = (_pSHORT_FILENAME)DIRData;
    DIRIndex = 0;
    SectorOrder = 0;

    while (pSFN->Name[0] != SFN_EMPTY)
    {
        if (pSFN->Attribute == SFN_ATTR_LFN)
        {
            pLFN = (_pLONG_FILENAME)pSFN;

            if (pLFN->Ordinal & 0x40)       //The Last Field
            {
                pLFN->Ordinal &= 0x3f;
                LFNData[pLFN->Ordinal * LFN_SEGMENT] = 0;
            }

            pLFN->Ordinal--;        //Order-1

            for (ii = 0; ii < 5; ii++)          //UniCode 0~4
                LFNData[(pLFN->Ordinal * LFN_SEGMENT) + ii] = pLFN->UniCode0[ii];

            for (; ii < 11; ii++)                   //UniCode 5~10
                LFNData[(pLFN->Ordinal * LFN_SEGMENT) + ii] = pLFN->UniCode1[ii - 5];

            for (; ii < 13; ii++)                   //UniCode 11~12
                LFNData[(pLFN->Ordinal * LFN_SEGMENT) + ii] = pLFN->UniCode2[ii - 11];

//          if (pLFN->Ordinal == 0)
//              fshost_dump_lfn(LFNData);
        }
        else
        {
            if (pSFN->Name[0] == SFN_DELETE)        //Deleted
                printf("?");
            else if (pSFN->Name[0] == SFN_E5)       //0xE5
                printf("%c", 0xe5);
            else
                printf("%c", pSFN->Name[0]);

            for (ii = 1; ii < 8; ii++)
                printf("%c", pSFN->Name[ii]);

            if (!(pSFN->Attribute & SFN_ATTR_VOLUME))
                printf(".");

            for (ii = 0; ii < 3; ii++)
                printf("%c", pSFN->Extension[ii]);

            if (pSFN->Attribute & SFN_ATTR_VOLUME)
            {
                printf("\t<Volnme>\r\n");
            }

            if (pSFN->Attribute & SFN_ATTR_SUBDIR)
            {
                printf("\t<DIR>\r\n");
            }

            if (pSFN->Attribute & SFN_ATTR_ARCHIVE)
            {
                printf("\t<Archive>\r\n");

                if (!fshost_identify_name(pHost, sizeof(uint8_t), pSFN->Extension, pExt))
                {
                    if (!fshost_identify_name(pHost, sizeof(uint8_t), pSFN->Name, (uint8_t *)pSFName))
                    {
//                      fshost_dump_lfn(pHost, LFNData);
                        fshost_dump_file(pHost, (pSFN->FirstCluster32 << 16) | pSFN->FirstCluster16);
                        return (0);
                    }
                }
            }
        }

        pSFN++;
        DIRIndex++;

        if (DIRIndex == (FAT_SECTOR_SZ / sizeof(_SHORT_FILENAME)))
        {
            if (pLUNEntry->FATMedia == MEDIA_FAT16)
                RootLBA++;

            if (pLUNEntry->FATMedia == MEDIA_FAT32)
            {
                RootLBA++;
                SectorOrder++;

                if (SectorOrder == pLUNEntry->SectorsPerCluster)
                {
                    RootCluster = fshost_next_cluster(pHost, RootCluster);
                    RootLBA = Cluster2LBA(RootCluster);
                    SectorOrder = 0;
                }
            }

            printf("RootLBA=0x%x\r\n", RootLBA);
            fshost_read_lba(pHost, RootLBA, DIRData);
            pSFN = (_pSHORT_FILENAME)DIRData;
            DIRIndex = 0;
        }
    }

    return (1);
}
/****
    * @brief    Get MSC File Information
    * @param    pStartClus  : Pointer the First Cluster to Get
    * @param    pFileSize   : Pointer the File Size to Get
    * @retval   0x0:No Error, 0x1:Error
****/
uint8_t     fshost_app_info(_pUSB_FS_HOST pHost, uint32_t *pStartClus, uint32_t *pFileSize)
{
    *pStartClus = ((pSFN->FirstCluster32 << 16) | pSFN->FirstCluster16);
    *pFileSize = pSFN->Size;
    return (0);
}
/****
    * @brief    Get Physical Sector from Cluster
    * @param    pCluster    : Pointer the Current Cluster
    * @param    pIndex      : Pointer the Sector Index on Cluster
    * @retval   Sector Number
****/
uint32_t    fshost_app_sector(_pUSB_FS_HOST pHost, uint32_t *pCluster, uint8_t *pIndex)
{
    uint32_t    Sector;

    Sector = Cluster2LBA(*pCluster) + *pIndex;
    (*pIndex)++;

    if (*pIndex == pLUNEntry->SectorsPerCluster)
    {
        *pIndex = 0;
        *pCluster = fshost_next_cluster(pHost, *pCluster);

        if (pLUNEntry->FATMedia == MEDIA_FAT16)
        {
            if (*pCluster < ENDOFCHAIN16)
                return (Sector);
            else
                return (0);
        }

        if (pLUNEntry->FATMedia == MEDIA_FAT32)
        {
            if (*pCluster < ENDOFCHAIN32)
                return (Sector);
            else
                return (0);
        }
    }

    return (Sector);
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

