/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    18 May 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          18 May 2021     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include "main.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_host_enum.h"
#include "app_host_mscbldr.h"
#include <stdio.h>

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/
/** @addtogroup Public_Function
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
/** @defgroup RCU_Init_Struct RCU Init Structure
  * @{
  */
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV1,
    MD_RCU_MSW_MCO_DISABLE,
    MD_RCU_PLLSRC_HRC48,
    MD_RCU_PLLCLK_PASS,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SW_SYSCLK_HRC48,
    (RCU_CON_HRCON_MSK | RCU_CON_PLL0ON_MSK | RCU_CON_HRC48ON_MSK),
};

/** @defgroup UART_Init_Struct UART Init Structure
  * @{
  */
md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
    MD_UART_BAUDRATE_115200,
    MD_UART_LCON_LSB_FIRST,
    MD_UART_LCON_PS_NONE,
    MD_UART_LCON_STOP_1,
    MD_UART_LCON_DLS_8,
};

extern  uint8_t     DUTEnumData[];
_USB_FS_HOST    MSCBLDR_FS_Host;

/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Private function prototypes ------------------------------------------------*/

/* Public functions -----------------------------------------------------------*/
/** @addtogroup Public_Function
  * @{
  */

/**
  * @brief  IrqInit
  * @param  None
  * @retval None
  */
void IrqInit(void)
{
    NVIC->ICER[0] = 0xFFFFFFFFUL;
    NVIC_EnableIRQ((IRQn_Type)USB_IRQn);        //USB IRQ
}

/**
  * @brief  set irq priority
  * @param  None
  * @retval None
  */
void IrqPriority(void)
{
}

/**
  * @brief  configure i/o multiplexer
  * @param  None
  * @retval None
  */
void Iomux(void)
{
    /*  UART IOMux  */
    md_gpio_set_pin_mode(GPIOB, MD_GPIO_PIN_6,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_mode(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_MODE_FUNCTION);
    md_gpio_set_pin_pull(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_PULL_UP);
    md_gpio_set_pin_function0_7(GPIOB, MD_GPIO_PIN_6,   MD_GPIO_AF2);
    md_gpio_set_pin_function0_7(GPIOB, MD_GPIO_PIN_7,   MD_GPIO_AF2);
    /*  PC08--Host Power Enable Pin  */
    md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_8,   MD_GPIO_MODE_OUTPUT);      //PC08=Output
}

/**
  * @brief  System Peri init
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
    /*  AHB Clock Enable  */
    md_rcu_enable_gpiod(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_usb(RCU);

    /*  APB Clock Enable  */
    md_rcu_enable_uart1(RCU);
}

/**
  * @brief  CoreInit.
  * @param  None
  * @retval None
  */
void  CoreInit()
{
    md_rcu_pll0_init(RCU, &rcu_initStruct);
    md_rcu_sys_init(RCU, &rcu_initStruct);

    SysPeriInit();
    IrqInit();
    Iomux();

    md_uart_init(UART1, &uart_initStruct);
    md_tick_init(SystemFrequency_AHBClk);

}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
    while (!md_uart_is_active_flag_tfempty(UART1));  /* Tx FIFO full */

    md_uart_set_send_data8(UART1, ch);          // Send byte
    return (ch);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    CoreInit();

    printf("USBH MSC BLDR\r\n");
    printf("SysFosc=%dHz\r\n", SystemCoreClock);

//for PDS, PC08 must output high to supply power to USB port
    md_gpio_set_pin_high(GPIOC, MD_GPIO_PIN_8);     //PC08=1

    mscbldr_host_init(&MSCBLDR_FS_Host);
    usb_core_mode_set(eUSBModeForceHost);

    while (1)
    {
//Wait for connection
        printf("Wait Device...");

        while (!MSCBLDR_FS_Host.DevConnected)
        {
            usbh_dev_connect(&MSCBLDR_FS_Host);
        }

        printf("Plugged\r\n");
//Enumeration
        usbh_issue_reset(&MSCBLDR_FS_Host);
        mscbldr_host_ep_config(&MSCBLDR_FS_Host);

        if (usbh_enumeration(&MSCBLDR_FS_Host, DUTEnumData))
        {
            printf("Can't Enumerate Legal Device\r\n");

            while (1);
        }

//Descriptor parser
        if (mscbldr_host_parser(&MSCBLDR_FS_Host, DUTEnumData))
        {
            printf("Not MSC Device\r\n");

            while (1);
        }

//Transaction
        if (mscbldr_host_transaction(&MSCBLDR_FS_Host))
        {
            printf("MSC Update Fail\r\n");

            while (1);
        }

        printf("Update done!!!Remove U-Disk\r\n");

        while (1);
    }

    while (1);
}

/**
  * @} Public_Function
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


