/**********************************************************************************
 *
 * @file    app_host_mscbldr.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usbmsc.h"
#include "usb_host_enum.h"
#include "usb_host_msc.h"
#include "app_host_mscbldr.h"
#include "iap_update.h"
#include "mscfs_host.h"
#include "mscfat.h"
#include <stdio.h>

#pragma pack(1)
#define APP_MSCBLDR_MSG 1

extern  _USB_FS_HOST    MSCBLDR_FS_Host;

uint8_t     CBWData[USBH_BULK_PKTSZ];
uint8_t     CSWData[USBH_BULK_PKTSZ];
uint8_t     MSCData[USBH_BULK_PKTSZ];

static  uint32_t    LBAData[MSC_BLOCK_SZ / 4];

static  uint8_t     MSCMaxLun;

static  uint32_t    IAPAddr, IAPSize;

static  uint8_t UpdateName[] =
{
    'F', 'I', 'R', 'M', 'W', 'A', 'R', 'E', 0,
};
static  uint8_t     UpdateExt[] =
{
    'B', 'I', 'N', 0,
};
uint8_t     DUTEnumData[MSC_DESC_MAX];

/****
    * @brief    USB Application MSC Device Endpoint 0 Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    mscbldr_host_ep0_config(_pUSB_FS_HOST pHost)
{
//Configure EP0
    md_usb_set_epx_index(USB, USBEP0);          //Endpoint 0
    md_usb_set_tx_max_packet_size(USB, (USBH_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64
    md_usb_set_rx_max_packet_size(USB, (USBH_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_rx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64

    md_usb_enable_it_ep0(USB);      //Interrupt Enable for EP0
}
/****
    * @brief    USB Application MSC Device Endpoint Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    mscbldr_host_ep_config(_pUSB_FS_HOST pHost)
{
    uint8_t     EPFIFOIdx;

    EPFIFOIdx = 0x8;
//Configure EP1 In
    md_usb_set_epx_index(USB, MSC_BULK_IN_EP);      //Endpoint 1
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, (USBH_BULK_PKTSZ / 8));  //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0x12
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_enable_tx_double_packet(USB);            //Enable Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((USBH_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((USBH_BULK_PKTSZ + 7) / 8);

//Configure EP1 Out
    md_usb_set_epx_index(USB, MSC_BULK_OUT_EP);     //Endpoint 1
    md_usb_disable_epx_rx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_RX);        //RX
    md_usb_set_rx_max_packet_size(USB, USBH_BULK_PKTSZ / 8);    //64 Bytes
    md_usb_set_rx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0x1a
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_enable_rx_double_packet(USB);            //Enable Double-Packet Buffering
    md_usb_trigger_epx_rx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((USBH_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_rx_double_packet(USB) & USB_RXFIFO2_DPB)  //Double-Packet Buffering
        EPFIFOIdx += ((USBH_BULK_PKTSZ + 7) / 8);

//Initial MUSB
    md_usb_enable_it_ep1_tx(USB);       //Tx Interrupt Enable for EP1
    md_usb_enable_it_ep1_rx(USB);       //Rx Interrupt Enable for EP1

#if APP_MSCBLDR_MSG
    printf("MSC EP Config Done\r\n");
#endif
}
/****
    * @brief    USB Application MSC Device Initialization
    * @param    pDev    : Host Instance
    * @retval   None
****/
void    mscbldr_host_init(_pUSB_FS_HOST pHost)
{
    mscbldr_host_ep0_config(pHost);

    pHost->DevAddr = DEFAULT_ADDRESS;
    pHost->ConfigValue = DEFAULT_CONFIGURATION;
    pHost->DevConnected = 0;
}

/*****
    * @brief    MSCBLDR Descriptor Parser
    * @param    pHost       : Host Instance
    * @param    *pEnumData  : Data Buffer for Descriptor Parser
    * @retval   eUSBHResult
*****/
eUSBHResult mscbldr_host_parser(_pUSB_FS_HOST pHost, uint8_t *pEnumData)
{
    _pUSB_DEVICE_DESC           pDeviceDesc;
    _pUSB_CONFIGURATION_DESC    pConfigurationDesc;
    _pUSB_INTERFACE_DESC        pInterfaceDesc;
    _pUSB_ENDPOINT_DESC         pEndpointDesc;
    uint16_t    wTotalLength;
    uint8_t     bConfig;
    uint8_t     bMSCInEP, bMSCOutEP;
    uint8_t     ii;

#if APP_MSCBLDR_MSG
    printf("MSC Descriptor Parser...\r\n");
#endif
    bMSCInEP = 0;
    bMSCOutEP = 0;
    pDeviceDesc = (_pUSB_DEVICE_DESC)pEnumData;
    pEnumData += sizeof(_USB_DEVICE_DESC);      //Skip Device Descriptor

    for (ii = 0; ii < pDeviceDesc->bNumConfigurations; ii++)
    {
        if (*(pEnumData + 1) != USB_DTYPE_CONFIGURATION)    //Configuration Descriptor
        {
#if APP_MSCBLDR_MSG
            printf("Illegal Descriptor Data\r\n");
#endif
            return (USBH_RESULT_FAIL);
        }

        pConfigurationDesc = (_pUSB_CONFIGURATION_DESC)pEnumData;
        bConfig = pConfigurationDesc->bConfigurationValue;
        wTotalLength = pConfigurationDesc->wTotalLength;

        while (wTotalLength)
        {
            wTotalLength -= *pEnumData;     //Minus wTotalLebgth
            pEnumData += *pEnumData;        //Skip Descriptor

            if (*(pEnumData + 1) == USB_DTYPE_INTERFACE)    //Interface Descriptor
            {
                pInterfaceDesc = (_pUSB_INTERFACE_DESC)pEnumData;

                if (pInterfaceDesc->bInterfaceClass != USB_CLASS_MASS_STORAGE)
                {
                    printf("Not MSD Device\r\n");
                    pEnumData += wTotalLength;      //Skip to Next Configuration
                    wTotalLength = 0;
                }

                continue;
            }

            if (*(pEnumData + 1) != USB_DTYPE_ENDPOINT) //Not Endpoint Descriptor
            {
                continue;
            }

            pEndpointDesc = (_pUSB_ENDPOINT_DESC)pEnumData;

            if (pEndpointDesc->bmAttribute != USB_EP_ATTR_BULK)     //Not Bulk Endpoint
                continue;

            if (pEndpointDesc->bEndpointAddress & 0x80) //In EP
            {
                printf("Bulk In EP=0x%x\r\n", pEndpointDesc->bEndpointAddress);
                bMSCInEP = (pEndpointDesc->bEndpointAddress & 0x7f);    //Endpoint x
            }
            else                                                                            //Out EP
            {
                printf("Bulk Out EP=0x%x\r\n", pEndpointDesc->bEndpointAddress);
                bMSCOutEP = (pEndpointDesc->bEndpointAddress);  //Endpoint x
            }
        }
    }

    if (bMSCInEP == 0)
    {
        printf("Can't Find Bulk In Endpoint\r\n");
        return (USBH_RESULT_FAIL);
    }

    if (bMSCOutEP == 0)
    {
        printf("Can't Find Bulk Out Endpoint\r\n");
        return (USBH_RESULT_FAIL);
    }

    md_usb_set_epx_index(USB, MSC_BULK_IN_EP);      //Bulk In, EP1
    md_usb_set_rx_target_endpoint(USB, bMSCInEP);   //Mapping to BULKInEP
    md_usb_set_epx_index(USB, MSC_BULK_OUT_EP);     //Endpoint x
    md_usb_set_tx_target_endpoint(USB, bMSCOutEP);  //Bulk Out, EP1
    usbh_set_configuration(pHost, bConfig);
    return (USBH_RESULT_PASS);
}
/****
    * @brief    MSCBLDR Enumeration Startup
    * @param    pHost       : Host Instance
    * @retval   eUSBHResult
****/
uint8_t mscbldr_host_startup(_pUSB_FS_HOST pHost)
{
    uint8_t MSCMaxLun;

    if (msc_issue_get_max_lun(pHost, &MSCMaxLun))
        return (1);

    if (msc_issue_request_sense(pHost, CBWData, CSWData, MSCData))
        return (1);

    if (msc_issue_inquiry(pHost, CBWData, CSWData, MSCData))
        return (1);

//  if (msc_issue_read_format_capacities(pHost, CBWData, CSWData, MSCData))
//      return(1);
    if (msc_issue_read_capacity(pHost, CBWData, CSWData, MSCData))
        return (1);

//  if (msc_issue_mode_sense_6(pHost, CBWData, CSWData, CSWData))
//      return(1);
//  if (msc_issue_request_sense(pHost, CBWData, CSWData, CSWData))
//      return(1);
    if (msc_issue_test_unit_ready(pHost, CBWData, CSWData, CSWData))
        return (1);

    return (0);
}

/****
    * @brief    MSCBLDR Progam Application
    * @param    pHost       : Host Instance
    * @retval   eUSBHResult
****/
eUSBHResult mscbldr_host_transaction(_pUSB_FS_HOST pHost)
{
    uint32_t    Cluster, Sector;
    uint32_t    APPStart, FileSize;
    uint8_t     Index;

    mscbldr_host_startup(pHost);

    if (fshost_enumeration(pHost))
    {
#if APP_MSCBLDR_MSG
        printf("File System Not Supported\r\n");
#endif
        return (USBH_RESULT_FAIL);
    }

    if (fshost_app_identify(pHost, UpdateName, UpdateExt))
    {
#if APP_MSCBLDR_MSG
        printf("Update File Not Found\r\n");
#endif
        return (USBH_RESULT_FAIL);
    }

    iap_flash_info(&IAPAddr, &IAPSize);

    if (fshost_app_info(pHost, &Cluster, &FileSize))
    {
#if APP_MSCBLDR_MSG
        printf("Application Info Error\r\n");
#endif
        return (USBH_RESULT_FAIL);
    }

    if (FileSize > IAPSize)
    {
#if APP_MSCBLDR_MSG
        printf("Application Size Error\r\n");
#endif
        return (USBH_RESULT_FAIL);
    }

    APPStart = IAPAddr;
    IAPSize = FileSize;
#if APP_MSCBLDR_MSG
    printf("Start updating!!!Don't remove U-Disk\r\n");
    printf("E-0x%x-0x%x\r\n", APPStart, IAPSize);
#endif
    iap_flash_erase(IAPAddr, IAPSize);
    Index = 0;

    while (FileSize)
    {
        if ((Sector = fshost_app_sector(pHost, &Cluster, &Index)) == 0)
        {
#if APP_MSCBLDR_MSG
            printf("App Get Sector Error\r\n");
#endif
            return (USBH_RESULT_FAIL);
        }

        if (msc_read_lba(pHost, Sector, 1, (uint8_t *)LBAData))
            return (USBH_RESULT_FAIL);

#if APP_MSCBLDR_MSG
        printf("S-0x%x-", Sector);
        printf("P-0x%x-", APPStart);
#endif

        if (FileSize >= MSC_BLOCK_SZ)
        {
            if (iap_flash_program(APPStart, MSC_BLOCK_SZ, LBAData))
                return (USBH_RESULT_FAIL);

#if APP_MSCBLDR_MSG
            printf("0x%x\r\n", FileSize);
#endif
            APPStart += MSC_BLOCK_SZ;
            FileSize -= MSC_BLOCK_SZ;
        }
        else
        {
            if (iap_flash_program(APPStart, FileSize, LBAData))
                return (USBH_RESULT_FAIL);

#if APP_MSCBLDR_MSG
            printf("0x%x\r\n", FileSize);
#endif
            FileSize = 0;
        }
    }

#if APP_MSCBLDR_MSG
    printf("Update Complete\r\n");
#endif
    iap_check_and_run(IAPAddr);
    return (USBH_RESULT_PASS);
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
