/**********************************************************************************
 *
 * @file    mscfs_dev.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


//-------------------------------------------------------//
//----------------------   DRIVER    --------------------//
//-------------------------------------------------------//
#include "es32f0283.h"
#include "mscfat.h"
#include "fat16_bldr.h"
#include "mscfs_dev.h"
#include "iap_update.h"
#include <stdio.h>

#define LFN_SEGMENT     13
#define LFN_LAST_FIRLD  0x40
#define MAX_FILENAME    30

uint32_t    MSCIAPStart;
uint8_t     MSCIAPDone = 0;

static  const   _BPB_ENTRY  FAT16_BPBEntry =
{
    0xeb, 0x3c, 0x90,                   //Jump Code
    {'M', 'S', 'D', 'O', 'S', '5', '.', '0',}, //OEM Name
    BPB_BYTES_PER_SECTOR,               //Bytes Per Sector
    BPB_SECTORS_PER_CLUSTER,            //Sectors Per Cluster
    BPB_RESERVED_SECTORS,               //Reserved Sectors
    BPB_NUMBER_OF_FAT,                  //Number of FAT
    BPB_NUMBER_OF_ROOT_ENTRY,           //Maxinum Number of FAT12/16 Root Entries
    0x00,                               //FAT16 Total Logical Sectors
    0xf8,                               //Media ID
    BPB_SECTORS_PER_FAT,                //Logical Sectors Per FAT16
    0x3f,                               //Sectors Per Track
    0xff,                               //Number of Heads
    0,                                  //Number of Hidden Sectors
    BPB_NUMBER_OF_SECTORS,              //Number of Sectors
//Ext16
    0x80,                               //Logical Drive Number of Partition
    0x00,                               //Reserved
    0x29,                               //Extended Sugnature
    0x1e6fd2bb,                         //Serial Number of Partition
    {'N', 'O', ' ', 'N', 'A', 'M', 'E', ' ', ' ', ' ', ' ',}, //Volume Name of Partition
    {'F', 'A', 'T', '1', '6', ' ', ' ', ' ',}, //FAT32 Name
    {0x00},                             //Executable Code
    SIGNATUREID,                        //Signature
};

static  const   _SHORT_FILENAME SFN_MSDVolume =
{
    'M', 'S', 'C', ' ', 'B', 'L', 'D', 'R', //File Name
    ' ', ' ', ' ',                      //Extension Name
    SFN_ATTR_VOLUME,                    //Attribute
    0x00,                               //Reserved for NT
    0x00,                               //Create Time, mS
    0x0000,                             //Create Time
    0x0000,                             //Create Date
    0x0000,                             //Last Access Date
    0x0000,                             //First Cluster 3~2
    0x8e41,                             //Last Modified Time
    0x32bb,                             //Last Modified Date
    0x0000,                             //First Cluster 1~0
    0x00000000                          //File Size
};

static  const   _SHORT_FILENAME SFN_VirtualFile =
{
    ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', //File Name
    ' ', ' ', ' ',                      //Extension Name
    SFN_ATTR_READONLY,                  //Attribute
    0x00,                               //Reserved for NT
    0x00,                               //Create Time, mS
    ((12 << 11) | (0 << 5) | (0 << 0)), //Create Time
    ((41 << 9) | (4 << 5) | (30 << 0)), //Create Date
    ((41 << 9) | (4 << 5) | (30 << 0)), //Last Access Date
    0x0000,                             //First Cluster 3~2
    ((12 << 11) | (0 << 5) | (0 << 0)), //Last Modified Time
    ((41 << 9) | (4 << 5) | (30 << 0)), //Last Modified Date
    0x0000,                             //First Cluster 1~0
    0x00000000                          //File Size
};

static  const uint8_t FAT16_ReadyFileName[] =
{
    'R', 'E', 'A', 'D', 'Y', ' ', ' ', ' ', /*00-07 - Short File Name */
};

static  const uint8_t FAT16_SuccessFileName[] =
{
    'S', 'U', 'C', 'C', 'E', 'S', 'S', ' ', /*00-07 - Short File Name */
};

const uint8_t FAT16_FailFileName[] =
{
    'F', 'A', 'I', 'L', 'E', 'D', ' ', ' ', /*00-07 - Short File Name */
};

static  const uint8_t FAT16_StartedFileName[] =
{
    'S', 'T', 'A', 'R', 'T', 'E', 'D', ' ', /*00-07 - Short File Name */
};

static  const   uint8_t FAT16_FATHeader[] =
{
    0xf8, 0xff, 0xff, 0xff,
};

static  eMSCBootState eBootState = BOOT_STATE_READY;
static  uint16_t    UpdateName[MAX_FILENAME];
static  uint32_t    UpdateLBA;
static  uint32_t    IAPAddr, IAPSize;
/****
    * @brief  Read MSD LBA
    * @param  LBA: LBA Number
    * @param  pBuf: Data Buffer to Read
    * @retval 0 : No error, 1 : Error
****/
uint8_t mscfs_dev_read_lba(uint32_t LBA, uint32_t *pRBuf)
{
    uint32_t    EFAddr;
    uint16_t    ii;
    uint8_t     *pBuf;

    pBuf = (uint8_t *)pRBuf;

    switch (LBA)
    {
        /* Boot Sector */
        case DBR_SECTOR:
            for (ii = 0; ii < sizeof(_BPB_ENTRY); ii++)
                *pBuf++ = *((uint8_t *)&FAT16_BPBEntry + ii);

            break;

        /* FAT Table Sector */
        case FAT16_FAT0_SECTOR:
        case FAT16_FAT1_SECTOR:
            for (ii = 0; ii < sizeof(FAT16_FATHeader); ii++)
                *pBuf++ = FAT16_FATHeader[ii];

            for (; ii < FAT_SECTOR_SZ; ii++)
                *pBuf++ = 0;

            break;

        /* Root Directory Sector */
        case FAT16_ROOT_SECTOR:
            for (ii = 0; ii < sizeof(_SHORT_FILENAME); ii++)
                *pBuf++ = *((uint8_t *)&SFN_MSDVolume + ii);

            switch (eBootState)
            {
                case BOOT_STATE_READY:
                    for (ii = 0; ii < sizeof(FAT16_ReadyFileName); ii++)
                        *pBuf++ = FAT16_ReadyFileName[ii];

                    break;

                case BOOT_STATE_FLASH_ERROR:
                    for (ii = 0; ii < sizeof(FAT16_FailFileName); ii++)
                        *pBuf++ = FAT16_FailFileName[ii];

                    break;

                case BOOT_STATE_SUCCESS:
                    for (ii = 0; ii < sizeof(FAT16_SuccessFileName); ii++)
                        *pBuf++ = FAT16_SuccessFileName[ii];

                    break;

                case BOOT_STATE_START:
                    for (ii = 0; ii < sizeof(FAT16_StartedFileName); ii++)
                        *pBuf++ = FAT16_StartedFileName[ii];

                    break;
            }

            for (; ii < sizeof(_SHORT_FILENAME); ii++)
                *pBuf++ = *((uint8_t *)&SFN_VirtualFile + ii);

            for (; ii < FAT_SECTOR_SZ; ii++)
                *pBuf++ = 0;

            break;

        /* All other sectors empty */
        default:
#if 0
            if (LBA >= FAT16_DATA_SECTOR)
            {
                iap_flash_info(&MSCIAPStart, &IAPSize);
                EFAddr = (LBA - FAT16_DATA_SECTOR) * FAT_SECTOR_SZ;
                EFAddr += MSCIAPStart;
                iap_flash_read(EFAddr, FAT_SECTOR_SZ, pBuf);
            }
            else
            {
                for (ii = 0; ii < FAT_SECTOR_SZ; ii++)
                    *pBuf++ = 0;
            }

#else

            for (ii = 0; ii < FAT_SECTOR_SZ; ii++)
                *pBuf++ = 0;

#endif
            break;
    }

    return (0);
}
/****
    * @brief  Dump MSD LFN
    * @param  pLFNData: LFN Data
    * @retval 0 : No error, 1 : Error
****/
void    mscfs_dev_dump_lfn(uint16_t *pLFNData)
{
    while (*pLFNData)
    {
        if (*pLFNData >> 8)
            printf("%c%c", (*pLFNData >> 8), (*pLFNData & 0xff));
        else
            printf("%c", (*pLFNData & 0xff));

        pLFNData++;
    }

    printf("\t<LFN>\r\n");
}
/****
    * @brief  Write MSD LBA
    * @param  LBA: LBA Number
    * @param  pBuf: Data Buffer to Write
    * @retval 0 : No error, 1 : Error
****/
uint8_t mscfs_dev_write_lba(uint32_t LBA, uint32_t *pBuf)
{
    _pSHORT_FILENAME    pSFN;
    _pLONG_FILENAME     pLFN;
    uint32_t    EFAddr;
    uint8_t     ii;

    if (LBA == FAT16_ROOT_SECTOR)           //Parser File Name
    {
        pSFN = (_pSHORT_FILENAME)pBuf;

        while (pSFN->Name[0] != SFN_EMPTY)
        {
            if (pSFN->Attribute == SFN_ATTR_LFN)
            {
                pLFN = (_pLONG_FILENAME)pSFN;

                if (pLFN->Ordinal & LFN_LAST_FIRLD)     //The Last Field
                {
                    pLFN->Ordinal &= (LFN_LAST_FIRLD - 1);
                    UpdateName[pLFN->Ordinal * LFN_SEGMENT] = 0;
                }

                pLFN->Ordinal--;                //Order-1

                for (ii = 0; ii < 5; ii++)      //UniCode 0~4
                    UpdateName[(pLFN->Ordinal * LFN_SEGMENT) + ii] = pLFN->UniCode0[ii];

                for (; ii < 11; ii++)           //UniCode 5~10
                    UpdateName[pLFN->Ordinal * LFN_SEGMENT + ii] = pLFN->UniCode1[ii - 5];

                for (; ii < 13; ii++)           //UniCode 11~12
                    UpdateName[(pLFN->Ordinal * LFN_SEGMENT) + ii] = pLFN->UniCode2[ii - 11];

                if (pLFN->Ordinal == 0)
                {
//                  if (!mscfs_dev_bin_check(UpdateName))
                    eBootState = BOOT_STATE_IDENTIFY;
                }
            }

            if (pSFN->Attribute == SFN_ATTR_ARCHIVE)
            {
                if (eBootState != BOOT_STATE_IDENTIFY)
                {
                    printf("Identify Error\r\n");
                    eBootState = BOOT_STATE_READY;
                    pSFN++;
                    continue;
                }

                if ((pSFN->Extension[0] != 'B') || (pSFN->Extension[1] != 'I') || (pSFN->Extension[2] != 'N'))
                {
                    printf("Ext Error\r\n");
                    eBootState = BOOT_STATE_READY;
                    pSFN++;
                    continue;
                }

                iap_flash_info(&MSCIAPStart, &IAPSize);

                if (pSFN->Size > IAPSize)
                {
                    printf("Size Error\r\n");
                    eBootState = BOOT_STATE_READY;
                    pSFN++;
                    continue;
                }

                IAPSize = pSFN->Size;
                eBootState = BOOT_STATE_START;
                UpdateLBA = (((pSFN->FirstCluster16 - 2) * BPB_SECTORS_PER_CLUSTER) + FAT16_DATA_SECTOR);
                mscfs_dev_dump_lfn(UpdateName);
                printf("FileSize=%d\r\n", IAPSize);
                printf("Cluster=0x%x\tLBA=0x%x\r\n", pSFN->FirstCluster16, UpdateLBA);
            }

            pSFN++;
        }
    }

    if (LBA >= UpdateLBA)       //Parser File Name
    {
        if (eBootState != BOOT_STATE_START)
            return (0);

        EFAddr = ((LBA - UpdateLBA) * FAT_SECTOR_SZ);
        EFAddr += IAP_FLASH_START;

        if ((EFAddr & FAT_SECTOR_SZ) == 0)      //Page Boundary
        {
            iap_flash_erase(EFAddr, IAP_FLASH_PAGE_SIZE);
            printf("PErase 0x%x\r\n", EFAddr);
        }

        iap_flash_program(EFAddr, FAT_SECTOR_SZ, pBuf);
        printf("Program 0x%x\r\n", EFAddr);

        if ((LBA - UpdateLBA) == ((IAPSize - 1) / FAT_SECTOR_SZ))
        {
            printf("Success\r\n");
            eBootState = BOOT_STATE_SUCCESS;
            MSCIAPDone = 1;
        }
    }

    return (0);
}
/****
    * @brief  Read MSD Cluster
    * @param  Cluster: Cluster Number
    * @param  pBuf: Data Buffer to Read
    * @retval 0 : No error, 1 : Error
****/
uint8_t mscfs_dev_read_cluster(uint32_t Cluster, uint32_t *pBuf)
{
    return (0);
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

