/**********************************************************************************
 *
 * @file    app_dev_mscbldr.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "usbhid.h"
#include "usbcdc.h"
#include "usbmsc.h"
#include "usb_dev_msc.h"
#include "app_mscbldr_desc.h"
#include "mscfs_dev.h"
#include "cb_dev_mscbldr.h"
#include <stdio.h>

#pragma pack(1)
#define CB_DEV_APP_MSG      1

//_CB_ENUM_REQUEST
static  eUSBDState  cb_dev_request_enum(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST);

//_CB_CLASS_REQUEST
static  eUSBDState  cb_dev_request_cdc(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST);
static  eUSBDState  cb_dev_request_hid(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST);
static  eUSBDState  cb_dev_request_msc(_pUSB_FS_DEVICE, _pUSB_SETUP_REQUEST);

_CB_APP_DEV_REQUEST cb_app_dev_mscbldr =
{
    cb_dev_request_enum,    //cb_dev_request_enum
    NULL,                   //cb_dev_request_msc
};

/****
    * @brief    USB Device Enumeration Callback
    * @param    pDev        : Device Instance
    * @param    pSetupReq   : Pointer to Standard Request Data
    * @retval   Updated eUSBDState
****/
static  eUSBDState  cb_dev_request_enum(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
    uint8_t u8ValueH, u8ValueL;

#if CB_DEV_APP_MSG
    printf("CB Dev Enum - ");
#endif

    if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) != USB_REQ_TYPE_STANDARD)    //bmRequestType != Standard
        return (USBD_STATE_IDLE);

    if (pSetupReq->bRequest == USB_REQ_GET_DESCRIPTOR)
    {
#if CB_DEV_APP_MSG
        printf("Get Descriptor\r\n");
#endif
        u8ValueL = (pSetupReq->wValue & 0xff);
        u8ValueH = (pSetupReq->wValue >> 8);

        switch (u8ValueH)   //wValueH
        {
            case USB_DTYPE_DEVICE:  //Device
                pDev->pCtrlBurstData = (uint8_t *)&MSCDeviceDesc;       //Device Descriptor
                pDev->CtrlBurstRemain = sizeof(_USB_DEVICE_DESC);       //bLength
                break;

            case USB_DTYPE_CONFIGURATION:   //Configuration
                pDev->pCtrlBurstData = (uint8_t *)&MSCConfigurationDesc;        //Configuration Descriptor
                pDev->CtrlBurstRemain = sizeof(_MSC_CONFIGURATION_DESC);        //wTotalLength
                break;

            case USB_DTYPE_STRING:  //String
                switch (u8ValueL)
                {
                    case 0:
                        pDev->pCtrlBurstData = (uint8_t *)&MSCString0Desc;      //String 0 Descriptor
                        pDev->CtrlBurstRemain = sizeof(_USB_STRING_0_DESC);     //bLength
                        break;

                    case 1:
                        pDev->pCtrlBurstData = (uint8_t *)&MSCString1Desc;      //String 1 Descriptor
                        pDev->CtrlBurstRemain = sizeof(_USB_STRING_1_DESC);     //bLength
                        break;

                    case 2:
                        pDev->pCtrlBurstData = (uint8_t *)&MSCString2Desc;      //String 2 Descriptor
                        pDev->CtrlBurstRemain = sizeof(MSCString2Desc);         //bLength
                        break;

                    default:        //Not Supported
                        return (USBD_STATE_STALL);
                }

                break;

            case USB_DTYPE_QUALIFIER:       //Qualifier
                pDev->pCtrlBurstData = NULL;
                pDev->CtrlBurstRemain = 0;
                break;

            default:        //Not Supported
                return (USBD_STATE_STALL);
        }

        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == USB_REQ_SET_ADDRESS)
    {
#if CB_DEV_APP_MSG
        printf("Set Address\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == USB_REQ_SET_CONFIGURATION)
    {
#if CB_DEV_APP_MSG
        printf("Set Configuration\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == USB_REQ_SET_INTERFACE)
    {
#if CB_DEV_APP_MSG
        printf("Set Interface\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Device HID Class Callback
    * @param    pDev        : Device Instance
    * @param    pSetupReq   : Pointer to Standard Request Data
    * @retval   Updated eUSBDState
****/
static  eUSBDState  cb_dev_request_hid(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
    uint16_t    u16BCnt;
    uint8_t     u8ReportID;
    uint32_t    u32FData;

#if CB_DEV_APP_MSG
    printf("CB Dev HID - ");
#endif

    if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) != USB_REQ_TYPE_CLASS)       //bmRequestType != Class
        return (USBD_STATE_IDLE);

    if (pSetupReq->bRequest == HID_REQ_GET_REPORT)
    {
#if CB_DEV_APP_MSG
        printf("Get Report\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == HID_REQ_GET_IDLE)
    {
#if CB_DEV_APP_MSG
        printf("Get Idle\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == HID_REQ_GET_PROTOCOL)
    {
#if CB_DEV_APP_MSG
        printf("Get Protocol\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == HID_REQ_SET_REPORT)
    {
#if CB_DEV_APP_MSG
        printf("Set Report\r\n");
#endif
        return (USBD_STATE_CTRLOUT);
    }

    if (pSetupReq->bRequest == HID_REQ_SET_IDLE)
    {
#if CB_DEV_APP_MSG
        printf("Set Idle\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == HID_REQ_SET_PROTOCOL)
    {
#if CB_DEV_APP_MSG
        printf("Set Protocol\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Device CDC Class Callback
    * @param    pDev        : Device Instance
    * @param    pSetupReq   : Pointer to Standard Request Data
    * @retval   Updated eUSBDState
****/
static  eUSBDState  cb_dev_request_cdc(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
#if CB_DEV_APP_MSG
    printf("CB Dev CDC - ");
#endif

    if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) != USB_REQ_TYPE_CLASS)       //bmRequestType != Class
        return (USBD_STATE_IDLE);

    if (pSetupReq->bRequest == CDC_SET_LINE_CODING)
    {
#if CB_DEV_APP_MSG
        printf("Set Line Coding\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == CDC_GET_LINE_CODING)
    {
#if CB_DEV_APP_MSG
        printf("Get Line Coding\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == CDC_SET_CONTROL_LINE_STATE)
    {
#if CB_DEV_APP_MSG
        printf("Set Control Line State\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Device MSC Class Callback
    * @param    pDev        : Device Instance
    * @param    pSetupReq   : Pointer to Standard Request Data
    * @retval   Updated eUSBDState
****/
static  eUSBDState  cb_dev_request_msc(_pUSB_FS_DEVICE pDev, _pUSB_SETUP_REQUEST pSetupReq)
{
#if CB_DEV_APP_MSG
    printf("CB Dev MSC - ");
#endif

    if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) != USB_REQ_TYPE_CLASS)       //bmRequestType != Class
        return (USBD_STATE_IDLE);

    if (pSetupReq->bRequest == MSC_REQ_GET_MAX_LUN)
    {
#if CB_DEV_APP_MSG
        printf("Get Max Lun\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    if (pSetupReq->bRequest == MSC_REQ_STORAGE_RESET)
    {
#if CB_DEV_APP_MSG
        printf("Storage Reset\r\n");
#endif
        return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
