/**********************************************************************************
 *
 * @file    app_dev_mscbldr.c
 * @brief   c File
 *
 * @date    5 May 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 May 2022      AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#include "es32f0283.h"
#include "md_usb.h"
#include "usb_core.h"
#include "usb_enum.h"
#include "usb_dev_enum.h"
#include "usbmsc.h"
#include "usb_dev_msc.h"
#include "app_mscbldr_desc.h"
#include "app_dev_mscbldr.h"
#include "cb_dev_mscbldr.h"
#include "iap_update.h"
#include "mscfs_dev.h"
#include "mscfat.h"
#include "fat16_bldr.h"
#include <stdio.h>

#pragma pack(1)
#define APP_MSCBLDR_MSG 1

static  _pUSB_SETUP_REQUEST pSetupReq = (_pUSB_SETUP_REQUEST) &SetupReq;
static  uint8_t     u8MSCCBWData[sizeof(_MSC_COMMAND_BLOCK)], u8MSCCSWData[sizeof(_MSC_COMMAND_STATUS)];
static  uint32_t    u32LBAData[BPB_BYTES_PER_SECTOR / 4];
/****
    * @brief    SCAI Command Inquiry Data
****/
static  const   _SCSI_INQUIRY_DATA  SCSIInquiryData =
{
    0x00,       //PeripheralDeviceType, 0x00=Direct-Access Device
    0x80,       //RMB, 0x80=Removable Media
//  0x00,       //RMB, 0x00=Non Removable Media
    0x00,       //Version
    0x00,       //ResponseDataFormat
    0x1f,       //AdditionalLength=0x1f
    0x00, 0x00, 0x00,   //Res0[3]
    'E', 'S', 'S', 'E', 'M', 'I', ' ', ' ', //VendorID
    'U', 'S', 'B', ' ', 'M', 'S', 'D', ' ', 'F', 'A', 'T', '1', '6', ' ', ' ', ' ', //ProductID
    '1', '.', '0', '0', //Product Revison
};
/****
    * @brief    SCAI Command Capacity List Data
****/
static  _SCSI_CAPACITY_LIST_DATA    SCSICapacityListData =
{
//Header
    0x00, 0x00, 0x00,   //Res0[3]
    sizeof(_SCSI_CAPACITY_DESCRIPTOR),  //CapacityListLength
//Descriptor
    BYTE3(BPB_NUMBER_OF_SECTORS),   //NumberofBlocks3
    BYTE2(BPB_NUMBER_OF_SECTORS),   //NumberofBlocks2
    BYTE1(BPB_NUMBER_OF_SECTORS),   //NumberofBlocks1
    BYTE0(BPB_NUMBER_OF_SECTORS),   //NumberofBlocks0
    0x02,                           //DescriptorCode, 0x2=Formatted Media
    BYTE2(BPB_BYTES_PER_SECTOR),    //BlockLength2
    BYTE1(BPB_BYTES_PER_SECTOR),    //BlockLength1
    BYTE0(BPB_BYTES_PER_SECTOR),    //BlockLength0
};
/****
    * @brief    SCAI Command Read Capacity Data
****/
static  _SCSI_READ_CAPACITY_DATA    SCSIReadCapacityData =
{
    BYTE3(BPB_NUMBER_OF_SECTORS - 1),   //LastLBA3
    BYTE2(BPB_NUMBER_OF_SECTORS - 1),   //LastLBA2
    BYTE1(BPB_NUMBER_OF_SECTORS - 1),   //LastLBA1
    BYTE0(BPB_NUMBER_OF_SECTORS - 1),   //LastLBA0
    BYTE3(BPB_BYTES_PER_SECTOR),        //BlockLength3
    BYTE2(BPB_BYTES_PER_SECTOR),        //BlockLength2
    BYTE1(BPB_BYTES_PER_SECTOR),        //BlockLength1
    BYTE0(BPB_BYTES_PER_SECTOR),        //BlockLength0
};
/****
    * @brief    SCAI Command Mode Sense Data
****/
static  const   _SCSI_MODE_SENSE_6_DATA SCSIModeSense6Data =
{
    (sizeof(_SCSI_MODE_SENSE_6_DATA) - 1),  //ModeDataLength;
    0x00,       //MediumType;
    0x00,       //DeviceSpecificParameter;
    0x00,       //BlockDescriptorLength;
};
/****
    * @brief    SCAI Command Request Sense Data
****/
static  const   _SCSI_REQUEST_SENSE_DATA    SCSIRequestSenseData =
{
    SCSI_RS_CUR_ERRORS,                 //ResponseCode
    0,                                  //Reserved
    SCSI_RS_KEY_NO_SENSE,               //SenseKey;
    0, 0, 0, 0,                         //Information[4];
    10,                                 //AdditionalSenseLength;
    0, 0, 0, 0,                         //CommandSpecificInfo[4];
    0,                                  //AdditionalSenseCode;
    0,                                  //AdditionalSenseCodeQualifier;
    0,                                  //FieldReplaceableUnitCode;
    0, 0, 0,                            //SenseKeySpecific[3];
};
/****
    * @brief    USB Application MSC Device Endpoint 0 Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    mscbldr_dev_ep0_config(_pUSB_FS_DEVICE pDev)
{
//Configure EP0
    md_usb_set_epx_index(USB, USBEP0);          //Endpoint 0
    md_usb_set_tx_max_packet_size(USB, (USBD_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64
    md_usb_set_rx_max_packet_size(USB, (USBD_CTRL_PKTSZ / 8));  //64 bytes
    md_usb_set_rx_fifo_start_addr(USB, 0x0);        //Start Address=0x0
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64

    md_usb_enable_it_ep0(USB);      //Interrupt Enable for EP0
}
/****
    * @brief    USB Application MSC Device Endpoint Configuration
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    mscbldr_dev_ep_config(_pUSB_FS_DEVICE pDev)
{
    uint8_t     EPFIFOIdx;

    EPFIFOIdx = 0x8;
//Configure EP1 In
    md_usb_set_epx_index(USB, MSC_BULK_IN_EP);  //Endpoint 1
    md_usb_disable_epx_tx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_TX);        //TX
    md_usb_set_tx_max_packet_size(USB, (BLDR_BULK_PKTSZ / 8));  //64 bytes
    md_usb_set_tx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0x12
    md_usb_set_tx_max_fifo_packet_size(USB, MD_USB_TXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_enable_tx_double_packet(USB);            //Enable Double-Packet Buffering
    md_usb_trigger_epx_tx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((BLDR_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_tx_double_packet(USB))    //Double-Packet Buffering
        EPFIFOIdx += ((BLDR_BULK_PKTSZ + 7) / 8);

//Configure EP1 Out
    md_usb_set_epx_index(USB, MSC_BULK_OUT_EP); //Endpoint 1
    md_usb_disable_epx_rx_isochronous(USB);         //Non ISO
    md_usb_set_epx_mode(USB, MD_USB_TXCSRH_MODE_RX);        //RX
    md_usb_set_rx_max_packet_size(USB, BLDR_BULK_PKTSZ / 8);    //64 Bytes
    md_usb_set_rx_fifo_start_addr(USB, EPFIFOIdx);  //Start Address=0x1a
    md_usb_set_rx_max_fifo_packet_size(USB, MD_USB_RXFIFO2_MAXPKTSIZE_64);  //Size=64, No Double-Packet Buffering
    md_usb_enable_rx_double_packet(USB);            //Enable Double-Packet Buffering
    md_usb_trigger_epx_rx_flush_fifo(USB);          //Flush FIFO

    EPFIFOIdx += ((BLDR_BULK_PKTSZ + 7) / 8);

    if (md_usb_is_enabled_rx_double_packet(USB) & USB_RXFIFO2_DPB)  //Double-Packet Buffering
        EPFIFOIdx += ((BLDR_BULK_PKTSZ + 7) / 8);

//Initial MUSB
    md_usb_enable_it_ep1_tx(USB);       //Tx Interrupt Enable for EP1
    md_usb_enable_it_ep1_rx(USB);       //Rx Interrupt Enable for EP1

    md_usb_set_device_address(USB, 0);          //Default Address

#if APP_MSCBLDR_MSG
    printf("MSC EP Config Done\r\n");
#endif
}
/****
    * @brief    USB Application MSC Device Initialization
    * @param    pDev    : Device Instance
    * @retval   None
****/
void    mscbldr_dev_init(_pUSB_FS_DEVICE pDev)
{
    mscbldr_dev_ep0_config(pDev);
//Device Callback
    pDev->cb_app_dev_request = (void *)&cb_app_dev_mscbldr;
}
/****
    * @brief    USB Application MSC Device Enumeration
    * @param    pDev        : Device Instance
    * @param    u8USBState  : Current Enumeration State
    * @retval   Updated Enumeration State
****/
eUSBDState  mscbldr_dev_enum(_pUSB_FS_DEVICE pDev, eUSBDState u8USBState)
{
    u8USBState = usbd_dev_enum(pDev, pSetupReq, u8USBState);

    switch (u8USBState)
    {
        case USBD_STATE_IDLE:
            return (USBD_STATE_IDLE);

        case USBD_STATE_RESET:          //Reset
#if APP_MSCBLDR_MSG
            printf("Reset\r\n");
#endif
            mscbldr_dev_ep_config(pDev);
            return (USBD_STATE_IDLE);

        case USBD_STATE_RESUME:         //Resume
#if APP_MSCBLDR_MSG
            printf("Resume\r\n");
#endif
            return (USBD_STATE_IDLE);

        case USBD_STATE_SUSPEND:        //Suspend
#if APP_MSCBLDR_MSG
            printf("Suspend\r\n");
#endif
            return (USBD_STATE_IDLE);

        case USBD_STATE_SETUP:          //Setup
            if ((pSetupReq->bmRequestType & USB_REQ_TYPE_MASK) == USB_REQ_TYPE_CLASS)       //bmRequestType=Class
                u8USBState = (*pMSCClassRequest[pSetupReq->bRequest - MSC_REQ_GET_MAX_LUN])(pDev, pSetupReq);

            return (u8USBState);

        case USBD_STATE_CTRLOUT:        //Ctrl Out
#if APP_MSCBLDR_MSG
//          printf("CtrlOut\r\n");
#endif
            return (USBD_STATE_CTRLOUT);

        case USBD_STATE_CTRLIN:         //Ctrl In
#if APP_MSCBLDR_MSG
//          printf("CtrlIn\r\n");
#endif
//          u8USBState=usbd_handle_ctrl_in();
//          return(u8USBState);
            return (USBD_STATE_CTRLIN);

        case USBD_STATE_EP1IN:          //EP1, Bulk In
            return (USBD_STATE_IDLE);

        case USBD_STATE_EP1OUT:         //EP1, Bulk Out
#if APP_MSCBLDR_MSG
//          printf("EP1Out\r\n");
#endif
            return (USBD_STATE_EP1OUT);

        default:
            return (USBD_STATE_IDLE);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Application MSC Device Handle Bulk In
    * @param    pDev    : Device Instance
    * @param    u16BCnt : Byte Count for Bulk Transfer
    * @param    pu8Buf  : Pointer to Data Buffer for Bulk Transfer
    * @retval   Idle State
****/
eUSBDState  mscbldr_handle_bulk_in(_pUSB_FS_DEVICE pDev, uint16_t u16BCnt, uint8_t *pu8Buf)
{
    while (u16BCnt)
    {
        md_usb_set_epx_index(USB, MSC_BULK_IN_EP);  //Endpoint 1

        while (md_usb_is_enabled_epx_tx_ready(USB));    //Wait TxRdy=0

        if (u16BCnt > BLDR_BULK_PKTSZ)
        {
            usbd_handle_epx_in(pDev, MSC_BULK_IN_EP, BLDR_BULK_PKTSZ, pu8Buf);
            pu8Buf += BLDR_BULK_PKTSZ;
            u16BCnt -= BLDR_BULK_PKTSZ;
        }
        else
        {
            usbd_handle_epx_in(pDev, MSC_BULK_IN_EP, u16BCnt, pu8Buf);
            u16BCnt = 0;
        }
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Application MSC Device Handle Bulk Out
    * @param    pDev    : Device Instance
    * @param    u16BCnt : Byte Count for Bulk Transfer
    * @param    pu8Buf  : Pointer to Data Buffer for Bulk Transfer
    * @retval   Idle State
****/
eUSBDState  mscbldr_handle_bulk_out(_pUSB_FS_DEVICE pDev, uint16_t u16BCnt, uint8_t *pu8Buf)
{
    eUSBDState  u8USBState;
    uint16_t    u16DLen;

    u8USBState = USBD_STATE_IDLE;

    while (u16BCnt)
    {
        u8USBState = mscbldr_dev_enum(pDev, u8USBState);

        if (u8USBState == USBD_STATE_EP1OUT)        //MSC Bulk Out
        {
            u16DLen = usbd_epx_out_available(pDev, MSC_BULK_OUT_EP);
            usbd_handle_epx_out(pDev, MSC_BULK_OUT_EP, u16DLen, pu8Buf);
            pu8Buf += u16DLen;
            u16BCnt -= u16DLen;
            u8USBState = USBD_STATE_IDLE;
        }

//      printf("u16BCnt=0x%x\r\n", u16BCnt);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Application MSC Device Handle Error
    * @param    pDev        : Device Instance
    * @param    u8CBWData   : Pointer to CBW Data Buffer
    * @param    u8CSWData   : Pointer to CSW Data Buffer
    * @retval   Idle State
****/
eUSBDState  mscbldr_error_handler(_pUSB_FS_DEVICE pDev, uint8_t *u8CBWData, uint8_t *u8CSWData)
{
    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Application MSC Device to Parse CBW Command from Host
    * @param    pDev        : Device Instance
    * @param    u8CBWData   : Pointer to CBW Data Buffer
    * @param    u8CSWData   : Pointer to CSW Data Buffer
    * @retval   Idle State
****/
eUSBDState  mscbldr_cbw_parser(_pUSB_FS_DEVICE pDev, uint8_t *u8CBWData, uint8_t *u8CSWData)
{
    _pMSC_COMMAND_BLOCK     pCBWData;
    _pMSC_COMMAND_STATUS    pCSWData;
    _pSCSI_READ_10_COMMAND  pSCSIRead10;
    _pSCSI_WRITE_10_COMMAND pSCSIWrite10;
    uint32_t    u32TrandferRemain, u32DataResidue;
    uint32_t    u32LBA;
    uint16_t    u16LBACnt;

    pCBWData = (_pMSC_COMMAND_BLOCK)u8CBWData;

    if (pCBWData->dSignature != CBW_SIGNATURE)
        return (USBD_STATE_IDLE);

    u32TrandferRemain = pCBWData->sDataTransferLength;

    switch (pCBWData->CB[0])
    {
        case SCSI_TEST_UNIT_READY:
#if APP_MSCBLDR_MSG
            printf("Test Unit Ready\r\n");
#endif
            u32DataResidue = 0;
            break;

        case SCSI_INQUIRY:
#if APP_MSCBLDR_MSG
            printf("Inquery\r\n");
#endif
            usbd_handle_epx_in(pDev, MSC_BULK_IN_EP, sizeof(_SCSI_INQUIRY_DATA), (uint8_t *)&SCSIInquiryData);
            u32DataResidue = 0;
            break;

        case SCSI_READ_FORMAT_CAPACITIES:
#if APP_MSCBLDR_MSG
            printf("Read Format Capacities\r\n");
#endif
            usbd_handle_epx_in(pDev, MSC_BULK_IN_EP, sizeof(_SCSI_CAPACITY_LIST_DATA), (uint8_t *)&SCSICapacityListData);
            u32DataResidue = 0;
            break;

        case SCSI_READ_CAPACITY:
#if APP_MSCBLDR_MSG
            printf("Read Capacity\r\n");
#endif
            usbd_handle_epx_in(pDev, MSC_BULK_IN_EP, sizeof(_SCSI_READ_CAPACITY_DATA), (uint8_t *)&SCSIReadCapacityData);
            u32DataResidue = 0;
            break;

        case SCSI_MODE_SENSE_6:
#if APP_MSCBLDR_MSG
            printf("Mode Sense 6\r\n");
#endif
            usbd_handle_epx_in(pDev, MSC_BULK_IN_EP, sizeof(_SCSI_MODE_SENSE_6_DATA), (uint8_t *)&SCSIModeSense6Data);
            u32DataResidue = 0;
            break;

        case SCSI_REQUEST_SENSE:
            usbd_handle_epx_in(pDev, MSC_BULK_IN_EP, sizeof(_SCSI_REQUEST_SENSE_DATA), (uint8_t *)&SCSIRequestSenseData);
#if APP_MSCBLDR_MSG
            printf("Request Sense\r\n");
#endif
            break;

        case SCSI_READ_10:
#if APP_MSCBLDR_MSG
            printf("Read 10-");
#endif
            pSCSIRead10 = (_pSCSI_READ_10_COMMAND)pCBWData->CB;
            u32LBA = DATA32(pSCSIRead10->LBA3, pSCSIRead10->LBA2, pSCSIRead10->LBA1, pSCSIRead10->LBA0);
            u16LBACnt = DATA16(pSCSIRead10->TransferLength1, pSCSIRead10->TransferLength0);
#if APP_MSCBLDR_MSG
            printf("LBA=0x%x-Count=0x%x\r\n", u32LBA, u16LBACnt);
#endif

            while (u16LBACnt--)
            {
                mscfs_dev_read_lba(u32LBA++, u32LBAData);
                mscbldr_handle_bulk_in(pDev, BPB_BYTES_PER_SECTOR, (uint8_t *)u32LBAData);
                u32TrandferRemain -= BPB_BYTES_PER_SECTOR;
            }

            u32DataResidue = u32TrandferRemain;
            break;

        case SCSI_WRITE_10:
#if APP_MSCBLDR_MSG
            printf("Write 10-");
#endif
            pSCSIWrite10 = (_pSCSI_WRITE_10_COMMAND)pCBWData->CB;
            u32LBA = DATA32(pSCSIWrite10->LBA3, pSCSIWrite10->LBA2, pSCSIWrite10->LBA1, pSCSIWrite10->LBA0);
            u16LBACnt = DATA16(pSCSIWrite10->TransferLength1, pSCSIWrite10->TransferLength0);
#if APP_MSCBLDR_MSG
            printf("LBA=0x%x-Count=0x%x\r\n", u32LBA, u16LBACnt);
#endif

            while (u16LBACnt--)
            {
                mscbldr_handle_bulk_out(pDev, BPB_BYTES_PER_SECTOR, (uint8_t *)u32LBAData);
                mscfs_dev_write_lba(u32LBA++, u32LBAData);
                u32TrandferRemain -= BPB_BYTES_PER_SECTOR;
            }

            u32DataResidue = u32TrandferRemain;
            break;

        default:
            break;
    }

    pCSWData = (_pMSC_COMMAND_STATUS)u8CSWData;
    pCSWData->dSignature = CSW_SIGNATURE;
    pCSWData->dTag = pCBWData->dTag;
    pCSWData->sDataResidue = u32DataResidue;
    pCSWData->bStatus = CSW_STATUS_SUCCESS;

//  md_usb_set_epx_index(MSC_BULK_IN_EP);   //Endpoint 1
//  while (md_usb_is_enabled_epx_tx_ready());   //Wait TxRdy=0
    usbd_handle_epx_in(pDev, MSC_BULK_IN_EP, sizeof(_MSC_COMMAND_STATUS), (uint8_t *)pCSWData);
#if APP_MSCBLDR_MSG
//  printf("CSW\r\n");
#endif

    if (MSCIAPDone)
    {
#if APP_MSCBLDR_MSG
        printf("Remap-0x%x\r\n", MSCIAPStart);
#endif
        md_usb_set_epx_index(USB, MSC_BULK_IN_EP);  //Endpoint 1

        while (md_usb_is_enabled_epx_tx_ready(USB));    //Wait TxRdy=0

        usb_core_dev_disconnect();
        iap_check_and_run(MSCIAPStart);
    }

    return (USBD_STATE_IDLE);
}
/****
    * @brief    USB Application MSC Device Transaction
    * @param    pDev        : Device Instance
    * @param    u8USBState  : Current Transaction State
    * @retval   Updated Transaction State
****/
eUSBDState  mscbldr_dev_transaction(_pUSB_FS_DEVICE pDev, eUSBDState u8USBState)
{
    uint16_t    u16DLen;

    if (u8USBState != USBD_STATE_EP1OUT)
        return (USBD_STATE_IDLE);

    u16DLen = usbd_epx_out_available(pDev, MSC_BULK_OUT_EP);
    usbd_handle_epx_out(pDev, MSC_BULK_OUT_EP, u16DLen, u8MSCCBWData);

    if (u16DLen != sizeof(_MSC_COMMAND_BLOCK))      //Not Valid CBW
    {
        printf("CBW Error\r\n");
        mscbldr_error_handler(pDev, u8MSCCBWData, u8MSCCSWData);
        return (USBD_STATE_IDLE);
    }

    mscbldr_cbw_parser(pDev, u8MSCCBWData, u8MSCCSWData);
    return (USBD_STATE_IDLE);
}
/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/
